#pragma once

#include <Inventor/Axis.h>
#include <Inventor/ViewerComponents/nodes/SceneInteractor.h>

class SoMouseWheelEvent;
class SoKeyboardEvent;
class SoLocation2Event;
class SoEvent;
class SoViewingCube;
class InteractionOrbiter;
class SoNodeSensor;
class SoFieldSensor;

/**
* @PREVIEWTAG
* Tool class for building a basic interactive OpenInventor application
* with mode-less scene "orbiter" viewing behavior.
*
* @ingroup ViewerComponentsNodes
*
* The SceneOrbiter is an extension of the SceneInteractor node that provides
* camera and headlight manipulations like panning, zooming and orbiting similar
* to the classic Open Inventor "Examiner" viewer classes (e.g. SoWinExaminerViewer)
* and the SceneExaminer (in Orbit navigation mode).
*
* The SceneOrbiter provides a "mode-less" viewer which is very convenient for users.
* The Examiner viewer and SceneExaminer viewers are always in either viewing mode
* (mouse events control the camera) or in selection mode (mouse events are sent to
* the scene graph and objects can be selected). This often requires the user to
* frequently switch between viewing and selection mode (for example by pressing
* the ESC key) while interacting with the scene. SceneOrbiter does not have modes.
* For example, a mouse click without moving the mouse is interpreted as a selection
* and the event is sent to the scene graph, but a mouse click and "drag" is
* interpreted as controlling the camera.
*
* The SceneOrbiter provides two different methods to control the rotation of the
* camera around the scene : trackball and turntable, see #RotationMethod enum.\n
* The trackball method allows the camera to move along any circular orbit around the scene
* and look in the direction of its center. The orbit orientation can change at any time
* while interpreting a mouse drag. Thus the trackball is known as an unconstrained
* orbiting mode. The trackball is the unique method used by the historical examiner
* viewer and by the SceneExaminer.\n
* The additional turntable method is a more constrained orbiting mode. It lets the
* camera move only along vertical orbits such as the earth meridians, and along
* horizontal circles such as the earth parallels. The vertical orbits and the horizontal
* circles are related to an up axis, which can be changed calling #setUpAxis. The
* horizontal circles are centered around this up axis, and the vertical orbits share
* their south and north pole on this up axis.
* As the camera cannot move along any other circle, this mode is more constrained
* compared to the trackball mode. However, it provides a more natural navigation
* experience when viewing a scene that has strong vertical and horizontal references.
* For instance, when orbiting in turntable mode around a scene representing the Eiffel
* tower, this one always looks vertical on your screen.
* Compared to the trackball mode, the turntable mode does not need any tedious small
* mousemove correction to keep the scene well aligned with the borders of the screen.
*
* The 'headlight', an SoDirectionalLight
* node, is automatically aligned with the camera's view direction.
*
* An SoViewingCube node is automatically added to the scene.
* The viewing cube can be hidden by calling the #enableViewingCube() method.
* Note that the up axis of the viewing cube is synchronized with the up axis of the SceneOrbiter.
*
* See parent class SceneInteractor for more details about the structure
* of the internal scene graph.
*
* The SceneOrbiter uses an instance of SoCameraInteractor
* to manipulate the camera in response to OpenInventor events.
*
* Notes:
* - Window system integration @BR
*   The SceneOrbiter needs a component that integrates the Open Inventor 3D
*   rendering window with the native window system.  System dependent tasks
*   include creating a window, placing the window in the application's user
*   interface, initializing OpenGL and processing the native event/message loop.
*   System independent support for this is provided by the SoRenderAreaCore class.
*   Example components are provided for various window systems, for example,
*   WinRenderArea for native Microsoft Windows.
*
* - Event handling @BR
*   The SceneOrbiter needs a component that builds OpenInventor events (SoEvent)
*   from native system events. System independent support for this is provided by
*   the SoEventBuilder class. Example components are provided for various window
*   systems, for example, WinEventToSoEvent for native Microsoft Windows.
*
* - Library @BR
*   A basic version of SceneOrbiter is a supported part of the Open Inventor API
*   and a prebuilt library is provided.
*
* - Source code @BR
*   The basic version of SceneOrbiter is also provided as source code
*   to allow applications to customize and build their own interactive tool class. @BR
*   See $OIVHOME/source/Inventor/ViewerComponents/nodes.
*
* - Scene graph @BR
*   The application scene graph should be the last child of the SceneOrbiter.
*   The initial application scene graph can be added by simply calling the inherited
*   method addChild(). But note that if you need to replace the application scene
*   graph, for example loading a new data set, do @I not@i call removeAllChildren().
*   That would also remove the SceneOrbiter's camera, headlight and event handler nodes.
*   Add an SoSeparator to the SceneOrbiter to serve as the "scene graph holder", then
*   add and remove the application scene graph from this node.
*
* - Clip planes @BR
*   SceneOrbiter automatically adjusts the 'near' and 'far' clipping planes when
*   events modifying the camera are handled. This adjustment, based on the
*   bounding box of the scene, ensures that shapes will not be clipped as the
*   camera orbits and also that depth buffer precision is maximized.
*   @BR Note: Updating clipping planes after a camera move can be insufficient.
*   If the scene graph is modified or if a dragger or a rendered shape is moved,
*   they can disappear or become partially clipped. A classic implementation
*   of a render area must adjust clipping planes before each rendering by calling the
*   provided method #adjustClippingPlanes(). See render area's implementations available
*   in $OIVHOME/source/Inventor/gui folder for examples of adjustClippingPlanes use.
*
* - Compatibility with classical viewers @BR
*   Please note that some interaction behaviors are different from the classic
*   Open Inventor viewer classes (e.g. SoXtExaminerViewer): @BR @BR
*   - @B Left Mouse + Shift @b does Zoom in/out.
*   - The @B Mouse wheel@b performs a dolly relative to the @I cursor position@i,
*     not the center of the viewport.
*     @BR @BR
*   - The classic @B ESC key@b behavior is not implemented. \n
*     (Not necessary because viewing and selection are supported without separate modes.)
*   - The classic @B Alt key@b behavior is not implemented.
*     This key is reserved for application use.
*   - The @B Right Mouse@b button does not display a popup menu.
*     This button is reserved for application use.
*   - Seek mode is not supported.
*
* - Compatibility with SceneExaminer @BR
*   - Only the left mouse button has defined behaviors. \n
*     Pan and zoom behaviors require pressing a keyboard key while dragging the mouse.
*
* @USAGE
*     @BR @BR
*   - @B Left Mouse: @b Rotate the scene.
*   - @B Left Mouse + Ctrl: @b Pan the scene.
*   - @B Left Mouse + Shift: @b Zoom in/out.
*   - @B Mouse Wheel: @b Zoom in/out (zoom center is the mouse cursor location).
*     @BR @BR
*
* \htmlonly </UL> \endhtmlonly
*
* @SEE_ALSO
*   SceneInteractor, SoCameraInteractor, SiInteractionModeListener
*
* @PREVIEWFEATURES
*/
class VIEWERCOMPONENTS_API SceneOrbiter : public SceneInteractor
{

public:

  /**
   * Defines how the mouse events or touch events control the rotations of the scene.
   */
  enum RotationMethod
  {
    /**
     * Constrained mode which forces the camera to follow meridians and parallel circles
     * related to the up-axis. This mode is very useful to help the user to see the up-axis
     * of the scene vertical, i.e. parallel to the left/right borders of the screen.
     */
    TURNTABLE,
    /**
     * Unconstrained mode which allows the camera to follow any circular orbit.
     * Thus, this mode allows to render the scene with any orientation.
     */
    TRACKBALL
  };

  /** Constructor */
  SceneOrbiter();

  /** Destructor */
  virtual ~SceneOrbiter();

  /**
  * Sets the constraint level to limit the turntable rotation.
  *
  * The level is a value between [0,1] used to adjust the constraint on the
  * turntable rotation according to the speed of movement :
  * - 0 means no constraint on the rotation regardless of the interaction speed
  * - 1 means very strongly constrained even if interaction is slow.
  *
  * This prevents unexpected changes in the axis of rotation during fast
  * movements.
  * Default is 0 which means no constraint on the rotation.
  *
  * @param level
  *          the constraint level.
  */
  void setConstraintLevel( float level );

  /**
  * Returns the current constraint level which limits the
  * turntable rotation.
  *
  * @return the constraint level.
  */
  float getConstraintLevel();

  /**
   * Sets the scene rotation method.
   * Default is TURNTABLE.
   */
  void setRotationMethod( RotationMethod method );

  /**
   * Returns the current scene rotation method.
   */
  RotationMethod getRotationMethod();

  /**
   * Sets the up axis of the scene.
   * Default is Y.
   * Note that the up axis of the viewing cube is synchronized with the up axis of the SceneOrbiter.
   */
  void setUpAxis( openinventor::inventor::Axis::Type axis );

  /**
   * Returns the current up axis of the scene.
   */
  openinventor::inventor::Axis::Type getUpAxis();

  /**
   * Enables or disables viewing cube. Default is true.
   */
  void enableViewingCube( bool enabled );

  /**
   * Returns if viewing cube is enabled.
   */
  bool isViewingCubeEnabled() const;

  /**
   * Returns the viewing cube object.
   */
  SoViewingCube* getViewingCube();

  /**
   * Selects perspective or orthographic camera. Default is perspective.
   */
  virtual void setCameraMode( SceneInteractor::CameraMode mode );

protected:
  virtual void mouseWheelMoved( SoMouseWheelEvent* wheelEvent, SoHandleEventAction* action );
  virtual void mouseMoved( SoLocation2Event* mouseEvent, SoHandleEventAction* action );
  virtual void mousePressed( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action );
  virtual void mouseReleased( SoMouseButtonEvent* mouseEvent, SoHandleEventAction* action );
  virtual void keyPressed( SoKeyboardEvent* keyEvent, SoHandleEventAction* action );
  virtual void keyReleased( SoKeyboardEvent* keyEvent, SoHandleEventAction* action );
  virtual void touch( SoTouchEvent* touchEvent, SoHandleEventAction* action );
  virtual void zoom(SoScaleGestureEvent* scaleEvent, SoHandleEventAction* action);
  virtual void rotate( SoRotateGestureEvent* rotateEvent, SoHandleEventAction* action );

private:
  InteractionOrbiter* m_interaction;
  SoViewingCube* m_viewingCube;
  SoRef<SoSwitch> m_viewingCubeSwitch;
  SoFieldSensor* m_upAxisSensor;

  // Tools to handle the external camera update (like ViewingCube)
  SoNodeSensor* m_cameraSensor;
  static void cameraSensorCallback(void* data, SoSensor* sensor);
  // Keep up to date the viewport to use it in the cameraSensorCallback
  SbViewportRegion m_viewportRegion;
};
