/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/



#ifndef  _SO_WIN_DEVICE_
#define  _SO_WIN_DEVICE_

#include <Inventor/Win/SoWinBeginStrict.h>
#include <windows.h>
#include <Inventor/Win/SoWinDef.h>

#include <Inventor/SbLinear.h>
#include <Inventor/events/SoEvent.h>

/**
 * @VSGEXT Abstract base class for input devices.
 * 
 * @ingroup Win
 * 
 * @DESCRIPTION
 *   This is the abstract base class for devices in the Open Inventor Windows
 *   component and utility library. When a device is registered with an
 *   SoWinRenderArea, the device is able to generate messages in the render area
 *   window.
 *
 *   Used internally by SoWinRenderArea and derived classes.  Can also be used
 *   by the application to translate system events into SoEvent objects.
 * 
 * @SEE_ALSO
 *    SoWinMouse,
 *    SoWinKeyboard,
 *    SoWinRenderArea,
 *    SoWinTouchScreen
 * 
 * 
 */
class INVENTORW_API SoWinDevice {
 public:
  /**
   * Enables the device for the passed window handle. When enabled, the
   * callback function @B f @b will be invoked when messages occur in the window.
   * @B data @b is the @B clientData @b which will be passed.
   */
  virtual void        enable(SoWidget w, XtEventHandler f,
                             XtPointer data, Window win = NULL) = 0;
  /**
   * Disables the device.
   */
  virtual void        disable(SoWidget w, XtEventHandler f, XtPointer data) = 0;
    
  /**
   * Attempts to convert the passed message into an SoEvent. @BR
   * Returns a pointer to an internal SoEvent object owned by the class.
   * Do not delete this object. If the message was not generated by this 
   * device, then NULL is returned.
   */
  virtual const SoEvent * translateEvent(XAnyEvent *msg) = 0;

  /**
   * Sets the size of the window this device is registered for. This allows
   * the device to correctly convert position information from Windows coordinates
   * (origin at top left) to Open Inventor window coordinates (origin at bottom
   * left). (SoWinRenderArea will automatically call this method for each device
   * registered on it whenever the window size changes.) Note: Setting the window
   * size only affects the size as perceived by the device and not the actual window
   * size.
   */
  void                setWindowSize(const SbVec2s &s) { winSize = s; }
  /**
   * Gets the size of the window this device is registered for. 
   */
  const SbVec2s &     getWindowSize() const           { return winSize; }

 protected:
  /**
   *  Set the event's position from the passed window system x,y value. 
   *  This will flip the y value to convert to Inventor coordinates.
   *  Note : The position is float to handle particular use cases when a desktop is magnified
   *  on a wall of screens when using ScaleViz in correlation with a tracker
   *  device calibrated for this wall.
   */
  inline void         setEventPosition(SoEvent *event, float x, float y) const;

  SoWidget deviceWidget;
    
 private:
  SbVec2s             winSize;    // size of the window this device works in

};

// Inline functions
void
SoWinDevice::setEventPosition(SoEvent *event, float x, float y) const
{
  event->setPosition(SbVec2f(x, (((float)winSize[1]) - 1.f) - y));
}

#include <Inventor/Win/SoWinEndStrict.h>

#endif /* _SO_WIN_DEVICE_ */

