/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_ACTION_
#define  _SO_ACTION_

#include <Inventor/SoLists.h>
#include <Inventor/SoTypedObject.h>
#include <Inventor/SoPath.h>
#include <Inventor/STL/cassert>
#include <SoDeprecationRules.h>
#include <Inventor/misc/SoTempPath.h>
#include <Inventor/misc/SoCompactPathList.h>

class SoEnabledElementsList;
class SoActionMethodList;
class SoSceneManager;
class SoMultiPassManager;
class SoTraversalPass;
class SoActionImpl;

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

/**
 * @if_dotnet Base @else Abstract base @endif class for all actions.
 *
 * @ingroup actions
 *
 * @DESCRIPTION
 *   SoAction is the base class for all actions. Classes derived from
 *   SoAction define operations to be applied at each node encountered during
 *   traversal of a scene graph. The function that gets called to implement the
 *   action for a particular node type is determined by a lookup table in the global
 *   database.
 *
 *   An action may be applied to a node (SoNode), a path (SoPath) or a list of
 *   paths (SoPathList).
 *
 *   \if_cpp
 *   Reference counting:
 *   \par
 *   Warning: Most actions call ref() on the node (or path) before traversing the scene graph,
 *   then call unref() after traversal. @I If the node (or path) has a reference count of zero
 *   (the default), the call to apply() will cause it to be destroyed!@i
 *   A warning message is posted in this case when using a debug build of Open Inventor.
 *  \endif
 *
 *   Hidden references:
 *   \if_cpp
 *   \par
 *   Some actions, e.g. SoSearchAction and SoRayPickAction, create one or more SoPath
 *   objects when they are applied to the scene graph. The SoPath object calls ref()
 *   on each node in the path. This reference will prevent the node from being destroyed
 *   for as long as the SoPath object exists. These SoPath objects are stored internally
 *   in the action and exist until the action object itself is destroyed or reset.
 *   \else
 *   \par
 *   Some actions, e.g. SoSearchAction and SoRayPickAction, create one or more SoPath
 *   objects when they are applied to the scene graph. The SoPath object references
 *   each node in the path. This reference will prevent the node and its associated
 *   memory from being reclaimed for as long as the SoPath object exists. These SoPath
 *   objects are stored internally in the action and exist until the action object
 *   itself is reclaimed or reset.
 *   \endif
 *
 * @SEE_ALSO
 *    SoNode,
 *    SoPath,
 *    SoPathList,
 *    SoCallbackAction,
 *    SoGLRenderAction,
 *    SoGetBoundingBoxAction,
 *    SoGetMatrixAction,
 *    SoHandleEventAction,
 *    SoPickAction,
 *    SoRayPickAction,
 *    SoSearchAction,
 *    SoWriteAction
 *
 * [OIVNET-WRAPPER-CLASS ALL_DERIVABLE]
 */
class INVENTOR_API SoAction: public SoTypedObject {

 public:
  /**
   * Destructor.
   */
  virtual ~SoAction();

  /**
   * Returns the type identifier for this class.
   */
  static SoType       getClassTypeId();

  /**
   * Initiates an action on the graph defined by a node.
   * \if_cpp
   * Warning: Most actions call ref() on the node before traversing the scene
   * graph, then call unref() after traversal. If the node's reference count
   * was zero (the default), the call to apply() will cause it to be destroyed.
   * \endif
   */
  virtual void        apply(SoNode *node);

  /**
   * Initiates an action on the graph defined by a path.
   * \if_cpp
   * Warning: Most actions call ref() on the path before traversing the scene
   * graph, then call unref() after traversal. If the path's reference count
   * was zero (the default), the call to apply() will cause it to be destroyed.
   * \endif
   */
  virtual void        apply(SoPath *path);

  /**
   * Initiates an action on the graph defined by a list of
   * paths. TRUE can be passed for the @B obeysRules @b flag if the given path list
   * has the following 4 properties:
   *   -# All paths have the same head node.
   *   -# Paths are sorted in traversal order.
   *   -# If one path ends at node @B A@b, no other path continues through @B A@b.
   *   -# No two paths are the same.
   *
   * These rules will be obeyed by path lists returned by picking and by searches for
   * non-group nodes.
   */
  virtual void        apply(const SoPathList &pathList,
                            SbBool obeysRules = FALSE);

  /**
   * When applied, an action may reference nodes or create objects (e.g. SoPath)
   * that reference nodes.  This is especially true for SoSearchAction and
   * SoRayPickAction.  These references will prevent the nodes from being destroyed
   * and so may appear to be a "memory leak".
   *
   * All references are cleared when the action is destroyed or re-applied.
   * However it may be useful to clear them explicitly to remove references to nodes.
   */
  virtual void clearApplyResult() {};

  /**
   * Invalidates the current traversal state in the action, forcing it to be
   * recreated when the action is next applied. This is typically unnecessary in most
   * applications.
   */
  virtual void        invalidateState();

  /**
   * This function stops the action in the current Scene Graph branch.
   * Note: It doesn't mean we kill the action!
   *
   * It is mainly used to stop the GetBoundingBox action in the SoBBox node.
   */
  void stopActionInBranch()
  { m_continueInBranchFlag = FALSE; }

  /**
   * This function indicates if the action must stop in the current branch.
   *
   * @return TRUE if the action must continue for this branch.
   */
  SbBool getContinueActionInBranchFlag() const
  { return m_continueInBranchFlag; }

  /**
   * This function resets the continue action flag.
   */
  void resetContinueActionInBranchFlag()
  { m_continueInBranchFlag = TRUE; }

  /**
   * Tell the action to use alternate representations during traversal when available. @BR
   * See SoNode::getAternateRep(). Default is FALSE.
   * When true, it is the responsibility of each SoNode derived
   * object to choose to provide or not an alternate representation for
   * a given action.
   */
  void useAlternateRep(const SbBool enable);

  /**
   * Returns TRUE if current action is using alternate representations.
   * Default is FALSE.
   */
  SbBool isUsingAlternateRep() const;

  /** 
   * Set the scene manager associated with this action (if any ).
   * When appropriate, this allows the action to query information like the OpenGL
   * rendering context.  The SoSCeneManager class calls this method automatically
   * for the actions that it creates (SoHandleEventAction, SoRayPickAction, etc).
   */
  void setSceneManager(SoSceneManager* mgr)
  { m_sceneManager = mgr; }

  /**
  * Return the SoSceneManager associated with this action.
  */
  SoSceneManager* getSceneManager() const
  { return m_sceneManager; }

  /**
   * Traverse a node that is not part of the current scenegraph.
   *
   * Like an SoNodeKit, the contents of the specified sub-scenegraph won't be visible
   * as an SoPath.
   * \if_cpp Casting the path to a SoFullPath allows access to the whole path.\endif
   * \if_dotnet Using the SoPath.FullPath property allows access to the whole path.\endif
   * \if_java Using the SoPath.full field allows access to the whole path.\endif
   *
   * This method can be used to delegate an action to an internal/private scene graph.
   * So a node can have an internal scene graph without actually being a "group"
   * node. This is a convenient way to implement "composite" nodes that create
   * their behavior using other Open Inventor nodes. 
   *
   * The node parameter can be of any type.
   * Sensors may still be attached directly to nodes and fields in the internal scene graph.
   *
   * This method must not be called from a node derived from SoGroup or SoBaseKit.
   * This method should not be called on nodes that are part of the public/visible scene graph.
   */
  virtual void forwardTraversal( SoNode* node );

  /**
   * Traverse a path that is not part of the current scenegraph.
   * Similar to forwardTraversal(SoNode*) but for paths.
   */
  virtual void forwardTraversal( SoPath* path );

  // SoEXTENDER section: the following methods are dedicated to extend the toolkit only
  public:

  /**
   * This is used to define the way an action is distributed across a cluster.
   * @EXTENDER_API
   */
  enum DistribMode {
    /** Action is only applied by the application. */
    LOCAL_ONLY,
    /** Action is only applied by OIRU of the cluster */
    CLUSTER_ONLY,
    /** Both application and cluster apply the action */
    ALL
  };

  /**
   * Returns the distribution mode of this action across a cluster (ScaleViz-Cluster only).
   * Default is LOCAL_ONLY.
   * @EXTENDER_API
   */
  DistribMode getDistribMode() const
  { return m_distribMode; }

  /**
   * Null action method that can be stored in lookup table when desired.
   * @EXTENDER_API
   */
  static void         nullAction(SoAction *, SoNode *);

  /**
   * This enum is used to determine what the action is being applied to.
   * @EXTENDER_API
   */
  enum AppliedCode {
    /** Applied to graph rooted by a node */
    NODE,
    /** Applied to graph defined by a path */
    PATH,
    /** Applied to graphs defined by list of paths */
    PATH_LIST
  };

  /**
   * This enum may be used during traversal of nodes to indicate
   * where the node is with respect to the path being traversed.
   * @EXTENDER_API
   */
  enum PathCode {
    /** Not traversing a path */
    NO_PATH,
    /** In middle of path chain (not tail node) */
    IN_PATH,
    /** Tail node of path or below tail node */
    BELOW_PATH,
    /** None of the above (Probably to the left) */
    OFF_PATH
  };

  /**
   * Returns code indicating what action is being applied to.
   * @EXTENDER_API
   */
  AppliedCode         getWhatAppliedTo() const   { return appliedTo.code; }

  /**
   * Returns the node the action is being applied to.
   * Returns NULL if the action is not being applied to the appropriate class.
   * @EXTENDER_API
   */
  SoNode *            getNodeAppliedTo() const  { return appliedTo.node.ptr(); }

  /**
   * Returns the path the action is being applied to.
   * Returns NULL if the action is not being applied to the appropriate class.
   * @EXTENDER_API
   */
  SoPath *            getPathAppliedTo() const  { return appliedTo.path.ptr(); }

  /**
   * Returns the path list the action is being applied to.
   * Returns NULL if the action is not being applied to the appropriate class.
   * @note A single path list may be split into several, one for each
   * different head node. This method allows subclasses to
   * determine the current path list.
   * @EXTENDER_API
   */
  const SoPathList *  getPathListAppliedTo() const
    { return appliedTo.pathList; }

  /**
   * Returns the original path list the action is being applied to.
   * Returns NULL if the action is not being applied to the appropriate class.
   * @note A single path list may be split into several, one for each
   * different head node. This method allows subclasses to
   * determine the original path list.
   * @EXTENDER_API
   */
  const SoPathList *  getOriginalPathListAppliedTo() const
    { return appliedTo.origPathList; }

  /**
   * Returns TRUE if the current list is the last one from the original.
   * @note A single path list may be split into several, one for each
   * different head node. This method allows subclasses to
   * determine whether the current list is the last one from the original.
   * @EXTENDER_API
   */
  SbBool              isLastPathListAppliedTo() const
    { return appliedTo.isLastPathList; }

  /**
   * Returns path code based on where current node (the node at the
   * end of the current path) lies with respect to the path(s) the
   * action is being applied to. If this returns IN_PATH, indices is
   * set to point to an array of indices corresponding to the
   * children that must be traversed for a correct traversal of all
   * the paths to which the action is being applied. numIndices is set
   * to the number of such children.
   * @EXTENDER_API
   * [OIV-WRAPPER-ARG NO_WRAP,ARRAY{numIndices}]
   * [OIVJAVA-WRAPPER PACK{PathIndices}]
   */
  PathCode    getPathCode(int &numIndices, const int *&indices)
    {   if (appliedTo.curPathCode == IN_PATH) {
      usePathCode(numIndices, indices);
    }
    return appliedTo.curPathCode;
    }

  /**
   * @EXTENDER_API
   */
  PathCode    getPathCodeMI(int &numIndices, const int *&indices, const int *&instanceIndices)
  {
    if (appliedTo.curPathCode == IN_PATH) {
      usePathCodeMI(numIndices, indices, instanceIndices);
    }
    return appliedTo.curPathCode;
  }

  /**
   * Does traversal of a graph rooted by a node.
   * @EXTENDER_API
   */
  void                traverse(SoNode *node);

  /**
   * Returns TRUE if the traversal has reached a termination condition.
   * @EXTENDER_API
   */
  SbBool              hasTerminated() const   { return terminated; }

  /**
   * Returns the path accumulated during traversal,
   * i.e., the chain of nodes from the root of the traversed graph
   * to the current node being traversed.
   * @EXTENDER_API
   */
  const SoPath *      getCurPath();

  /**
   * Gets the state from the action.
   * @note Will return NULL if action has not been applied to anything
   *       or if invalidateState has been called.
   * @EXTENDER_API
   */
  SoState* getState() const
  { return m_state; }

  // Changed to virtual so that derived actions that need to do
  // things to the state such as the GLRenderAction. This action
  // needs to change the value of whatChanged to ALL if the state has
  // been recreated because the number of elements has changed.

  /**
   * Creates state if it is NULL or it is no longer valid because new
   * elements have been enabled since it was created.
   * Automatically called by the apply() methods, but can be useful for
   * application to get a non-null state pointer to "pre-load" certain elements.
   * @EXTENDER_API
   */
  virtual void                setUpState();

  /**
   * Returns the list of enabled elements for a given action subclass.
   * @EXTENDER_API
   * [OIVNET-WRAPPER-NO-WRAP]
   */
  virtual const SoEnabledElementsList & getEnabledElements() const;

  /**
   * Sets pipe identifier in the range [1..N] associated to this render action
   * while running a ScaleViz Multipipe configuration.
   * @EXTENDER_API
   */
  void setPipeId( int id )
  {  m_pipeId = id; }

  /**
   * Gets pipe identifier in the range [1..N] associated to this render action
   * while running a ScaleViz Multipipe configuration.
   * @EXTENDER_API
   */
  int getPipeId() const
  { return m_pipeId; }

  /**
   * Returns TRUE if this action is currently being applied.
   * @EXTENDER_API
   */
  bool isBeingApplied() { return m_isBeingApplied; }

  /** 
   * Method called by SoMultiPassManager before delayed pass traversals.
   * Overriding this method can be useful for actions that want to skip the traversal
   * of all delayed passes. Returning false skips the delayed pass traversals.
   * @M_SINCE 9.1
   * @EXTENDER_API
   */
  virtual bool preDelayedTraversal()
  { return true; }

  /**
   * Method called by SoMultiPassManager after delayed pass traversals.
   * Overriding this method can be useful for actions that want to apply some post
   * processing after all delayed pass traversals.
   * @M_SINCE 9.1
   * @EXTENDER_API
   */
  virtual void postDelayedTraversal()
  {}

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

  // Initialize ALL Inventor action classes
  static void         initClasses();
  static void         exitClasses();

  /** Returns TRUE if given elements is enabled for this action. */
  virtual bool isEnabledElement( const SoType& ) const;

  // These methods maintain the current path accumulated so far
  // during traversal. The action needs to know whether this path is
  // a subset of the path being applied to; it saves this info in
  // the onPath variable. Before a node is pushed onto the current
  // path, call getOnPath() to determine the current setting. The
  // value of this flag should be passed in to popCurPath() so the
  // onPath variable can be restored.
  PathCode            getCurPathCode() const { return appliedTo.curPathCode;}
  // lastFound is an inout parameter that contains the last indices found in the compactPathList
  // for the current separator. When iterating over children of a separator it
  // should be set to 0 on first call then the returned result has to be reused until the last
  // call for the separator.
  void                pushCurPath(int childIndex, int &lastFound);
  // lastFound is an inout parameter that contains the last indices found in the compactPathList
  // for the current separator. When iterating over children of a separator it
  // should be set to 0 on first call then the returned result has to be reused until the last
  // call for the separator.
  void                pushCurPathMI(int childIndex, int &lastFound, int instanceIndex);
  void                popCurPath(PathCode prevPathCode);
  // This is virtual, so that SoCallbackAction can use current node.
  virtual SoNode *    getCurPathTail();

  // ask if action is currently forward traversed
  bool isForwardTraversing() const { return appliedTo.isForwardTraversing;  }

  // get the public node if action is forward traversed
  SoNode* getForwardTraversalCaller() const { return appliedTo.forwardTraversalCaller.ptr(); }

  // called by inline getPathCode:
  void                usePathCode(int &numIndices, const int *&indices);
  // called by inline getPathCode:
  void                usePathCodeMI(int &numIndices, const int *&indices, const int *&instanceIndices);

  // Optimized versions of push/pop when we know path codes won't
  // change:
  void                pushCurPath()
    { curPath.append(-1); }
  void                popPushCurPath(int childIndex)
    { curPath.setTail(childIndex);}
  // generalization for multi-instancing
  void                popPushCurPathMI(int childIndex, int instanceIndex)
  { curPath.setTail(childIndex, instanceIndex);}
  void                popCurPath()
    { curPath.pop(); }

  // Returns true if the state has been reset since action traversal startup
  // ie. setupState recreated the state because of some new elements enabled.
  SbBool hasChangedState() const
  { return m_hasStateChanged; }
#if SoDEPRECATED_BEGIN(9620)
  /**
   * @copydoc SoAction::forwardTraversal( SoNode* node )
   */
  SoDEPRECATED_METHOD( 9620, "Use SoAction::forwardTraversal( SoNode* node ) instead." )
  virtual void switchToNodeTraversal( SoNode* node );

  /**
   * @copydoc SoAction::forwardTraversal( SoPath* path )
   */
  SoDEPRECATED_METHOD( 10600, "Use SoAction::forwardTraversal( SoPath* path ) instead." )
  void switchToPathTraversal( SoPath* path )
  {
    forwardTraversal( path );
  }
#endif /** @DEPRECATED_END */


  DistribMode getDistribMode() { return m_distribMode; }


  // Enter critical section for this action
  inline void lock()
  { m_criticalSection->lock(); }

  // Exit critical section for this action
  inline void unlock()
  { m_criticalSection->unlock(); }


  /**
   * Returns TRUE is the action can be abort due to some interactive events
   * If an action is abortable then shouldAbort() calls may returns TRUE or FALSE
   * depending on interactive pending events.
   *
   * Default is FALSE.
   */
  inline bool isAbortable() const
  { return m_isAbortable; }

  /**
   * Setup if the action is abortable or not. Default is FALSE
   */
  inline void setAbortable(bool flag)
  { m_isAbortable = flag; }

  /**
   * Returns true if the action calling the method (during its traversal) should abort
   * in order to respect some interactivity condition
   */
  bool shouldAbort() const;

  /**
   * Allows a nodes to completely abort the action in order to get back interactivity
   * to the application.
   * - call setTerminated on the action for current Pass traversal
   * - call setTerminated on the MultipassManager subsequent pass traversal
   * Application should managed aborted action in order to ensure that it will not use
   * its partial results.
   *
   * A typical example is the SoGLRenderAction that can be aborted by default.
   * In this case the SoSceneManger will check if the result of the action has been aborted
   * if so it does not swapBuffer and schedule a new redraw in order to finally get a complete frame latter. 
   */
  void abort();

  /**
   * Returns TRUE if action has been aborted.
   */
  bool isAborted() const;

 protected:
  // Constructor
  SoAction();

  // WARNING : Copy constructor should never be called.
  // WARNING : Implemented here only to add a debug message to derived class.
  SoAction(const SoAction&);

  // Begins traversal of an action at the given node. The default
  // method just calls traverse(node). This is virtual to allow
  // subclasses to do extra work before or after traversing the node.
  virtual void        beginTraversal(SoNode *node);

  // Ends traversal of an action at the given node. By default
  // nothing is gone
  virtual void        endTraversal(SoNode *) { ; }

  // Allows subclass instance to indicate that traversal has reached
  // a termination condition
  void                setTerminated(SbBool flag)      { terminated = flag; }

  // This method is used when applying an action to an SoPathList.
  // It returns TRUE if the action should create a compact version
  // of the path list before applying itself to it. The default
  // method returns TRUE, since the compact version is more
  // efficient. Some actions may choose to return FALSE; for
  // example, the SoWriteAction applies itself to each path
  // separately, so it doesn't need the extra overhead of compacting
  // the list.
  virtual SbBool      shouldCompactPathLists() const;

  /**
   * Setup the distribution mode of this action across a cluster (ScaleViz-Cluster only)
   * Default is LOCAL_ONLY.
   */
  void setDistribMode(DistribMode mode )
  { m_distribMode = mode; }

SoINTERNAL protected:

  // Holds list of enabled elements for the SoAction class
  static SoEnabledElementsList *enabledElements;

  // ... and the methods
  static SoActionMethodList *methods;

  SoLightPath* getCurLightPath()
  { return &curPath; }

 SoINTERNAL public:

  virtual SoActionMethodList* getActionMethods();
  SbBool getUseAlternateRep() const
  { return m_useAlternatRep; }

  SbBool isForcedUseAlternateRep() const
  { return m_forceUseAlternateRep; }

  SoMultiPassManager* getMultiPassManager() const
  { return m_multiPassManager; }

  bool isApplyingDelayed() const;

  SoActionImpl* getImpl() { return m_impl; }

 private:
  // Current Traversal state
  SoState* m_state;

  // Type identifier
  static SoType      classTypeId;
  static bool s_defaultAlternateRepUsage;

  // Critical section use in multi-thread to protect the action state
  SbThreadMutex* m_criticalSection;

  // This indicates if we must stop this action in the current Scene Graph branch
  SbBool m_continueInBranchFlag;

  // This is the current path through the graph the action is being applied to
  SoLightPath curPath;

  SbBool m_hasStateChanged;

  // Current Distribution mode (ScaleViz Cluster)
  DistribMode m_distribMode;

  // This structure holds the necessary information about what the
  // action is being applied to. It allows this information to be
  // saved and restored when necessary.
  struct AppliedTo {
    AppliedCode code;
    SoRef<SoNode> node;             // If code is NODE
    SoRef<SoPath> path;             // If code is PATH
    const SoPathList *pathList;       // If code is PATH_LIST
    const SoPathList *origPathList;
    SoRef<SoCompactPathList> compactPathList;
    int isLastPathList;
    PathCode curPathCode;
    bool isForwardTraversing;  // true if the action is inside a forward traversal
    SoRef<SoNode> forwardTraversalCaller; // node that call forwardTraversal, if any.

    // Default Constructor
    AppliedTo();
  } appliedTo;

  // This is set to TRUE while the action is being applied, so we
  // can tell if the action is being applied recursively.
  bool m_isBeingApplied;

  // Indicate if we should ask for alternate representation or not
  // default is FALSE.
  SbBool m_useAlternatRep;

  // Indicate if we force for using alternate representation or not
  // default is FALSE.
  SbBool m_forceUseAlternateRep;

  // This holds the current path (nodes and childindices) whenever
  // action->getCurPath is called:
  SoRef<SoTempPath>       tempPath;


  // Holds enabled-elements counter when state is created; used to
  // determine whether list of enabled elements is up to date.
  int                 enabledElementsCounter;

  // This is TRUE if the action has reached a termination condition
  SbBool              terminated;

  // Stores next index in path being applied to so a pointer to it
  // can be returned
  int                 index;


  // Cleans up after an action is done
  void                cleanUp();

  // Splits a sorted path list based on head nodes, then applies to each
  void                splitPathList(const SoPathList &sortedList,
                                    const SoPathList &origPathList);

  // Applies action to sorted path list
  void                apply(const SoPathList &sortedList,
                            const SoPathList &origPathList,
                            SbBool isLastPathList);

  int m_pipeId;   // this render action pipe id

  SoSceneManager* m_sceneManager;

  SoMultiPassManager* m_multiPassManager;

  bool m_isAbortable;

  SoActionImpl* m_impl;

  friend class SoDB;
  friend class SoMultiPassElement;
};

#ifndef HIDDEN_FROM_DOC
//////////////////////////////////////////////////////////////////////////////
//
//  Classes can use this macro to enable a specific element class
//  within a specific action class. E.g.:
//
//      SO_ENABLE(SoGetBoundingBoxAction, SoModelMatrixElement);
//

#define SO_ENABLE(actionClass, elementClass)                            \
  if ( elementClass::getClassTypeId().isBad() )                         \
          elementClass::initClass();                                    \
        actionClass::enableElement(elementClass::getClassTypeId(),      \
                                   elementClass::getClassStackIndex())

//
//////////////////////////////////////////////////////////////////////////////

#include <Inventor/lists/SoActionMethodList.h>
#include <Inventor/lists/SoEnabledElementsList.h>

#endif // HIDDEN_FROM_DOC

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif /* _SO_ACTION_ */

