/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Dave Immel (MMM yyyy)
** Modified by : Thad Beier (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_CALLBACK_ACTION_
#define  _SO_CALLBACK_ACTION_

#include <Inventor/SbBox.h>
#include <Inventor/actions/SoSubAction.h>
#include <Inventor/nodes/SoComplexity.h>
#include <Inventor/nodes/SoDrawStyle.h>
#include <Inventor/nodes/SoFont.h>
#include <Inventor/nodes/SoLightModel.h>
#include <Inventor/nodes/SoMaterialBinding.h>
#include <Inventor/nodes/SoNormalBinding.h>
#include <Inventor/nodes/SoPickStyle.h>
#include <Inventor/nodes/SoShapeHints.h>
#include <Inventor/nodes/SoTexture.h>
#include <Inventor/nodes/SoTextureCoordinateBinding.h>
#include <Inventor/nodes/SoUnits.h>
#include <Inventor/elements/SoDecimationTypeElement.h>
#include <Inventor/actions/SoGLRenderAction.h>

class SoPrimitiveVertex;
class SoShape;
class SoTexture2Transform;

/**
 * Typedef for callback routine used with the callbacks for generating triangles primitives.
 *
 * @memberof SoCallbackAction
 *
 * [OIV-WRAPPER NAME{TriangleCB}]
 * [OIV-WRAPPER-ARG IN,IN,IN,IN,IN]
 */
typedef void SoTriangleCB(void *userData,
                          SoCallbackAction *action,
                          const SoPrimitiveVertex *v1,
                          const SoPrimitiveVertex *v2,
                          const SoPrimitiveVertex *v3);
/**
 * Typedefs for callback routine used with the callbacks for generating line segments primitives.
 *
 * @memberof SoCallbackAction
 *
 * [OIV-WRAPPER NAME{LineSegmentCB}]
 * [OIV-WRAPPER-ARG IN,IN,IN,IN]
 */
typedef void SoLineSegmentCB(void *userData, 
                             SoCallbackAction *action,
                             const SoPrimitiveVertex *v1,
                             const SoPrimitiveVertex *v2);
/**
 * Typedefs for callback routine used with the callbacks for generating points primitives.
 *
 * @memberof SoCallbackAction
 *
 * [OIV-WRAPPER NAME{PointCB}]
 * [OIV-WRAPPER-ARG IN,IN,IN]
 */
typedef void SoPointCB(void *userData, 
                       SoCallbackAction *action,
                       const SoPrimitiveVertex *v);

#ifdef _WIN32
//JSG - Changed SoCallbackActionCB back to how it is done under unix,
//SoCallbackActionCB2 is included for backward compatibility on Windows
typedef int SoCallbackActionCB2(void *userData,
                                SoCallbackAction *action,
                                const SoNode *node);
#endif
//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoCallbackAction
//
//  State accumulation action. This action performs a generic
//  traversal of a scene graph. The caller can register callback
//  functions to be called when specific types of nodes are traversed.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Performs a generic traversal of a scene graph or path.
 * 
 * @ingroup actions
 * 
 * @DESCRIPTION
 *   Performs a generic traversal of a scene graph or path. In many cases a
 *   "custom" action can be implemented using SoCallbackAction
 *   with much less complexity than deriving a completely new action class.
 *
 *   The application can specify \if_dotnet delegate \else callback \endif
 *   methods that will be called when the action traverses a node of the
 *   specified class or a derived class.  For example a callback for the SoCone
 *   class will only be called for SoCone nodes, but a callback for the SoShape
 *   class will be called for any type of shape node.
 *
 *   This makes SoCallbackAction flexible and very convenient for "visiting"
 *   a set of nodes, especially when you only need the node itself and not a
 *   path to the node. For example, to count all the shapes in the scene graph.
 *   It is possible to get the path to the node being visited (see
 *   \if_dotnet GetCurPath() \else getCurPath() \endif ), but SoSearchAction
 *   may be more convenient for getting a path to one or a small number of nodes.
 *   
 *   In addition, \if_dotnet delegate \else callback \endif methods can be registered
 *   to collect the primitives (trangles, lines and points) generated by
 *   shapes in the scene graph. These primitives are the actual geometry if the shape
 *   contains triangles, lines or points, else they represent or approximate the actual
 *   geometry.  @B NOTE: @b Most, but not all, shapes can generate primitives.
 *   See below for a complete list. @B Triangle @b primitives are used
 *   for all surfaces (such as cubes, face sets, or 3D text), @B line segment @b
 *   primitives are used for line shapes, and @B point @b primitives are used for
 *   point shapes. Note that the type of primitives generated for a shape is the
 *   same, regardless of drawing style or other properties.
 *   
 *   Most of the methods on this class access information from the traversal state.
 *   They should be called only by \if_dotnet delegate \else callback \endif functions that are invoked during
 *   traversal, so there is a valid state to work with.
 *
 * @EXAMPLE
 * \if_cpp
 *   Apply a callback action to visit all shape nodes.
 *   \code
 *   SoCallbackAction action;
 *   action.addPreCallback     ( SoShape::getClassTypeId(), NodeCallback, NULL );
 *   action.addTriangleCallback( SoShape::getClassTypeId(), TriangleCallback, NULL );
 *   action.apply( root );
 *   \endcode
 *   This method will be called for each shape node in the scene graph.
 *   \code
 *   SoCallbackAction::Response
 *   NodeCallback( void*, SoCallbackAction*, const SoNode* node)
 *   {
 *     // Print node's name (if any) and class name
 *     std::cout << "Shape \"" << node->getName().getString() << "\": "
 *               << node->getTypeId().getName().getString()
 *               << std::endl;
 *      return SoCallbackAction::CONTINUE;
 *   }
 *   \endcode
 *   This method will be called for each triangle in the shape node.
 *   \code
 *   // This method will be called for each triangle in the shape node.
 *   void TriangleCallback( void*, SoCallbackAction*,
 *                          const SoPrimitiveVertex *vertex1,
 *                          const SoPrimitiveVertex *vertex2,
 *                          const SoPrimitiveVertex *vertex3)
 *   {
 *      std::cout << "  Triangle:\n";
 *      printVertex( vertex1, "1" );
 *      printVertex( vertex2, "2" );
 *      printVertex( vertex3, "3" );
 *   }
 *
 *   void
 *   printVertex( const SoPrimitiveVertex *vertex, const char* label )
 *   {
 *      const SbVec3f& point = vertex->getPoint();
 *      std::cout << "    Vert" << label << " = " << point << std::endl;
 *   }
 *   \endcode
 * \endif
 * \if_dotnet
 *   Apply a callback action to visit all shape nodes.
 *   \code
 *   SoCallbackAction action = new SoCallbackAction();
 *   action.AddPreCallback     ( typeof(SoShape), new SoCallbackAction.CallbackActionCB(NodeCallback) );
 *   action.AddTriangleCallback( typeof(SoShape), new SoCallbackAction.TriangleCB(TriangleCallback) );
 *   action.Apply( root );
 *   \endcode
 *   This method will be called for each shape node in the scene graph.
 *   \code
 *   SoCallbackAction.Responses NodeCallback(SoCallbackAction action, SoNode node)
 *   {
 *       Console.WriteLine("Shape \"" + node.GetName() + "\": " + node.GetType());
 *       return SoCallbackAction.Responses.CONTINUE;
 *   }
 *   \endcode
 *   This method will be called for each triangle in the shape node.
 *   \code
 *   void TriangleCallback( SoCallbackAction action,
 *                 ref SoPrimitiveVertex vertex1,
 *                 ref SoPrimitiveVertex vertex2,
 *                 ref SoPrimitiveVertex vertex3)
 *   {
 *       Console.WriteLine("  Triangle:");
 *       printVertex( vertex1, "1" );
 *       printVertex( vertex2, "2" );
 *       printVertex( vertex3, "3" );
 *   }
 *   void printVertex( SoPrimitiveVertex vertex, String label )
 *   {
 *       SbVec3f point = vertex.Point;
 *       Console.WriteLine("    Vert{0} = ({1}, {2}, {3})", label, point[0], point[1], point[2]);
 *   }
 *   \endcode
 * \endif
 * \if_java
 *   Apply a callback action to visit all shape nodes.
 *   \code
 *   SoCallbackAction action = new SoCallbackAction();
 *   action.addPreCallback     ( SoShape.class, new NodeCallback(), null );
 *   action.addTriangleCallback( SoShape.class, new TriangleCallback(), null );
 *   action.apply( root );
 *   \endcode
 *   This method will be called for each shape node in the scene graph.
 *   \code
 *   class NodeCallback extends SoCallbackActionCB {
 *       @Override
 *       public int invoke( SoCallbackAction s, SoNode node ) {
 *           // Print node's name (if any) and class name
 *           System.out.println( "Shape \"" + node.getName() + "\": " + node.getClass().getName() );
 *           return SoCallbackAction.Responses.CONTINUE.getValue();
 *       }
 *   }
 *   \endcode
 *   This method will be called for each triangle in the shape node.
 *   \code
 *   class TriangleCallback extends SoTriangleCB {
 *       @Override
 *       public void invoke( SoCallbackAction action, SoPrimitiveVertex v1,
 *                           SoPrimitiveVertex v2, SoPrimitiveVertex v3 ) {
 *           System.out.println("  Triangle:");
 *           printVertex(v1, "1");
 *           printVertex(v2, "2");
 *           printVertex(v3, "3");
 *       }
 *
 *       private void printVertex(SoPrimitiveVertex vertex, String label) {
 *           SbVec3f point = vertex.getPoint();
 *           System.out.println("    Vert" + label + " = " + point.getX() + ", "
 *                   + point.getY() + ", " + point.getZ());
 *       }
 *   }
 *   \endcode
 * \endif
 *
 * Example output:
 * \verbatim
 Shape "": SoCone
   Triangle:
     Vert1 = 0.0, -1.0, -1.0
     Vert2 = 0.0, 1.0, -0.0
     Vert3 = 0.38268346, -1.0, -0.9238795
   Triangle:
     Vert1 = 0.38268346, -1.0, -0.9238795
     Vert2 = 0.0, 1.0, -0.0
     Vert3 = 0.0, 1.0, -0.0
   . . .
 * \endverbatim
 * 
 * @SEE_ALSO
 *    SoCallback,
 *    SoEventCallback,
 *    SoShape
 * 
 * 
 */
class INVENTOR_API SoCallbackAction : public SoAction {

  SO_ACTION_HEADER(SoCallbackAction);

 public:

  /** Possible responses from a pre or post callback */
  enum Response {
    /**
     *  Continue traversal as if nothing happened 
     */
    CONTINUE,
    /**
     *  Abort traversal 
     */
    ABORT,
    /**
     *  Do not traverse node's children, but continue traversal 
     */
    PRUNE
  };

  /** The SoCallbackActionCB typedef is defined within the class, since
  * it needs to refer to the Response enumerated type.
  * The first argument is the data pointer that the user supplied
  * when the callback was registered. The second argument is the
  * action, from which the state can be extracted. The third
  * argument is the node that the callback is called from.  
  * [OIV-WRAPPER NAME{CallbackActionCB}]
  */
  typedef Response SoCallbackActionCB(void *userData,
                                      SoCallbackAction *action,
                                      const SoNode *node);

  /**
   * The constructor.
   */
  SoCallbackAction();

  // Destructor
#ifndef HIDDEN_FROM_DOC
  virtual ~SoCallbackAction();
#endif // HIDDEN_FROM_DOC

  //////////////////////////////////////////////////////////////////
  //
  // Adding callback functions:
  //

  /**
   * Adds a callback function to call when a node of the given type is
   * encountered during traversal. The PreCallback is called just before the node is
   * traversed, and the PostCallback is called just after. The value returned by a
   * callback function indicates whether the action should continue with the
   * traversal.
   */
  void addPreCallback(SoType type, SoCallbackActionCB *cb, void *data);

  /**
   * Adds a callback function to call when a node of the given type is
   * encountered during traversal. The PreCallback is called just before the node is
   * traversed, and the PostCallback is called just after. The value returned by a
   * callback function indicates whether the action should continue with the
   * traversal.
   */
  void addPostCallback(SoType type, SoCallbackActionCB *cb, void *data);

  /**
   * Sets up a callback function to call when the action is
   * applied to a path. This function is called just before the node at the
   * tail of the path is traversed.
   * [OIV-WRAPPER EVENT_NAME{PreTailTraversal},EVENT_INVOKE_METHOD{InvokePreTailEventDelegates}]
   */
  void addPreTailCallback(SoCallbackActionCB *cb, void *data);

  /**
   * Sets up a callback function to call when the action is
   * applied to a path. This function is called just after the node at the
   * tail of the path is traversed.
   * [OIV-WRAPPER EVENT_NAME{PostTailTraversal},EVENT_INVOKE_METHOD{InvokePostTailEventDelegates}]
   */
  void addPostTailCallback(SoCallbackActionCB *cb, void *data);

  /**
   * Routine to add callbacks for generated primitives (triangles)
   * for all shapes of the given type. The callback function will be
   * called for each triangle generated for all shapes of or derived from that type.
   */
  void addTriangleCallback(SoType type, SoTriangleCB *cb, void *data);

  /**
   * Routine to add callbacks for generated primitives (line segments)
   * for all shapes of the given type. The callback function will be
   * called for each line segment generated for all shapes of or derived
   * from that type.
   */
  void addLineSegmentCallback(SoType type, SoLineSegmentCB *cb, void *data);

  /**
   * Routine to add callbacks for generated primitives (points)
   * for all shapes of the given type. The callback function will be
   * called for each point generated for all shapes of or derived from that type.
   */
  void addPointCallback(SoType type, SoPointCB *cb, void *data);

  //////////////////////////////////////////////////////////////////
  //
  // Accessing state information from callbacks. Non-obvious return
  // values are documented.
  //

  //////////////////////// Complexity:
  /**
   * Returns complexity information from the state.
   */
  float getComplexity() const;

  /**
   * Returns complexity information from the state.
   */
  SoComplexity::Type getComplexityType() const;

  //////////////////////// Coordinates:
  /**
   * Returns the current coordinates from the state.
   */
  int32_t getNumCoordinates() const;

  /**
   * Returns the current coordinates from the state.
   */
  const SbVec3f &getCoordinate3(int index) const;

  /**
   * Returns the current coordinates from the state.
   */
  const SbVec4f &getCoordinate4(int index) const;

  /** 
   * Returns the current decimation type from the state.
   */
  SoDecimationTypeElement::Type getDecimationType() const;

  /** 
   * Returns the current decimation percentage from the state.
   */
  float getDecimationPercentage() const;

  /**
   * Returns the current drawing style information from the state.
   */
  SoDrawStyle::Style getDrawStyle() const;

  /**
   * Returns the current drawing style information from the state.
   */
  u_short getLinePattern() const;

  /** 
   * Returns the current drawing style information from the state.
   */
  int getLinePatternScaleFactor() const;

  /**
   * Returns the current drawing style information from the state.
   */
  float getLineWidth() const;

  /**
   * Returns the current drawing style information from the state.
   */
  float getPointSize() const;

  //////////////////////// Font:
  /**
   * Returns the current font information from the state.
   *
   */
  const SbString & getFontName() const;

  /**
   * Returns the current font information from the state.
   */
  float getFontSize() const;

  /**
   * Returns the current font information from the state.
   */
  SoFont::RenderStyle getFontRenderStyle() const;

  //////////////////////// Lighting:
  /**
   * Returns the current lighting model information from the state.
   */
  SoLightModel::Model getLightModel() const;

  /**
   * Returns the current lighting model information from the state.
   */
  const SbVec3f &getLightAttenuation() const;

  //////////////////////// Material:
  /**
   * Returns the current material information from the state. Providing a
   * @B mtlIndex @b will return the material defined for that index.
   * [OIVJAVA-WRAPPER PACK{Material}]
   */
  void getMaterial(SbColor &ambient,
                   SbColor &diffuse,
                   SbColor &specular,
                   SbColor &emission,
                   float &shininess,
                   float &transparency,
                   int mtlIndex = 0) const;

  /**
   * Returns the current material information from the state.
   */
  SoMaterialBinding::Binding getMaterialBinding() const;


  /**
   * Returns the current Transparency Type information from the state.
   */
  SoGLRenderAction::TransparencyType getTransparencyType() const;

  //////////////////////// Normals:
  /**
   * Returns the current normal information from the state.
   */
  int32_t getNumNormals() const;

  /**
   * Returns the current normal information from the state.
   */
  const SbVec3f & getNormal(int index) const;

  /**
   * Returns the current normal information from the state.
   */
  SoNormalBinding::Binding getNormalBinding() const;

  //////////////////////// Profiles:
  /**
   * Returns the current profiles and their coordinates from the state.
   */
  int32_t getNumProfileCoordinates() const;

  /**
   * Returns the current profiles and their coordinates from the state.
   */
  const SbVec2f &getProfileCoordinate2(int index) const;

  /**
   * Returns the current profiles and their coordinates from the state.
   */
  const SbVec3f &getProfileCoordinate3(int index) const;

  /**
   * Returns the current profiles and their coordinates from the state.
   */
  const SoNodeList & getProfile() const;

  //////////////////////// Shape Hints:
  /**
   * Returns the current shape hints from the state.
   */
  SoShapeHints::VertexOrdering getVertexOrdering() const;

  /**
   * Returns the current shape hints from the state.
   */
  SoShapeHints::ShapeType getShapeType() const;

  /**
   * Returns the current shape hints from the state.
   */
  SoShapeHints::FaceType getFaceType() const;

  /**
   * Returns the current shape hints from the state.
   */
  float getCreaseAngle() const;

  //////////////////////// Textures:
  /**
   * Returns texture information from the state. getNumTextureCoordinates()
   * returns 0 if texture coordinates are generated by a function.
   */
  int32_t getNumTextureCoordinates() const;

  /**
   * Returns texture information from the state. 
   */
  const SbVec2f &getTextureCoordinate2(int index) const;

  /**
   * Returns texture information from the state. 
   */
  const SbVec4f &getTextureCoordinate4(int index) const;

  /**
   * Returns texture information from the state. 
   */
  SoTextureCoordinateBinding::Binding   getTextureCoordinateBinding() const;

#ifdef _WIN32
  /**
   * Returns texture information from the state. 
   */
  const SbColor getTextureBlendColor() const;
#else
  const SbColor &getTextureBlendColor() const;
#endif

  /**
   * Returns texture information from the state.
   * getTextureImage() returns NULL if no texture is enabled.
   * [OIV-WRAPPER-RETURN-TYPE NAME{buffer},ARRAY{size.Value[0]*size.Value[1]*numComps}]
   * [OIVJAVA-WRAPPER PACK{TextureImage}]
   */
  const unsigned char *getTextureImage(SbVec2i32 &size, int &numComps) const;

  /**
   * Returns texture information from the state. 
   */
  const SbName &getTextureFileName() const;

  /**
   * Returns the current texture mapping information from the state.
   */
  const SbMatrix &getTextureMatrix() const;

  /**
   * Returns the current texture mapping information from the state.
   */
  const SoTexture2Transform *getTextureTransformNode() const;

  /**
   * Returns the current texture mapping information from the state.
   */
  SoTexture::Model getTextureModel() const;

  /**
   * Returns the current texture mapping information from the state.
   */
  SoTexture::Wrap getTextureWrapS() const;
  /**
   * Returns the current texture mapping information from the state.
   */
  SoTexture::Wrap getTextureWrapT() const;

  //////////////////////// Transformations:
  /**
   * Returns the current modeling transformation from the state.
   */
  const SbMatrix & getModelMatrix() const;

  /**
   * Returns the current units from the state.
   */
  SoUnits::Units getUnits() const;

  //////////////////////// Viewing:
  /**
   * Returns the current camera and viewing information from the state.
   */
  float getFocalDistance() const;

  /**
   * Returns the current camera and viewing information from the state.
   */
  const SbMatrix &getProjectionMatrix() const;

  /**
   * Returns the current camera and viewing information from the state.
   */
  const SbMatrix &getViewingMatrix() const;

  /**
   * Returns the current camera and viewing information from the state.
   */
  const SbViewVolume &getViewVolume() const;

  //////////////////////// Miscellaneous:
  /**
   * Returns the current picking style.
   */
  SoPickStyle::Style getPickStyle() const;

  /**
   * Returns the current switch value.
   */
  int32_t getSwitch() const;

  /**
   * Sets whether the callback uses normal traversal (switches, etc.)
   * or whether it traverses every single node. Default is FALSE.
   */
  void setCallbackAll(SbBool flag) { callbackAll = flag; }

  /**
   * Returns whether the callback uses normal traversal (switches, etc.)
   * or whether it traverses every single node. 
   */
  SbBool isCallbackAll() const  { return callbackAll; }

#if SoDEPRECATED_BEGIN(9400)

  /**
   * Returns texture information from the state.
   * getTextureImage() returns NULL if no texture is enabled.
   * [OIV-WRAPPER-RETURN-TYPE NAME{buffer},ARRAY{size.Value[0]*size.Value[1]*numComps}]
   */
  SoDEPRECATED_METHOD(9400,"Use a SbVec2i32 as the size parameter since a short can overflow.")
  const unsigned char *getTextureImage(SbVec2s &size, int &numComps) const;

#endif /** @DEPRECATED_END */

SoEXTENDER public:
#ifdef _WIN32
  void addPreCallback(SoType type, SoCallbackActionCB2 *cb, void *data);
  void addPostCallback(SoType type, SoCallbackActionCB2 *cb, void *data);
  void addPreTailCallback(SoCallbackActionCB2 *cb, void *data);
  void addPostTailCallback(SoCallbackActionCB2 *cb, void *data);
#endif

  // Returns the current response
  Response getCurrentResponse() const { return response; }

  // Call the pre-callbacks and post-callbacks
  virtual void invokePreCallbacks(const SoNode *node);
  virtual void invokePostCallbacks(const SoNode *node);

  // Call the primitive callbacks
  virtual void invokeTriangleCallbacks(const SoShape *shape,
                               const SoPrimitiveVertex *v1,
                               const SoPrimitiveVertex *v2,
                               const SoPrimitiveVertex *v3);
  virtual void invokeLineSegmentCallbacks(const SoShape *shape,
                                  const SoPrimitiveVertex *v1,
                                  const SoPrimitiveVertex *v2);
  virtual void invokePointCallbacks(const SoShape *shape,
                            const SoPrimitiveVertex *v);

  // Should primitives be generated for this shape?
  virtual SbBool shouldGeneratePrimitives(const SoShape *shape) const;    

 SoINTERNAL public:

  // This flag is used by the SoSwitch etc. nodes which must return a
  // different result from its 'affectsState' method when called
  // during a CallbackAction that is doing the callback on all children.
  static SbBool duringCallbackAll;
  static void initClass();
  static void exitClass();
  
  // Returns the node that caused the callback:
  virtual SoNode *getCurPathTail();
  
  // Set the current node during traversal:
  void setCurrentNode(SoNode *node)
    { currentNode = node;} 

  void setStartTransparencyType(  SoGLRenderAction::TransparencyType type) {
    transpType = type;
  };

  /**
  * Routine to remove callbacks for generated primitives (triangles)
  * for all shapes of the given type. The callback function will be
  * removed from the list of those called for each triangle
  * generated for all shapes of or derived from that type.
  * Returns true if removal succeeded
  */
  bool removeTriangleCallback(SoType type, SoTriangleCB *cb);

  /**
  * Routine to remove callbacks for generated primitives (lineSegments)
  * for all shapes of the given type. The callback function will be
  * removed from the list of those called for each lineSegment
  * generated for all shapes of or derived from that type.
  * Returns true if removal succeeded
  */
  bool removeLineSegmentCallback(SoType type, SoLineSegmentCB *cb);

  /**
  * Routine to remove callbacks for generated primitives (points)
  * for all shapes of the given type. The callback function will be
  * removed from the list of those called for each point
  * generated for all shapes of or derived from that type.
  * Returns true if removal succeeded
  */
  bool removePointCallback(SoType type, SoPointCB *cb);

  /**
   * Remove a callback function from those to call when a node of the given type is
   * encountered during traversal. The PreCallback is called just before the node is
   * traversed, and the PostCallback is called just after. The value returned by a
   * callback function indicates whether the action should continue with the
   * traversal.
   * Returns true if removal succeeded
   */
  bool removePostCallback(SoType type, SoCallbackActionCB *cb);

  /**
   * Remove a callback function from those to call when a node of the given type is
   * encountered during traversal. The PreCallback is called just before the node is
   * traversed, and the PostCallback is called just after. The value returned by a
   * callback function indicates whether the action should continue with the
   * traversal.
   * Returns true if removal succeeded
   */
  bool removePreCallback(SoType type, SoCallbackActionCB *cb);

  enum CBFuncTypes
  {
    NONE=0,
    PRE =1,
    POST=2,
    PRE_TAIL=4,
    POST_TAIL=8,
    ANY=PRE|POST|PRE_TAIL|POST_TAIL,
  };

  // get the container for the given type
  // for same type ambiguity is solved using type
  template< class T, int N >
  SbPList& getCallbackList( );

 protected:
  // Initiates action on graph
  virtual void beginTraversal(SoNode *node);

  // transforms vertices with the given matrix.
  // this is used to transform points during a forward traversal for callback actions
  void transformForwardTraversedVertices(const SbMatrix& model, size_t numVertices, SoPrimitiveVertex** vertices);

  // Response from last callback
  Response response;

 private:
  SbBool callbackAll;   // Callback all children or follow
  // normal traversal rules.
  // Callback lists
  SbPList preCallbackList;
  SbPList postCallbackList;

  SbPList preTailCallbackList;
  SbPList postTailCallbackList;

  SbPList triangleCallbackList;
  SbPList lineSegmentCallbackList;
  SbPList pointCallbackList;

  // Node where the callback occurs:
  SoNode *currentNode;

  SoGLRenderAction::TransparencyType transpType;     // Transparency quality type
};

#endif /* _SO_CALLBACK_ACTION_ */

