/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/

#ifndef  _SO_GL_RENDER_ACTION_
#define  _SO_GL_RENDER_ACTION_

#include <Inventor/SbViewportRegion.h>
#include <Inventor/nodes/SoViewport.h>
#include <Inventor/actions/SoSubAction.h>
#include <Inventor/elements/SoShapeStyleElement.h>
#include <Inventor/elements/SoDecimationTypeElement.h>
#include <Inventor/lists/SoPathList.h>
#include <Inventor/lists/SoSensorList.h>

#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

class SbBox3f;
class SoGetBoundingBoxAction;
class SoGLDepthPeeling;
class SoCache;
class SoCamera;
class SoSeparator;
class SoGLRenderActionSync;
class SoGLRenderActionImpl;


/**
 * Callback functions used between rendering passes should be of this type.
 *
 * @memberof SoGLRenderAction
 *
 * [OIV-WRAPPER NAME{PassCB}]
 */
typedef void    SoGLRenderPassCB(void *userData);

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoGLRenderAction
//
//  GL rendering action.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Renders a scene graph using Open Inventor's Render Engine.
 *
 * @ingroup actions
 *
 * @DESCRIPTION
 *   This class renders a scene graph using Open Inventor's render engine.
 *   This action traverses a scene graph and queues commands that allow the render
 *   engine to render the scene asynchronously.
 *   Note that (despite the "historical" name) this class is not dependent on any
 *   graphics library and does not directly provide rendering to a window.
 *
 *   The viewer classes, SoWin/SoXt/SoQtExaminerViewer etc, automatically create
 *   a render action for you and also expose most of the rendering options for
 *   convenience.  See also the viewer component classes (SceneExaminer etc).
 *
 *   @B Traversal order @b @BR
 *   Nodes from the scene graph are traversed using depth-first traversal. However, traversal of
 *   some objects may be delayed, depending on transparency options.
 *
 *   A transparent object is an object:
 *   - With a material that has a transparency value set
 *     either by a material node or a vertex property node.@BR
 *     or @BR
 *   - With a texture that has an alpha value < 1.@BR
 *
 *   Open Inventor automatically detects these cases.
 *
 *   The rendering order for transparent
 *   objects depends on the transparency "type" (transparency
 *   rendering method) assigned to each object. A transparency
 *   type may be set for the entire scene graph using the
 *   SoGLRenderAction or SoWinRenderArea method.
 *
 * There are several special cases including:
 *
 *   - Transparency type SORTED_PIXEL.@BR
 *     The environment variable OIV_NO_SHADER_LAYERS_BLEND allows you to disable the use
 *     of a fragment shader for this technique.
 *
 *   - SoRenderList nodes, e.g. SoOctreeOrdering.@BR
 *     Objects that are delayed are generally rendered in the sequence below,
 *     but other objects are rendered in various specific orders, not in scene
 *     graph order. See the specific node for details.
 *
 * The general rendering order is:

 * @OL
 *  @LI Scene graph traversal @BR
 *     All objects not in one of the following categories, specifically
 *     including transparent
 *     objects using the NO_SORT transparency type.
 *
 *  @LI FastEdit KEEP_ZBUFFER objects @BR
 *     Objects under an SoSeparator node with fastEditing field set to
 *     KEEP_ZBUFFER.
 *
 *  @LI Opaque delayed objects @BR
 *
 *  @LI Transparent delayed objects @BR
 *     Objects using transparency types OPAQUE_FIRST or SORTED_PIXEL. Note that
 *     by default the OpenGL depth buffer is not updated while rendering these
 *     objects.
 *
 *  @LI Transparent sorted objects @BR
 *     Objects using transparency types SORTED_OBJECT
 *     These objects are sorted by the distance of their bounding box from the
 *     camera, then rendered in back-to-front order. Note that
 *     by default the OpenGL depth buffer is not updated while rendering these
 *     objects.
 *
 *  @LI Annotation delayed objects @BR
 *     Objects under an SoAnnotation node.
 *
 *  @LI FastEdit CLEAR_ZBUFFER objects @BR
 *     Objects under an SoSeparator node with fastEditing field set to
 *     CLEAR_ZBUFFER.
 *
 * @ol
 *
 * @B Elements @b @BR
 * Sets: SoDecimationTypeElement, SoDecimationPercentageElement,
 *       SoUpdateAreaElement, SoRenderPassElement,
 *       SoViewportRegionElement, SoLogicalViewportElement
 *
 * Notes & Limitations:
 * - Since OpenInventor 10.0, most rendering and caching mechanisms are managed
 *   by SoSceneManager, which means that a SoGLRenderAction cannot be used
 *   outside of the render() call of SoSceneManager. Hence, manually calling
 *   apply() on an SoGLRenderAction can only be done during the traversal of
 *   the scene graph, for example inside an SoCallback. Calling apply() outside
 *   of this case leads to undefined behavior and will potentially result in
 *   incorrect rendering, performance issues, memory leaks and crashes.
 *
 * @SEE_ALSO
 *    SoSeparator,
 *    SoWinRenderArea,
 *    SoRenderAreaCore
 *
 */
class INVENTOR_API SoGLRenderAction : public SoAction {

  SO_ACTION_HEADER(SoGLRenderAction);

 public:

  /** 
   * Transparency rendering algorithm.
   * See #setTransparencyType() method.
   *
   * Note:
   * The transparency values deprecated since OpenInventor 9 have been removed.
   * You can find below a correspondence table:
   * <table style="border-collapse: collapse;">
   *   <tr>
   *     <th>OIV 10</th>
   *     <th>OIV 9</th>
   *   </tr>
   *   <tr>
   *     <td>NO_SORT</td>
   *     <td>NO_SORT<br>NO_TRANSPARENCY<br>SCREEN_DOOR<br>ADD<br>BLEND</td>
   *   </tr>
   *   <tr>
   *     <td>OPAQUE_FIRST</td>
   *     <td>OPAQUE_FIRST<br>DELAYED_ADD<br>DELAYED_BLEND</td>
   *   </tr>
   *   <tr>
   *     <td>SORTED_OBJECT</td>
   *     <td>SORTED_OBJECT<br>SORTED_OBJECT_ADD<br>SORTED_OBJECT_BLEND</td>
   *   </tr>
   *   <tr>
   *     <td>SORTED_PIXEL</td>
   *     <td>SORTED_PIXEL<br>SORTED_LAYERS_BLEND<br>SORTED_PIXELS_BLEND<br>DELAYED_SORTED_LAYERS_BLEND<br>
   *       DELAYED_SORTED_PIXELS_BLEND<br>SORTED_TRIANGLES_ADD<br>SORTED_TRIANGLES_BLEND<br>
   *       SORTED_OBJECT_TRIANGLES_ADD<br>SORTED_OBJECT_TRIANGLES_BLEND
   *     </td>
   *   </tr>
   * </table>
   */
  enum TransparencyType
  {

    SoDEPRECATED_ENUM_VALUE(10000, "No longer supported. Will automatically switch to NO_SORT.")
    NO_TRANSPARENCY,

    /**
     * The simplest transparency mode. Shapes are rendered in the same order as they are traversed. @BR
     * Alpha blending is used to combine the shape color with the color in the framebuffer. @BR
     * Depth buffer test and write are activated during rendering.
     *
     * Limitation: A shape will not be displayed if it is behind a semi-transparent shape
     * that has been rendered before it.
     */
    NO_SORT,

    /**
    * Same as NO_SORT, but the rendering of opaque objects is performed before
    * the rendering of semi-transparent objects. @BR
    *
    * Notes & Limitations:
    * - When using this algorithm, the depth buffer is not updated
    *   (depth buffer writes are disabled) during the rendering of semi-transparent
    *   objects. As a result, complex semi-transparent shapes may not be rendered correctly.
    * - Because opaque objects are forced to be rendered first, the objects rendering order
    *   does not necessarily correspond to their actual ordering in the scene graph.
    * - Because semi-transparent objects are not sorted and alpha blending is not a commutative operation,
    *   the blending of several semi-transparent objects will not be correct if their order in the scene graph
    *   does not match their back-to-front ordering relative to the camera.
    */
    OPAQUE_FIRST,

    /**
    * Same as OPAQUE_FIRST, but sorts semi-transparent objects by distances of bounding
    * boxes from camera.
    * Limitation : If the bounding box of a shape is ignored by setting its boundingBoxIgnoring field to true, the rendering won't be correct.
    */
    SORTED_OBJECT,

    /**
    * Same as OPAQUE_FIRST, but uses a fragment-level depth sorting technique during the rendering of semi-transparent objects. @BR
    * This mode generally gives the best results for complex semi-transparent objects.
    *
    * Since Open Inventor 9.4, if the hardware supports the necessary OpenGL features, this
    * transparency mode is implemented using a single-pass, order-independent fragment sorting
    * (K-buffer or A-buffer) algorithm. The required OpenGL hardware
    * features are: @B shader_image_load_store@b, @B shader_atomic_counters@b and
    * @B shading_language_420pack@b. These features are standard in core OpenGL 4.2. Use
    * SoGLExtension, if necessary, to check if they are supported.
    *
    * If the hardware does not support single-pass transparency, then a multi-pass "depth
    * peeling" algorithm is used. The required OpenGL hardware features are: @B Multi-Texture@b,
    * @B Texture Environment Combine@b, @B Depth texture@b, and @B Shadow @b. These features
    * are standard starting with OpenGL 1.4. The method setSortedLayersNumPasses() allows you
    * to set the number of rendering passes for more correct transparency. Usually, four passes
    * gives good results.
    *
    * If the graphics board does not support multi-pass transparency, SORTED_OBJECT is used.
    *
    * Note: If the application uses fragment sorting transparency and specifies a custom fragment
    *       shader, the application shader must call an Open Inventor GLSL method instead of
    *       setting gl_FragColor directly. See @ref TransparencyShaders for details.
    *
    * Limitations:
    *  - Since it uses a "depth peeling" algorithm, this transparency type does not work
    *    correctly if a node @ref SoDepthBuffer is used to change depth buffer settings.
    *  - This transparency type is not compatible with interlaced stereo.
    *  - Texturing on semi-transparent objects is limited to one texture.
    *  - Using the single-pass algorithm, it is possible to run out of GPU memory for
    *    storing fragment transparency values. This should only happen for scenes with
    *    both high depth complexity and a large number of semi-transparent objects. In
    *    this case fragments from the objects traversed last in the scene graph may not be
    *    sorted correctly, resulting in small artifacts.
    *  - Not compatible with FSAA. use SUPERSAMPLING, FXAA or SMAA instead.
    */
    SORTED_PIXEL,

  };

  /**
   * Fast edit traversal types. See SoSeparator for additional info on
   * fast editing. @BR
   * In the following description, we will call @B FE the sub-scene graph
   * allowing the fast editing@b feature,@BR
   * and @B NFE the non-fast editing sub-scene graph@b and consider the following scene graph:
   * \verbatim
     Separator {
       Separator {
         fastEditing # ( != DISABLE to activate )
         Translation {}
         Cube{}
       }
       Separator {
         Translation {} Sphere{}
         Translation {} Sphere{}
         Translation {} Sphere{}
       }
     }
     \endverbatim
   */
  enum FastEditSavePolicy {
    /**
     * Fast editing is disabled.
     * @BR
     * @IMAGE FE_Disable.jpg
     * @BR
     */
    DISABLE,

    /**
     * The fast edit feature is available. The image buffer
     * (and possibly the depth buffer, depending on the value
     * of the SoSeparator::fastEditing field) is saved each time
     * the scene graph is redrawn.
     *
     * We recommend using the EACH_FRAME flag when manipulating a very large
     * main scene graph. In this case, the time used for saving the buffer(s)
     * is insignificant compared to the time to draw the scene.
     * EACH_FRAME is recommended as well when the fast editing sub-scene graph
     * is frequently modified: the user interactivity is better with the fast edit
     * sub-scene graph even though the global scene frame rate may slow down.
     * @BR
     * @IMAGE FE_Each_Frame.jpg
     * @BR
     */
    EACH_FRAME,

    /**
     * The fast edit feature is available. No buffer save is made during ordinary scene rendering.
     * When a change to the fast editing sub-scene graph occurs, the entire scene is first
     * rendered and saved, and then the fast edit nodes are drawn. During the next rendering,
     * only the fast edit nodes are drawn. So, using this flag implies you need one more
     * full scene rendering before starting to move your fast edit sub-scene graph interactively.
     * It would be better to use WHEN_NEEDED when the fast editing sub-scene graph changes
     * very rarely. In this case you will be able to render the scene at full speed
     * because extra time will not be spent saving the image buffer.
     * @BR
     * @IMAGE FE_When_Needed.jpg
     * @BR
     */
    WHEN_NEEDED
  };

  /** Possible return codes from a render abort callback */
  enum AbortCode {
    /**
     * Continue traversal as usual
     */
    CONTINUE,

    /**
     *  Stop traversing the rest of the graph
     */
    ABORT,

    /**
     *  Do not traverse this node or its children, but continue
     */
    PRUNE,

    /**
     *  Delay rendering of this node until the second pass
     */
    DELAY
  };

  /** Callback functions for render abort should be of this type.
   * This typedef is defined within the class, since it needs to
   * refer to the AbortCode enumerated type.
   * [OIV-WRAPPER NAME{AbortCB}]
   */
  typedef AbortCode SoGLRenderAbortCB(void *userData);

  /** Invalidate cache mode */
    enum InvalidateCacheMode {
    /**
     * (Default) Render caches are handled normally. See #setInvalidateCacheMode().
     */
    OFF,

    /**
     *  Render caches are invalidated for each SoSeparator node before
     *  its children are traversed. The invalidate
     *  cache mode is automatically set to OFF after traversal. See #setInvalidateCacheMode().
     */
    ONCE,

    /**
     * Render caches are invalidated for each SoSeparator node before
     * its children are traversed. See #setInvalidateCacheMode().
     */
    ALWAYS
  };

  /**
   * Constructor. The parameter defines the viewport region into which rendering will
   * take place.
   */
  SoGLRenderAction(const SbViewportRegion &viewportRegion);

#ifndef HIDDEN_FROM_DOC
  // Destructor
  virtual ~SoGLRenderAction();
#endif // HIDDEN_FROM_DOC

  /**
   * Changes viewport region to use for rendering.
   */
  void setViewportRegion(const SbViewportRegion &newRegion);

  /**
   * Returns viewport region to use for rendering.
   */
  const SbViewportRegion &getViewportRegion() const   { return vpRegion; }

  /**
   * Sets current logical viewport region to use for rendering.
   * Setting an empty viewport region (0,0) unsets the logical viewport.
   */
  void setLogicalViewportRegion(const SbViewportRegion &newLogicalRegion);

  /**
   * Returns current logical viewport region
   */
  const SbViewportRegion& getLogicalViewportRegion() const;

  /**
   * Sets the current update area, which is the rectangular area of the
   * viewport region that will actually be rendered into. This can be used for
   * partial updates in applications that can manage them. The update area is
   * specified in normalized viewport coordinates, where (0,0) is the lower left
   * corner of the viewport and (1,1) is the upper right corner. The area is
   * specified or returned as an origin and a size.
   */
  void setUpdateArea(const SbVec2f &origin, const SbVec2f &size);

  /**
   * Returns the current update area. See setUpdateArea() for details.
   */
  void                getUpdateArea(SbVec2f &origin, SbVec2f &size) const;

  /**
   * Sets callback to call during rendering to test for an abort condition. It will
   * be called for each node that is traversed. This allows applications to terminate
   * rendering prematurely if some condition occurs. The callback function should
   * return one of the AbortCode codes to indicate whether traversal should continue.
   * Use of the various codes in a callback can allow applications to modify Open
   * Inventor's default order of rendering objects in a scene graph.
   */
  void setAbortCallback(SoGLRenderAbortCB *func, void *userData)
    { abortCB = func; abortData = userData; }

  /**
   * Sets global transparency algorithm to use when rendering. 
   * @useenum{TransparencyType}. The default is NO_SORT, but SORTED_PIXEL
   * is recommended for most cases. See also #getTransparencyType().
   * Transparency handling can be completely disabled by setting NO_SORT.
   *
   * The global transparency algorithm can also be set using a method in the viewer
   * class (e.g. SoWinRenderArea::setTransparencyType()).
   *
   * The default value for this field can be set using the environment variable OIV_TRANSPARENCY_TYPE.
   *
   * See SoGLRenderAction::TransparencyType for more discussion about transparency.
   * 
   * @B Notes: @b
   *
   * - When using transparency types with DELAYED in the name, the depth buffer is
   *   not updated (depth buffer writes are disabled) while rendering transparent objects.
   *   As a result complex 3D shapes may not be rendered correctly.
   *
   * - When using a delayed transparency mode, paths to transparent objects are saved in a list.
   *   After all opaque objects have been rendered, this list is rendered. In order to avoid
   *   unnecessary rendering, the common part of each path is rendered only once. This can increase performance on
   *   very complex scene graphs. However, in some cases, it
   *   can lead to incorrect rendering. In the figure below, node B is a transparent object and node A has a 
   *   side effect on node B (clipping, shader...).
   *   When the compact path optimization is on (the default), node A will not be traversed during the transparency pass,
   *   which will lead
   *   to incorrect rendering. When a case like this occurs in your scene graph, set the environment
   *   variable OIV_USE_COMPACT_PATHS to false to disable this optimization.
   * @BR
   * @IMAGE CompactPathList.png
   * @BR
   */
  void                setTransparencyType(TransparencyType type);

  /**
   * Returns transparency quality level to use when rendering.
   * See #setTransparencyType().
   */
  TransparencyType    getTransparencyType() const
  { return m_transpType; }

  /**
   * Sets fast editing save policy to use when rendering. The default is WHEN_NEEDED.
   * Valid values are DISABLE, EACH_FRAME, and WHEN_NEEDED. See SoSeparator.
   *
   * If fastEditDelayedObjects is set to TRUE, delayed objects won't be redrawn
   * when editing the scene graph. It means that the composition between delayed
   * transparent objects and the fast edit scene graph won't be correct, but
   * rendering will be much faster if there are a lot of transparent objects.
   */
  void setFastEditSavePolicy(FastEditSavePolicy policy, SbBool fastEditDelayedObjects = FALSE);

  /**
   * Returns fast editing save policy used when rendering.
   */
  FastEditSavePolicy  getFastEditSavePolicy() const
  { return m_fastEditPolicy; }

  /**
   *
   * Sets smoothing flag. When on, smoothing uses OpenGL's line- and
   * point-smoothing features to provide cheap antialiasing of lines and points.
   * The default is FALSE.
   *
   * The default value can be set using the environment variable
   * OIV_GL_SMOOTHING (0 = FALSE, 1 = TRUE).
   */
  SoDEPRECATED_METHOD(101000, "No longer used.")
  void                setSmoothing(SbBool smooth);

  /**
   * Returns smoothing flag.
   */
  SoDEPRECATED_METHOD(101000, "No longer used.")
  SbBool              isSmoothing() const             { return doSmooth; }

  /**
   * Sets number of rendering passes for multipass antialiasing.
   * The default is 1. Specifying more
   * than one pass will result in antialiasing of the rendered scene, using OpenGL's
   * accumulation buffer. (Camera nodes typically move their viewpoints a little bit
   * for each pass to achieve the antialiasing.) Each additional pass provides better
   * antialiasing, but requires more rendering time.
   *
   * The default value can be set using the environment variable
   * OIV_NUM_RENDER_PASSES.
   *
   * NOTE: Multipass antialiasing can have very poor performance for large scenes.
   * Full-scene antialiasing (multi-sample) is strongly recommended when the hardware
   * supports it. See SoSceneManager and SoFullSceneAntialiasing.
   */
  void setNumPasses(int num);

  /**
   * Gets number of rendering passes for multipass rendering.
   */
  int getNumPasses() const  { return numPasses; }

  /**
   * Sets a flag indicating whether intermediate results are displayed after
   * each antialiasing pass for progressive improvement (default is FALSE).
   */
  void                setPassUpdate(SbBool flag)      { passUpdate = flag; }

  /**
   * Returns a flag indicating whether intermediate results are displayed after
   * each antialiasing pass for progressive improvement.
   *
   * The default value can be set using the environment variable
   * OIV_ANTIALIAS_PASS_UPDATE.
   */
  SbBool              isPassUpdate() const            { return passUpdate; }

  /**
   * Sets a callback function to invoke between passes when antialiasing. Passing
   * NULL (which is the default state) will cause a clear of the color and depth
   * buffers to be performed.
   */
  void                setPassCallback(SoGLRenderPassCB *func, void *userData)
    { passCB = func; passData = userData; }

  /**
   * Sets the OpenGL cache context. A cache context is just an integer
   * identifying when OpenGL display lists (which are used for render caching) can be
   * shared between render actions; for example, see the documentation on GLX
   * contexts for information on when OpenGL display lists can be shared between GLX
   * windows.  See generateCacheContext().
   * Note: The value 0xFFFFFFFF (~0) is reserved to represent an undefined id.
   */
  void setCacheContext(uint32_t context);

  /**
   * Returns the OpenGL cache context id for this action.
   */
  uint32_t getCacheContext() const;

  /**
   * Generates and returns a new cache context id. The returned cache context id can then be
   * used to pass to the setCacheContext() method call.
   */
  static uint32_t generateCacheContext();

  /**
   * Returns TRUE if the specified cache context id is currently used by any SoGLRenderAction.
   */
  static bool isUsedCacheContext( uint32_t context );

  /**
   * Sets the number of rendering passes used when transparency type is #SORTED_PIXEL.
   * Default is 4.
   *
   * Note: Since Open Inventor 9.4, if the hardware supports the necessary OpenGL features, these
   * transparency modes are implemented using a single-pass, order-independent fragment sorting
   * (A-buffer) algorithm.  Therefore the value of this field is not used except in two special
   * cases:
   * - If hardware support for single pass transparency is not available, then the previous
   *   "depth peeling" algorithm is used (which normally requires multiple passes), or
   *
   * - If volume rendering (see SoVolumeRender) is combined with geometry rendering, then multiple
   *   layers of transparency information are required and this field specifies the number of layers.
   *
   * Use more passes for more correct transparency. Usually four passes (which is the default value)
   * gives good results. For example,
   * the number of passes needed for rendering an opaque sphere behind two cubes is the number
   * of planes separating the sphere from the viewer.
   * So, when backface culling is enabled, 2 faces
   * separate the sphere from the viewer: the front face of the first cube and the front face of
   * the second cube. The number of passes needed in this case is 2.
   *
   * The default value for this field can be set using the environment variable
   * OIV_NUM_SORTED_LAYERS_PASSES.
   */
  void setSortedLayersNumPasses(int num)
    { m_sortedLayersNumPasses = num; }


  /**
   *
   * Returns the number of rendering passes used
   * when transparency type is #SORTED_PIXEL.
   */
  int getSortedLayersNumPasses() const
    { return m_sortedLayersNumPasses ; }

  /**
   * Indicates if #SORTED_PIXEL transparency is supported by
   * your graphics board on the current context.
   *
   */
  static SbBool isSortedLayersSupported(SoState* state=NULL) ;

  /**
   * Enable or disable the OpenGL alpha test for rendering.
   * Default is FALSE (alpha test disabled) and the default test is
   * "not equal to zero".  The alpha test operates on all pixel writes,
   * including those resulting from the scan conversion of points, lines,
   * polygons, and bitmaps, and from pixel draw and copy operations.
   * It does not affect screen clear operations.
   *
   * For example, if rendering
   * with textures containing "binary" alpha values (0 or 1), the alpha test may
   * be faster than blended transparency and does not require the use of
   * delayed rendering transparency types.
   *
   * The default value can be set using the environment variable
   * OIV_ALPHA_TEST (0 = FALSE, 1 = TRUE).
   */
  void setAlphaTest( SbBool flag );

  /**
   * Returns whether the OpenGL alpha test is enabled for rendering.
   */
  SbBool isAlphaTest() const
    { return m_alphaTest; }


  /**
   * Sets whether or not "remote" rendering is done.
   * Inventor's auto-render-caching algorithm will choose to cache
   * more often when rendering is remote (the assumption being that
   * performance will be better with display lists stored on the
   * remote machine). By default, it is assumed rendering is NOT
   * remote.
   */
  void                  setRenderingIsRemote(SbBool flag);
  /**
   * Returns TRUE if remote rendering is being done.
   */

  SbBool              getRenderingIsRemote() const;

  /**
   * Invalidate the state, forcing it to be recreated at the next apply
   */
  virtual void        invalidateState();

  /** Sets decimation percentage, the value should be a
   * number between 0.0 and 1.0
   * Useful for reducing the number of triangles rendered
   * in the scene.
   */
  void setDecimationValue(SoDecimationTypeElement::Type type,
                          float percentage = 1.0);

  /**
   * Gets decimation type.
   */
  SoDecimationTypeElement::Type getDecimationType() { return decType; };

  /**
   * Gets decimation percentage, the value should be a
   * number between 0.0 and 1.0.
   */
  float               getDecimationPercentage() { return decPercent; };

  /**
   * Enables or disables the invalidation of render caches.
   * - When set to ALWAYS, the caches are invalidated for each SoSeparator node before
   *   its children are traversed. No new caches will be built. This value forces
   *   all nodes to be visited during each render traversal.
   *
   * - When set to ONCE, the caches are invalidated for each SoSeparator node before
   *   its children are traversed. The
   *   invalidate cache mode is automatically changed to OFF at the end of the traversal.
   *
   * - When set to OFF (default), caches are managed by their respective render caching
   *   nodes in the usual way.
   *
   * This method is useful to force all nodes to be visited during render traversal.
   */
  void setInvalidateCacheMode(InvalidateCacheMode icm);

  /**
   * Returns the current cache invalidation mode.
   */
  InvalidateCacheMode getInvalidateCacheMode();

  /**
   * Returns the current frame counter.
   * The frame counter starts at zero and is automatically incremented
   * each time the action is applied.  The frame counter will have the
   * same value throughout a single apply, even if there are multiple
   * render passes, delayed rendering, etc.
   */
  uint32_t getFrameCounter() const { return m_frameCounter; }

  /**
   * Reset the frame counter.
   * Do not call this method during a traversal.
   */
  void resetFrameCounter( uint32_t count = 0 ) { m_frameCounter = count; }

  /** @copydoc SoAction::apply(SoNode*) */
  virtual void apply(SoNode* node);

  /** @copydoc SoAction::apply(SoPath*) */
  virtual void apply(SoPath* path);

  /** @copydoc SoAction::apply(const SoPathList&,SbBool) */
  virtual void apply(const SoPathList& pathList, SbBool obeysRules = FALSE);

  /**
   * Two bits are stored.  Nodes that should be cached will set the
   * DO_AUTO_CACHE bit, nodes that should NOT be cached will set the
   * DONT_AUTO_CACHE bit.  By default, DO_AUTO_CACHE is FALSE unless
   * remote rendering is being done.  DONT_AUTO_CACHE is FALSE by
   * default.  Separators will auto cache if DO_AUTO_CACHE is TRUE
   * and DONT_AUTO_CACHE is FALSE, otherwise they won't auto-cache.
   */
  enum AutoCacheMode
  {
    DO_AUTO_CACHE = 1,
    DONT_AUTO_CACHE = 2
  };

  /**
   * Called by nodes to say that they should/shouldn't be
   * auto-cached (pass TRUE if should, FALSE if shouldn't, don't
   * call this method at all if the node doesn't care):
   */
  void shouldAutoCache( AutoCacheMode mode );

SoEXTENDER public:
  /** Return true if depth peeling is active*/
  inline bool isDoingDepthPeeling() const { return m_isDoingDepthPeeling; }

  /**
   * Get the current pass of depth peeling, -1 if not doing depth peeling
   */
  int getDepthPeelingPass();

  /**
   * Use 3D texturing instead of 2D for the fixed functionality emulation
   * in depth peeling
   */
  void enableDepthPeeling3D(bool flag) const;

  SoGLDepthPeeling* getDepthPeelingInterface() const;

  // Returns current rendering pass number
  int getCurPass() const              { return curPass; }

  // Returns TRUE if render action should abort - checks user callback
  SbBool abortNow()
    { return (hasTerminated() || (abortCB != NULL && checkAbort())); }

  // Returns TRUE if render action handles processing of a
  // transparent object (if it is to be sorted and rendered later).
  // If this returns FALSE, the object should just go ahead and
  // render itself.
  // The optional argument isTransparent ensures that the object being
  // rendered will be taken as transparent, regardless of transparency
  // in the state.  If it is false, the state is checked for transparency.
  virtual SbBool handleTransparency(SbBool isTransparent = FALSE, SbBool isLinesOrPoints = FALSE);

  /** initialize the action state. */
  virtual void setUpState();

  /**
   * Control whether delayed transparent objects update the depth buffer.
   * By default they do not (historical behavior)
   */
  void setDelayedObjDepthWrite( SbBool flag ) { m_delayedObjDepthWrite = flag; }

  /**
   * Returns whether delayed transparent objects update the depth buffer.
   * By default they do not (historical behavior)
   */
  SbBool getDelayedObjDepthWrite() const { return m_delayedObjDepthWrite; }

 /**
   * Method called by SoMultiPassManager before delayed pass traversals.
   * Overriding this method can be done action that may want to discard the traversal
   * of all delayed passes. Returning false aborts the delayed passes traversal.
   * @M_SINCE 9.1
   */
  virtual bool preDelayedTraversal();

  /**
   * Method called by SoMultiPassManager after delayed pass traversals.
   * Overriding this method can be done action that may want to apply some post
   * processing after all delayed passes traversal.
   * @M_SINCE 9.1
   */
  virtual void postDelayedTraversal();

  /**
   * Sets the main cache to use with this render action.
   * When calling apply() on this action, the SoCacheElement will be
   * set with this cache.
   */
  void setCache( SoCache* cache );

  /** Returns the cache that has been set with setCache() */
  SoCache* getCache() const;

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  // Returns TRUE if rendering is being delayed because of transparency sorting or delaying
  SbBool isDelaying() const { return delayObjs; }

  // Returns true if the current state contains any transparency
  bool stateContainsTransparency() const;

  // Used in SoLocateHighlight in order to clear the transparent path cache
  void removeAllDelayedPath();

  bool isTraversalPassTransparent() const;

  bool handleFastEdit(SoSeparator* sep);

  int getCullTestResults() { return cullBits; }
  void setCullTestResults(int b) { cullBits = b; }

  SbBool isLogicalRegionSet() const;

  void setTiledViewportRegion(const SbVec4f& region, int info = 0);

  /**
   * @brief Set the maximum depth offset of the render action
   * @param maxDepthOffset the maximum depth offset
   */
  void setMaxDepthOffset(float maxDepthOffset);

/* FAST EDIT METHODS*/
  // set the buffer type.
  void setDoubleBuffer(SbBool db);
  // The main scene must be redrawn.
  void setForceRender();
  // Set viewing mode (from viewer).
  void setViewing(SbBool viewing);
  // Is there any delayed object to render ?
  SbBool hasDelayedNodesToRender() const;
  // initialize the start of a stereo pass.
  void setFirstStereoPass(SbBool first);

  SoPathList transpPaths; // Keep for compatibility. should be removed in OIV 9.2

  void setRenderingFromOffscreen(SbBool renderFromOffscreen)
  { m_renderingFromOffscreen = renderFromOffscreen; }

  SbBool getRenderingFromOffscreen() const
  { return m_renderingFromOffscreen; }

  //Return true after an apply if there were transparent objects
  //(used in SoLocateHighlight in order to redraw the whole scene only in this case)
  SbBool hasDrawnTransparentObjs() const
  { return m_hasDrawnTransparentObjs; }

  // Set the fullscene antialiasing element
  void setFullSceneAntialiasingElement( SbBool enable, int filterMask );

  // Set the number of samples for MSAA
  void setMSNumSamples(int numSamples);

  /** Render mode */
  enum RenderMode {
    /**
     * Eveything is rendered as usual (default).
     */
    ALL_OBJECTS,
    /**
     * Delayed objects are not rendered.
     */
    OPAQUE_OBJECTS,
    /**
     * Only delayed objects are rendered.
     */
    DELAYED_OBJECTS
  };

  //
  // Allow to restrict which object will be render during the next ::render call.
  // After the ::render call, this mode is reset to its default value.
  // Note :  this is used by ScaleViz Depth compositing to manage delayed object.
  void setRestrictObjectRenderingMode( const RenderMode mode = ALL_OBJECTS )
  { m_restrictObjectRendering = mode; };

  int getRestrictObjectRenderingMode() const
  { return m_restrictObjectRendering; };

  SoGLRenderActionSync* getSynchronizer() const;
  void setSynchronizer( SoGLRenderActionSync* );

  virtual void updateParametersFrom( SoGLRenderAction* );
  SoGLRenderPassCB* getPassCallback( void*& data )
  {
    data = passData;
    return passCB;
  }

  SoGLRenderActionImpl* getImpl();

  // Used to temporary change the depth peeling interface, used
  // for depth only. Returns the value in use before the call.
  SoGLDepthPeeling* setDepthPeelingInterface(SoGLDepthPeeling*);

  bool isFastEditManipulating();

  SoPathList delayedPaths; // Keep for compatibility. should be removed in OIV 9.2

  /** indicate if action is currently called to evaluate lazy eval or for classical traversal.
   *  GLRenderAction may be applied in 2 cases:
   *  - During classical traversal
   *  - During lazyEvaluation pass. In that case, traversed node may not need to travers all its code. It just
   *  have to set its elements on state. */
  bool isLazyEvaluating() const { return m_lazyEvaluating; }
  void setLazyEvaluating(bool value) { m_lazyEvaluating = value; }

  void copyParametersFrom( SoGLRenderAction* action );

  SoViewport* getViewportNode() const
  {
    return m_viewport.ptr();
  }

  /**
   * Indicates that a floating-point color buffer should be used for rendering.
   * Can be called by nodes during traversal.
   */
  void setNeedFloatColorBuffer();

  // These flags determine which things have to be sent to GL when
  // the action is applied. They indicate what's changed since the
  // last time the action was applied.
  enum flags {
    TRANSPARENCY_TYPE = 0x01,
    SMOOTHING         = 0x02,
    ALPHA_TEST        = 0x04,
    FASTEDIT_SAVE_POLICY = 0x08,
    SCENE_GRAPH       = 0x10,
    ANTIALIASING      = 0x20,
    ALL               = 0x3F  // Initial value
  };

protected:
  // Initiates action on graph
  virtual void beginTraversal(SoNode *node);

SoINTERNAL protected:
  // used internally to temporary change the fastEditPolicy without reseting any caches
  void setFastEditSavePolicyInternal(FastEditSavePolicy policy)
  { m_fastEditPolicy = policy; }

  // used internally to temporary change the transparencyType without reseting any caches
  void setTransparencyTypeInternal(TransparencyType type)
  { m_transpType = type; }

  uint32_t getWhatChanged() const
  {
    return whatChanged;
  }

private:

  /** Contains common code part between apply(SoNode), apply(SoPath) and apply(SoPathList). */
  template<typename ApplyType> void commonApply(ApplyType applyObject, SbBool obeysRules);

  /** specialized version for applying action on objects by the commonApply */
  template<typename ApplyType> void applyTo(ApplyType applyObject, SbBool obeysRules);

  SoGLRenderActionSync* m_synchronizer;

  FastEditSavePolicy  m_fastEditPolicy;     // Transparency quality type
  TransparencyType    m_transpType;     // Transparency quality type

  /** Remove paths selected during the updateTranspList
   (paths which don't exist anymore because of a switch change, a group removeChild...)
  */
  void removeInvalidPaths();

  /** True if doing depth peeling */
  bool m_isDoingDepthPeeling;

  SbBool m_hasDrawnTransparentObjs;
  SbBool m_isViewing;
  SoNode* m_appliedNode;
  SbBool m_isFirstApply;
  SbBool m_firstStereoPass;
  int m_lastStereoMode;

  int    m_useCacheOpt;
  int    m_useCompactPaths;

  SbViewportRegion    vpRegion;       // Current viewport region

  // super viewport node that manage viewport, modify viewport and logical viewport
  SoRef<SoViewport> m_viewport;

  /// Node that manage maximum depth offset of the whole traversal.
  /// Useful so it can create dependencies when necessary (c.f. camera which needs
  /// to be updated according to maximum depth offset modifications).
  SoRef<SoNode> m_maxDepthOffset;

  SbVec2f             updateOrigin;   // Origin of update area
  SbVec2f             updateSize;     // Size of update area


  // Variables for render abort:
  SoGLRenderAbortCB   *abortCB;       // Callback to test abort
  void                *abortData;     // User data for abort callback

  // Variables for transparency, smoothing, and multi-pass rendering:
  SbBool              doSmooth;       // Doing smoothing ?
  int                 numPasses;      // Number of rendering passes
  int                 curPass;        // Current pass
  SbBool              passUpdate;     // Whether to update after each pass
  SoGLRenderPassCB    *passCB;        // Callback between passes
  void                *passData;      // User data for pass callback

  // For SORTED_OBJECT transparency:
  SbBool              delayObjs;           // TRUE if transp objects are to be delayed until after opaque ones

  SbBool              remoteRendering;// Remote rendering?

  // BA 2/98 - added decimation variables
  SoDecimationTypeElement::Type decType;    // Decimation type
  float                         decPercent; // Decimation percentage

  uint32_t whatChanged;

  // Enables or disables GL blending
  void enableBlending(SbBool enable);

  // Enables or disables GL smoothing
  void enableSmoothing(SbBool enable);

  // Returns TRUE if render action should abort based on callback
  SbBool checkAbort();

  // Keep track of which planes we need to view-volume cull test
  // against:
  int cullBits;

  SbBool m_delayedObjDepthWrite;

  // Indicates if the next render is asked for Offscreen
  SbBool m_renderingFromOffscreen;

  SbBool m_alphaTest;

  // Count frames rendered with this action (for debugging)
  uint32_t  m_frameCounter;

  // indicate if action is currently called to evaluate lazy eval or for classical traversal.
  // See isLazyEvaluating() doc for details.
  bool m_lazyEvaluating;

  //================== Begin sorted layers rendering ========================

  int m_sortedLayersNumPasses ;

  // Realizes a sorted layers rendering using depth peeling technics.
  void renderSortedLayers(SoNode *node) ;
  void initDepthPeeling(SoNode *node);

  // Rendering callback for sorted layers rendering.
  static void renderSortedLayersCB(void *userData, SoGLDepthPeeling *mgr) ;

  SoGLDepthPeeling *m_glDepthPeeling ;

  static void cameraPostCB( SoCamera *camera, SoGLRenderAction* renderAction );
  bool m_ignoreCameraPostCB;
  //================== End Sorted layers rendering ========================


  InvalidateCacheMode m_invalidateAllCaches;
  RenderMode m_restrictObjectRendering;

  bool m_floatingPointFramebuffer;

  //================ FSAA element management ============================
  SbBool m_fsaaEnable;
  int m_fsaaFilterMask;
  int m_fsaaNumSamples;

  SoGLRenderActionImpl* m_impl;
};


// INLINE METHODS
inline SoGLRenderActionSync* SoGLRenderAction::getSynchronizer() const
{
  return m_synchronizer;
}

inline SoGLRenderActionImpl* SoGLRenderAction::getImpl()
{
  return m_impl;
}

#ifdef _MSC_VER
#pragma warning( pop )
#endif
#endif /* _SO_GL_RENDER_ACTION_ */

