/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Pascal DOUX (Jan 1998)
**=======================================================================*/


#ifndef _SO_GLOBAL_SIMPLIFY_ACTION_
#define _SO_GLOBAL_SIMPLIFY_ACTION_


//------------------------------------------------------------------------------
// Includes

#include <Inventor/actions/SoAction.h>
#include <Inventor/actions/SoSubAction.h>
#include <Inventor/actions/SoCallbackAction.h>
#include <Inventor/actions/SoSimplifyAction.h>
#include <Inventor/nodes/SoMaterial.h>
#include <Inventor/nodes/SoBaseColor.h>
#include <Inventor/fields/SoMFFloat.h>

#include <Inventor/SoType.h>

class SoSimplifier;

/**
 * @VSGEXT Traverses the scene graph and collects all the triangles in a single list. This
 * list is then simplified and the result is stored in a new scene graph.
 * 
 * @ingroup actions LMV
 * 
 * @DESCRIPTION
 *   This class traverses the scene graph and collects all the triangles in a single
 *   list. This list is then simplified and the result is stored in a new scene
 *   graph.
 *   
 *   The SoGlobalSimplifyAction provides two simplification strategies:
 *   
 *   - SIMPLIFY_GLOBALLY: all the triangles found in the scene graph are collected in
 *   a single object. Then this object is simplified.
 *   
 *   - SIMPLIFY_BY_SUBGROUP: triangles are collected until a material change or a
 *   separator is found. 
 *   
 *   The default strategy is SIMPLIFY_BY_SUBGROUP.
 *
 *   NOTE: Colors and normal vectors in the existing shapes are @I not@i preserved!
 *
 *   Calling generateNormals(TRUE), which is the default, tells Open Inventor to
 *   compute per-vertex normal during this action. When disabled (calling
 *   generateNormals(FALSE)), objects may appear faceted instead of appearing smooth.
 *
 *   If generateNormals is false, it may be useful to call addShapeHints(TRUE).
 *   This tells Open Inventor to add an SoShapeHints node to the result scene graph
 *   that will cause normals to be computed during the next render traversal.
 *   
 *   By default the SoGlobalSimplifyAction collects all the triangles even if they are
 *   part of a simple shape (cube, cone, sphere, text3...). Setting the
 *   setCatchAllShapesFlag() flag to FALSE tells the action to collect triangles
 *   only if they belong to a complex shape.
 *   
 *   The getSimplifiedSceneGraph() method returns the root of the new scene
 *   graph after applying the SoGlobalSimplifyAction.
 *   
 *   Typically you will pass an instance of SoDecimator to the constructor. Please see
 *   SoDecimator for info about which geometry objects can be simplified.
 * 
 * @SEE_ALSO
 *    SoAction,
 *    SoLevelOfSimplification,
 *    SoShapeSimplifyAction,
 *    SoSimplifyAction
 * 
 * 
 */
class INVENTOR_API SoGlobalSimplifyAction : public SoSimplifyAction
{
  SO_ACTION_HEADER(SoGlobalSimplifyAction);

public:

  /**
   * Constructor. The algorithm used for simplifying is defined by the
   * SoSimplifier class that is passed in the constructor.
   */
  SoGlobalSimplifyAction (SoSimplifier *_simplifier) ;

  /**
   * Destructor.
   */
  ~SoGlobalSimplifyAction();

  /** @copydoc SoAction::clearApplyResult() */
  virtual void clearApplyResult();

  /** Strategy */
  enum Strategy {
    /** 
     * All triangles are stored in a single list and decimated globally.
	 *
     * WARNING: SoTexture (and possibly other properties), will be lost during
	 * simplification. Please use the SIMPLIFY_BY_SUBGROUP Strategy to keep SoTexture. */
    SIMPLIFY_GLOBALLY, 
    /**
     * Triangles are stored in multiple lists based on material, texture and Separator.
	 * Each list is simplified.  [Default]
     */
    SIMPLIFY_BY_SUBGROUP
  };
    
  virtual void apply(SoNode *node);
  virtual void apply(SoPath *path);
  virtual void apply(const SoPathList &path_list, SbBool obeys_rules = FALSE);

  /**
   * Sets the strategy used to determine how to group triangles into the
   * simplified shapes. The choices are SIMPLIFY_GLOBALLY and SIMPLIFY_BY_SUBGROUP.
   * @useenum{Strategy} Default is SIMPLIFY_BY_GROUP.
   */
  void setSimplificationStrategy(Strategy st) { simplificationStrategy = st;}
  /**
   * Gets the strategy used to determine how to group triangles into the
   * simplified shapes.
   */
  Strategy getSimplificationStrategy() const { return simplificationStrategy ;}

  /**
   * This returns the new scene graph which contains the simplified result.
   */
  SoSeparator *getSimplifiedSceneGraph() const { return simplifiedRoot; }
  
  /**
   * Sets the normal generation flag.
   * If true, per-vertex normals are calculated while simplifying.
   * Default value is TRUE.
   */
  void generateNormals(SbBool g) { generateNormal = g;}
  /**
   * Gets the normal generation flag.
   * If true, per-vertex normals are calculated while simplifying.
   */
  SbBool areNormalGenerated() const { return generateNormal ;}

  /**
   * Sets the flag that specifies whether simple shapes are simplified.
   * If true, simple shapes such as SoSphere, SoCone, etc, are faceted and simplified
   * also. If false, then they are ignored. Default is true.
   */
  void setCatchAllShapesFlag(SbBool c) { catchAllShapes = c;}
  /**
   * Gets the flag that specifies whether simple shapes are simplified.
   */
  SbBool areAllShapesCatched() const { return catchAllShapes ;}

  /**
   * Sets whether a shape hints node is added.
   * If true, a @B shapeHints @b node is added to force smooth normals if they were
   * not generated, and to turn on two-sided lighting. Default is false.
   */
  void addShapeHintsNode(SbBool a) { addShapeHints = a;}
  /**
   * Gets whether a shape hints node will be added.
   */
  SbBool isShapeHintAdded() const { return addShapeHints ;}

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

protected:
  virtual void beginTraversal(SoNode *) { beginTraversal();}
  virtual void endTraversal  (SoNode *) { endTraversal()  ;}

private: 

  virtual void prepareAction (SoCallbackAction &action);

  virtual void beginTraversal();
  virtual void endTraversal  ();
  
  SbBool         collectTriangles;

  // how to handle decimation
  Strategy       simplificationStrategy;

  // new scene graph
  SoSeparator   *simplifiedRoot; 

  // generate per-vertex normals
  SbBool         generateNormal;

  // catch all shapes flag: TRUE by default.
  // if FALSE, only complex shapes are catched, others are ignored
  SbBool         catchAllShapes;

  // if TRUE, adds a ShapeHint node with two sided lighting on.
  // when generateNormals is FALSE, the shape hints crease angle is
  // set to 0.55 thus objects will appear smooth even witouh normals.
  // This should be used carefully since it may slow down drawing but
  // it can improve rendering aspect when normals are not computed
  // default FALSE
  SbBool         addShapeHints;

  // last material added into simplified scenegraph
  SoMaterial    *lastMaterial;

  static SoCallbackAction::Response
  catchShapeCB    ( void* object, SoCallbackAction* action, const SoNode* node) 
  { return ((SoGlobalSimplifyAction*) object)->catchShape (action, node); }

  SoCallbackAction::Response
  catchShape      ( SoCallbackAction* action, const SoNode* node) ;

  static SoCallbackAction::Response
  catchEndShapeCB    ( void* object, SoCallbackAction* action, const SoNode* node) 
  { return ((SoGlobalSimplifyAction*) object)->catchEndShape (action, node); }

  SoCallbackAction::Response
  catchEndShape      ( SoCallbackAction* action, const SoNode* node) ;

  static SoCallbackAction::Response
  catchShapeAttrCB( void* object, SoCallbackAction* action, const SoNode* node)
  { return ((SoGlobalSimplifyAction*) object)->catchShapeAttr (action, node); }

  SoCallbackAction::Response
  catchShapeAttr  ( SoCallbackAction* action, const SoNode* node) ;

  static SoCallbackAction::Response
  catchGroupNodesCB ( void* object, SoCallbackAction* action, const SoNode* node)
  { return ((SoGlobalSimplifyAction*) object)->catchGroupNodes (action, node); }

  SoCallbackAction::Response
  catchGroupNodes   ( SoCallbackAction* action, const SoNode* node) ;

  void addTriangle (SoCallbackAction *action,
                    const SoPrimitiveVertex *vertex1,
                    const SoPrimitiveVertex *vertex2,
                    const SoPrimitiveVertex *vertex3);

  static void addTriangleCB (void *userData,
                             SoCallbackAction *action,
                             const SoPrimitiveVertex *vertex1,
                             const SoPrimitiveVertex *vertex2,
                             const SoPrimitiveVertex *vertex3)
  { ((SoGlobalSimplifyAction*) userData)->addTriangle (action, vertex1, vertex2, vertex3); }


};


#endif // _SO_GLOBAL_SIMPLIFY_ACTION_









