/*=============================================================================
** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),       **
**              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                **
**                                                                           **
**  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS  **
**  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR  **
**  WRITTEN AUTHORIZATION OF FEI S.A.S.                                      **
**                                                                           **
**                        RESTRICTED RIGHTS LEGEND                           **
**  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS **
**  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN **
**  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT **
**  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN **
**  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.        **
**                                                                           **
**                   COPYRIGHT (C) 1996-2022 BY FEI S.A.S,                   **
**                             BORDEAUX, FRANCE                              **
**                           ALL RIGHTS RESERVED                             **
=============================================================================*/

#pragma once

#include <Inventor/actions/SoHighlightRenderAction.h>

SO_PIMPL_PUBLIC_DECLARATION( SoHaloHighlightRenderAction )

/**
 * A halo highlight style.
 *
 * @ingroup actions
 *
 * @DESCRIPTION
 * SoHaloHighlightRenderAction is a render action which renders the specified
 * scene graph, then renders a "halo" highlight around each selected object.
 * See the images below for examples of this highlight technique.
 * Halo highlighting works with all kinds of shapes including VolumeViz and
 * MeshVizXLM.  However, if a selected object does not write to the depth buffer
 * for any reason, it will not be highlighted.
 *
 * Selected objects are specified by the first selection node (SoSelection or
 * SoExtSelection) node in the scene to which this action is applied. If there is
 * no selection node in the scene graph, then no objects will be highlighted.
 *
 * The application must explicitly tell Open Inventor to use a highlight render
 * action in place of the default SoGLRenderAction in the SoSceneManager class
 * using the setGLRenderAction() method. The viewer classes provide a convenience
 * setGLRenderAction() method that calls SoSceneManager (see e.g. SoWinRenderArea).
 *
 * For convenience, the viewer classes also provide the redrawOnSelectionChange()
 * method. This method attaches a sensor to the specified selection node and
 * automatically schedules a redraw if the set of selected objects is modified.
 *
 * Here is an example of an SoAlgebraicCone highlighted with an
 * SoHaloHighlightRenderAction:
 * @IMAGE halo_highlight_cone.png
 *
 * Here are parameters that can be adjusted:
 * - The halo color: see #setHaloColor()
 * - The halo width: see #setHaloWidth()
 * - The halo attenuation factor: see #setHaloSmoothFactor()
 * - Whether or not the halo can be seen through scene objects:
 *   see #setHaloAlwaysVisible()
 *
 * @B NOTES @b
 * - The rendering of the halo uses the depth buffer of the selected shapes as
 *   its boundary.
 * - Increasing the halo width value decreases performance.
 * - When multiple objects are selected, they are highlighted as one unique
 *   shape, i.e. the halo is done on the union of their depth buffer.
 *   This is illustrated on the following image:
 *   @TABLE_1B
 *     @TR 2 selected objects are highlighted as one
 *     @TR @IMAGE halo_highlight_multiple_objects.png
 *   @TABLE_END
 *
 * @EXAMPLE
 *   Here is an example of how a halo highlight can be specified for a particular
 *   selection node and render area:
 * \if_cpp
 *   \code
 *   // Scene graph
 *   SoSelection* selectionNode = new SoSelection();
 *   . . .
 *   // Viewer
 *   SoWinExaminerViewer* viewer = new SoWinExaminerViewer();
 *   viewer->setSceneGraph( selectionNode );
 *
 *   // Set the highlight render action
 *   viewer->setGLRenderAction( new SoHaloHighlightRenderAction() );
 *
 *   // Request automatic redraw when selection changes
 *   viewer->redrawOnSelectionChange( selectionNode );
 *   \endcode
 * \endif
 * \if_dotnet
 *   \code
 *   // Scene graph
 *   SoSelection selectionNode = new SoSelection();
 *   . . .
 *   // Viewer
 *   SoWinExaminerViewer viewer = new SoWinExaminerViewer();
 *   viewer.SetSceneGraph( selectionNode );
 *
 *   // Set the highlight render action
 *   viewer.SetGLRenderAction( new SoHaloHighlightRenderAction() );
 *
 *   // Request automatic redraw when selection changes
 *   viewer.RedrawOnSelectionChange( selectionNode );
 *   \endcode
 * \endif
 * \if_java
 *   \code
 *   // Viewer
 *   ViewerExaminer viewer = new ViewerExaminer();
 *
 *   // Set the highlight render action
 *   final RenderArea area = viewer.getRenderArea();
 *   area.setGLRenderAction( new SoHaloHighlightRenderAction() );
 *
 *   // Scene graph
 *   SoSelection selectionNode = new SoSelection();
 *
 *   // Request automatic redraw when selection changes
 *   selectionNode.addChangeCallback(new SoSelectionClassCB()
 *   {
 *     public void invoke(SoSelection s)
 *     {
 *       area.scheduleRedraw();
 *     }
 *   });
 *   . . .
 *   viewer.setSceneGraph( selectionNode );
 *   \endcode
 * \endif
 *
 * @SEE_ALSO
 *    SoLineHighlightRenderAction,
 *    SoColorHighlightRenderAction,
 *    SoBoxHighlightRenderAction,
 *    SoHighlightRenderAction,
 *    SoGLRenderAction,
 *    SoSelection,
 *    SoWinRenderArea
 *
 *
 */
class INVENTOR_API SoHaloHighlightRenderAction : public SoHighlightRenderAction
{
  SO_ACTION_HEADER( SoHaloHighlightRenderAction );
  SO_PIMPL_PUBLIC_HEADER( SoHaloHighlightRenderAction )

public:
  SoHaloHighlightRenderAction();
  SoHaloHighlightRenderAction( const SbViewportRegion& viewportRegion );

  virtual ~SoHaloHighlightRenderAction();

  /**
   * Defines the color of the highlight. Default is red (1,0,0).
   *
   * Here are examples of halo highlighting with different colors:
   *  @TABLE_1B
   *    @TR Red @TD Green @TD Blue
   *    @TR @IMAGE halo_highlight_red.png
   *    @TD @IMAGE halo_highlight_green.png
   *    @TD @IMAGE halo_highlight_blue.png
   *  @TABLE_END
   */
  void setHaloColor( const SbColor& color );

  /**
   * See #setHaloColor()
   */
  const SbColor& getHaloColor() const;

  /**
   * Defines the width of the halo in pixels. Default is 3.
   *
   * Only values greater or equal to 1 are accepted.
   * If a value of 0 or lower is given, the effective value
   * will be clamped to 1.
   *
   * The following images shows an example of halo highlighting with different
   * values for the halo width:
   *
   *  @TABLE_1B
   *    @TR width = 1 @TD width = 3 @TD width = 7 @TD width = 15
   *    @TR @IMAGE halo_highlight_width_1.png
   *    @TD @IMAGE halo_highlight_width_3.png
   *    @TD @IMAGE halo_highlight_width_7.png
   *    @TD @IMAGE halo_highlight_width_15.png
   *  @TABLE_END
   */
  void setHaloWidth( int width );

  /**
   * See #setHaloWidth()
   */
  int getHaloWidth() const;

  /**
   * Defines whether or not the halo can be seen through objects that are
   * placed in front of selected shapes. Default is FALSE.
   *
   * The following images shows an example of halo highlighting with
   * alwaysVisible set to FALSE and TRUE:
   *
   *  @TABLE_1B
   *    @TR alwaysVisible = FALSE @TD alwaysVisible = TRUE
   *    @TR @IMAGE halo_highlight_always_visible_F.png
   *    @TD @IMAGE halo_highlight_always_visible_T.png
   *  @TABLE_END
   */
  void setHaloAlwaysVisible( bool alwaysVisible );

  /**
   * See #setHaloAlwaysVisible()
   */
  bool isHaloAlwaysVisible() const;

  /**
   * Defines the amount of alpha attenuation of the halo contour.
   * Default is 0.
   * The value will be clamped between 0 and 1.
   *
   * The following images shows an example of halo highlighting with different
   * values for the smooth factor:
   *
   *  @TABLE_1B
   *    @TR smoothFactor = 0 @TD smoothFactor = 0.33 @TD smoothFactor = 0.66 @TD smoothFactor = 1
   *    @TR @IMAGE halo_highlight_smooth_00.png
   *    @TD @IMAGE halo_highlight_smooth_03.png
   *    @TD @IMAGE halo_highlight_smooth_06.png
   *    @TD @IMAGE halo_highlight_smooth_10.png
   *  @TABLE_END
   */
  void setHaloSmoothFactor( float smoothFactor );

  /**
   * See #setHaloSmoothFactor()
   */
  float getHaloSmoothFactor() const;

  static void initClass();
  static void exitClass();

SoINTERNAL public:
  /** @copydoc SoHighlightRenderAction::shouldTraverseHighlightPass() */
  virtual bool shouldTraverseHighlightPass() const;

protected:
  /** @copydoc SoHighlightRenderAction::getPreTraversalNode() */
  virtual SoNode* getPreTraversalNode();

  /** @copydoc SoHighlightRenderAction::getPostTraversalNode() */
  virtual SoNode* getPostTraversalNode();

private:
  // Copy Constructor Forbidden
  SoHaloHighlightRenderAction( const SoHaloHighlightRenderAction& ) : SoHighlightRenderAction() {}

  void commonConstructor();
};
