/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_HANDLE_EVENT_ACTION_
#define  _SO_HANDLE_EVENT_ACTION_

#include <Inventor/SbViewportRegion.h>
#include <Inventor/actions/SoSubAction.h>
#include <Inventor/actions/SoRayPickAction.h>

class SoEvent;
class SoPickedPoint;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoHandleEventAction
//
//  Event handling action. This traverses a graph looking for a node
//  or nodes interested in a particular event.
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Allows nodes in a graph to receive input events.
 * 
 * @ingroup actions
 * 
 * @DESCRIPTION
 *   This class is used to allow nodes in a scene graph to handle input events. It is
 *   usually invoked from a component derived from SoWinRenderArea when the component
 *   receives a window system event. SoSceneManager automatically creates a handle
 *   event action and provides a convenience method processEvent() to pass events to
 *   its managed scene graph.  The Open Inventor viewer classes use this method to
 *   pass events to the scene graph in selection mode.
 *
 *   A node can get the event currently being handled by calling getEvent().
 *   
 *   Manipulator, dragger and selection nodes respond to and process events. Most
 *   group nodes just pass the event to their children, while most other nodes
 *   simply ignore the action entirely. The SoEventCallback node invokes a callback
 *   / delegate method when traversed by a handle event action processing a specific 
 *   type of event. This is the primary mechanism for handling event input in Open
 *   Inventor. Once a node has indicated to the action that it has handled the event, 
 *   traversal stops.
 *   
 *   A node that handles an event can also grab future events. Once it has done so,
 *   all events will be sent directly to that node, with no traversal taking place,
 *   until the node releases the grab.  Draggers use this mechanism to ensure that
 *   they see the "finish" (e.g. mouse button up) event for their interaction.
 *
 *   Picking
 *
 *   A node can conveniently find out what, if any, geometry is under the cursor by
 *   calling the getPickedPoint() method. Applications can do this in the callback
 *   function assigned to an SoEventCallback node. The first time this method is
 *   called during a handle event traversal, the handle event action will automatically
 *   apply its internal SoRayPickAction to the scene graph returned by getPickRoot().
 *   The result is stored in case other nodes make the same query during the same
 *   traversal. The stored result can be cleared by calling clearApplyResult().
 *
 *   Some, but not all, options can be modified on the internal pick action (see for
 *   example setPickRadius()). Note that the internal pick action does not compute
 *   texture coordinates or normal vector for the picked point. Thus,
 *   getPickedPoint().getNormal() returns (0,0,0) and
 *   getPickedPoint().getTextureCoords() returns (0,0,0,0).
 *
 *   If a node needs to apply the pick action itself, for example to set different
 *   options, it can get the appropriate root node by calling getPickRoot().
 *
 *   Sets: SoViewportRegionElement
 * 
 * @SEE_ALSO
 *    SoEvent,
 *    SoEventCallback,
 *    SoPickedPoint,
 *    SoRayPickAction
 * 
 * 
 */
class INVENTOR_API SoHandleEventAction : public SoAction {

  SO_ACTION_HEADER(SoHandleEventAction);

 public:
  /**
   * Constructor takes viewport region to use; this is needed to perform a pick
   * operation when requested.
   */
  SoHandleEventAction(const SbViewportRegion &viewportRegion);

  // Destructor
#ifndef HIDDEN_FROM_DOC
  virtual ~SoHandleEventAction();
#endif // HIDDEN_FROM_DOC

  /**
   * Sets current viewport region to use for action.
   */
  void                setViewportRegion(const SbViewportRegion &newRegion);


  /**
   * Returns current viewport region to use for action.
   */
  const SbViewportRegion &getViewportRegion() const   { return vpRegion; }

  /** @copydoc SoAction::clearApplyResult() */
  virtual void clearApplyResult();

  /**
   * Sets the event being handled.
   */
  void                setEvent(const SoEvent *ev)     { event = ev; }
  /**
   * Returns the event being handled.
   */
  const SoEvent *     getEvent() const                { return event; }

  /**
   * Sets whether any node has yet handled the event.
   */
  void                setHandled()            { setTerminated(TRUE); }
  /**
   * Returns whether any node has yet handled the event.
   */
  SbBool              isHandled() const       { return hasTerminated(); }

  /**
   * Initiates grabbing of future events. All events will be sent to the given node
   * until the grab is released.
   */
  void                setGrabber(SoNode *node);
  /**
   * Releases the grab.
   */
  void                releaseGrabber()            { setGrabber(NULL); }
  /**
   * Returns the node that is currently grabbing events, or NULL if there is none.
   */
  SoNode *            getGrabber() const          { return eventGrabber; }

  /**
   * Sets the root node used for initiating a pick action for those nodes
   * that want to know what is under the cursor.
   */
  void                setPickRoot(SoNode *node);
  /**
   * Returns the root node used for initiating a pick action for those nodes
   * that want to know what is under the cursor.
   */
  SoNode *            getPickRoot() const             { return pickRoot; }

  /**
   * Sets the radius (in pixels) around the viewport-space point through which the ray
   * passes when doing ray picking. Default is 5 pixels. Ray picking is performed when 
   * getPickedPoint() is called.
   *
   * @B Note: @b By default the pick radius set here is only used when testing the ray 
   * against lines and points.  To enable pick radius for triangle based geometry use the
   * enableRadiusForTriangles() method.
   */
  void                setPickRadius(float radiusInPixels) 
    { pickAct->setRadius(radiusInPixels); }

  /** 
   * Returns the radius (in pixels) around the viewport-space point through which the ray
   * passes when doing ray picking
   */
  float               getPickRadius() const
    { return pickAct->getRadius(); }

   /** 
   * Enable pick radius for triangle-based shapes.
   *
   * If TRUE, the radius of the ray specified by setPickRadius()
   * is taken in account when checking for a ray intersection with 
   * triangle-based shapes (e.g. SoIndexedFaceSet).
   * Otherwise, the pick radius for these shapes is 1 pixel regardless of
   * the specified pick radius.  Default is FALSE for performance.
   */
  void enableRadiusForTriangles(SbBool flag);

  /**
   * Returns the frontmost object hit (as an SoPickedPoint) by performing a pick
   * based on the mouse location specified in the event for which the action is being
   * applied. The first time this is called for a particular event, an
   * SoRayPickAction is applied to find this object; subsequent calls for the same
   * event return the same information. The storage for the picked point remains
   * valid as long as the action is not re-applied or deleted.
   * 
   * Note: The applied SoRayPickAction does not compute
   * texture coordinates and normal vector for the picked point.
   * Thus, getPickedPoint().getNormal() returns (0,0,0) and
   * getPickedPoint().getTextureCoords() returns (0,0,0,0).
   */
  const SoPickedPoint *getPickedPoint();

  /**
   * Returns a list of objects intersected by a picking operation, sorted from
   * nearest to farthest.
   */
  const SoPickedPointList &getPickedPointList();

 SoEXTENDER public:
  /**
   * Set the SoPickedPoint object that will be returned by getPickedPoint.
   * Useful in some special cases where a pick operation has already been
   * done and the picked point is known.  Avoids doing the pick operation
   * again.  In other words, events will be processed as if the specified
   * point had been picked.  Reset the picked point by calling with NULL.
   */
  void                setPickedPoint( SoPickedPoint *point = NULL );

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

 protected:
  // Initiates action on graph
  virtual void        beginTraversal(SoNode *node);

 private:
  const SoEvent       *event;         // Event being handled
  SoNode              *pickRoot;      // Root node for initiating picking
  SoPickedPoint       *pickedPoint;   // PickedPoint from last pick
  SbBool              pickValid;      // Whether last pick is still valid
  SbBool              usedPickAll;    // TRUE if last pick used pickAll=TRUE
  SoRayPickAction     *pickAct;       // Pick action
  SoNode              *eventGrabber;  // Event grabber - gets all events
  SbViewportRegion    vpRegion;       // Current viewport region
  SbBool              pickedPtSet;    // TRUE if picked pt was set by app

};

#endif /* _SO_HANDLE_EVENT_ACTION_ */

