/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_RAY_PICK_ACTION_
#define  _SO_RAY_PICK_ACTION_

#include <Inventor/SoLists.h>
#include <Inventor/nodes/SoCamera.h>
#include <Inventor/actions/SoPickAction.h>
#include <Inventor/SbBox.h>
#include <Inventor/lists/SoPickedPointList.h>

class PickedPointListImpl;
class HomTransfData;
class SoDeviceContext;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoRayPickAction
//
//  Picking action that intersects a ray with objects in the scene
//  graph. The ray can be specified by calling setPoint() with a point
//  in a viewport in a rendering window or by calling setRay() with a
//  world-space ray. In the setPoint() case, a valid camera must be
//  encountered in the graph to set up the mapping to world space.
//
//  The "pickAll" flag indicates whether all intersections along the
//  ray should be returned (sorted by distance from the starting point
//  of the ray), or just the closest one. In either case, the
//  intersections are returned as an SoPickedPointList. Each
//  intersection can be examined by accessing the appropriate
//  SoPickedPoint in the list. The SoPickedPoint class provides
//  methods to get the intersection point, normal, and other info.
//
//////////////////////////////////////////////////////////////////////////////

/** 
 * Intersects objects with a ray cast into scene.
 * 
 * @ingroup actions
 * 
 * @DESCRIPTION
 *   This class performs picking by casting a ray into a scene and performing
 *   intersection tests with each object. The ray is extended to be a frustum
 *   a pyramid or a rectangular prism, a cone or a cylinder, depending 
 *   on the camera type and client inputs (refer to #setRay and #enableConicPickVolume)
 *   for intersection with points and lines.
 *   Each intersection is returned as an SoPickedPoint instance.
 *   
 *   The picking ray can be specified as either a ray from the camera 
 *   location through a particular viewport pixel, or as a world-space ray.  
 *   Calling any of the #setPoint, #setNormalizedPoint, or #setRadius methods 
 *   tells the action to compute the picking ray from a viewport pixel. 
 *   In this case, a camera node must be encountered during traversal of 
 *   the scene graph in order to determine the location of the ray in
 *   world space.
 *   
 *   Callers can request the action to compute all intersections along the ray (sorted
 *   closest to farthest) by setting the @B pickAll @b flag to TRUE. By default, the
 *   action computes only the closest intersection. In either case, the intersections
 *   are returned in an SoPickedPointList. Each intersection can be examined by
 *   accessing the appropriate SoPickedPoint in the list. The SoPickedPoint object
 *   provides methods to get the path (SoPath) to the picked geometry in the scene
 *   graph, the intersection point in 3D space and other info.
 *
 *   The SoPickedPoint object can also return one of the subclasses of SoDetail,
 *   which contains more information about the picked geometry.  For example, if
 *   a polygonal geometry node like SoIndexedFaceSet was picked, an SoFaceDetail
 *   object is returned which provides methods to get the index of the face in
 *   the primitive, the vertices of the face and so on.  For vertex based geometry
 *   each vertex can then be queried as an SoPointDetail.  @BR
 *   Note: Texture coordinates for the picked point are not computed by default (to save time).
 *   If you need this information, use #enableTexCoordsGeneration(). Or set the 
 *   environment variable OIV_PICK_GENERATE_ALL_PROPERTIES. You can also
 *   disable computing the normal vector for the picked if you do not need this
 *   information.  See #enableNormalsGeneration().
 *
 *   In the default mode, Inventor computes the intersection of the pick ray with 
 *   geometry nodes (face, line, point, volume, mesh, etc). In this case
 *   SoPickedPoint::getPoint() returns the coordinate of the intersection and
 *   SoPickedPoint::getDetail() typically returns an SoDetail class specific to 
 *   the picked geometry.  Since Open Inventor 9.0, SoRayPickAction also supports
 *   a POINT_PICKING mode (see #PickingMode).  In this mode, Inventor finds all the
 *   vertices inside the pick radius. This is only supported for SoBufferedShape 
 *   and classes derived from SoIndexedShape.  This mode can be much faster because,
 *   for example, it does not need to check for intersection with all the triangles
 *   of an SoIndexedFaceSet.
 *
 *   Applications can use the SoPickStyle node to control if and how geometry 
 *   can be picked.  For example application might want to specify that annotation
 *   geometry, e.g. a legend, is not pickable.  It can also specify that geometry
 *   should be picked using its bounding box rather than exact geometry.  This may
 *   be more efficient for text strings when it is not necessary to know which
 *   character in the string was picked.
 *
 *   The application can get platform independent input events, e.g. mouse button
 *   press, as SoEvent objects using the SoEventCallback node.  In the \if_dotnet 
 *   delegate \else callback \endif function the application can create an
 *   SoRayPickAction and apply it to the scene graph.  @B Note @b however that the
 *   application can also simply call the node's getPickedPoint() method.  In this
 *   case Open Inventor automatically applies a pick action to the scene graph and 
 *   returns the result, so the application does not need to use SoRayPickAction 
 *   directly.  Creating and using an SoRayPickAction explicitly does allow more
 *   options to be set.  In this case the application will normally call #setPoint
 *   with the position obtained from the event object.  If using
 *   system events directly, remember that Open Inventor expects Y pixel values to
 *   start from zero at the bottom of the window.
 *
 *   The SoSelection node provides an even higher level way to manage selecting and
 *   de-selecting objects in the scene.  This node automatically applies a pick
 *   action and maintains a list of currently selected objects (paths).  Using
 *   SoSelection the application does not need to use SoRayPickAction directly.
 *   The SoExtSelection node provides more complex picking algorithms.  For example
 *   it allows the user to select objects using a "rubberband" rectangle or a 
 *   freeform shape (lasso).  Open Inventor provides special render actions that
 *   can automatically highlight objects selected using an SoSelection or 
 *   SoExtSelection node. See SoBoxHighlightRenderAction, SoHaloHighlightRenderAction
 *   and SoLineHighlightRenderAction.
 *
 *   \anchor Picking_algo
 *   @B Picking algorithm on vertex shapes @b
 *
 *   Open Inventor implements two different picking algorithms for shapes defined by vertices:
 *   the GPU picking algorithm and the CPU picking algorithm.
 *
 *   The GPU picking algorithm is very efficient when the shapes have millions of vertices. For now, only
 *   SoPointSet and MeshVizXLM surface shapes (such as MoMeshSkin) are compatible with GPU picking.
 *   These conditions are required to activate GPU picking:
 *   - MeshVizXLM surface shapes are rendered with MoDrawStyle::displayFaces true
 *     and MoMaterial::enhancedColoring false.
 *   - The picking mode is DEFAULT, see setPickingMode().
 *   - The pick all property is FALSE, see setPickAll().
 *   - The picked shape is not in a SoRenderToTarget.
 *   - The action is associated with a scene manager, see setSceneManager().
 *
 *   Note: if gl_FragDepth is modified in the shader, the retrieved picked point is wrong.
 *
 *   The CPU picking algorithm is used when the previous conditions are not met.
 *   In this case, it does true geometric picking in 3D coordinates,
 *   so there are no limits to the number of objects
 *   that can be picked. Geometric picking means that precise intersections with the
 *   geometry are computed. It also means that picking works for any type of primitive,
 *   including polygonal geometry, meshes and volumes (VolumeViz).
 *   The picking volume can be projected through the scene using orthogonal or
 *   perspective projection. See #setRay for details.
 *   The shape of the picking volume can be rectangular (prism if orthogonal, frustum
 *   if perspective) or conic (cylinder if orthogonal, cone if perspective). See
 *   #enableConicPickVolume().
 *
 *   If the SoPickStyle::method selected is AUTO, Open Inventor chooses the most appropriate method (CPU or GPU)
 *   for each shape traversed in the scene graph.
 *
 *   @B CPU picking optimization @b
 *
 *   Pick traversals are optimized using hierarchical bounding boxes cached at the
 *   SoSeparator (and a few other) grouping nodes.  If the pick ray does not intersect
 *   the bounding box then the pick action will not traverse the children of that
 *   separator.  When optimizing for picking, the application should first consider
 *   how quickly Open Inventor can find the geometry to pick.
 *   Organizing the scene graph spatially to take advantage of the bounding box
 *   optimization can help with this.  Second the application should consider how
 *   long it will take to find the face to pick.  For a very large
 *   surface this can take a significant time.  Enabling triangle culling, splitting large
 *   surfaces into smaller pieces or using proxy geometry may help to reduce the time taken.
 *
 *   @B Traversing the camera node: @b
 *
 *   Note that when using SoRayPickAction with pixel coordinates (setPoint()) the
 *   pick action must traverse a camera node in order to unproject the coordinate
 *   back into 3D space.  If the application explicitly creates its own camera, 
 *   this is usually not a problem because the camera is in the application scene
 *   graph.  However, if the application allows the viewer to automatically create
 *   a camera then the camera is in the viewer's scene graph @I above @i the
 *   application scene graph.  Calling the viewer's getSceneGraph() method
 *   returns the application scene graph, not the complete viewer scene graph.
 *   To ensure that the traversed scene graph contains a camera, call the viewer's
 *   getSceneManager() method, then call the scene manager's getSceneGraph method.
 *   Alternatively, in an event callback call the event action's getPickRoot() method.
 *
 *  @B Picking VolumeViz shapes: @b
 *
 *  The SoVolumeRender and SoHeightFieldRender nodes uses the GPU to
 *  compute the picked voxel during a SoRayPickAction. For this to work, the SoRayPickAction
 *  must have its scene manager initialised using the method SoAction::setSceneManager().
 *  SoHandleEventAction does this automatically, so it is not necessary for the application
 *  to take any action when using (for example) an SoEventCallback node and calling the
 *  getPickedPoint() method. However, if the application creates its own SoRayPickAction
 *  then it should set the scene manager. If no scene manager is specified, a warning
 *  message is issued and the ray pick action is processed on the CPU.
 *
 *  @note
 *    - If necessary, you can force the SoVolumeRender to use only the CPU by setting the
 *      environment variable IVVR_GPU_PICKING to 0 (see SoPreferences).
 *      The SoPickStyle::method is ignored by SoVolumeRender.
 *    - The class SoHeightFieldRender uses the GPU to process a ray pick action.
 *      IVVR_GPU_PICKING variable and SoPickStyle::method are ignored.
 *      You cannot force the SoHeightFieldRender to use only the CPU.
 *    - The classes SoOrthoSlice, SoObliqueSlice, SoFenceSlice and SoVolumeSkin
 *      do not use the GPU to process a ray pick action.
 *      IVVR_GPU_PICKING variable and SoPickStyle::method are ignored.
 *      You cannot force these classes to use the GPU.
 *
 *  @B Hidden references: @b
 *
 *  \if_cpp
 *  SoRayPickAction creates one or more SoPath objects when applied to the scene graph.
 *  An SoPath object calls ref() on each node in the path. This reference will prevent
 *  the node from being destroyed for as long as the SoPath object exists. These SoPath
 *  objects are stored internally in the action and exist until the action object itself
 *  is destroyed or reset (see #clearApplyResult()).
 *  \else
 *  SoRayPickAction creates one or more SoPath objects when applied to the scene graph.
 *  The SoPath object references each node in the path. This reference will prevent the
 *  node and its associated memory from being reclaimed for as long as the SoPath object
 *  exists. These SoPath objects are stored internally in the action and exist until the
 *  action object itself is reclaimed or reset (see \if_dotnet #clearApplyResult()).
 *  \else #clearApplyResult()). \endif
 *  \endif
 *
 * @B Shapes that redefine the rayPick method: @b
 *
 * In order to make this action work properly with respect
 * to the picked path stored in the picked point list, any shape that redefines
 * the rayPick method must either call the method 
 * SoRayPickAction::setObjectSpace() or the method SoShape::computeObjectSpaceRay()
 * in its rayPick method before calling the addIntersection() methods.
 * See the chapter "Creating a node - Creating a Shape Node - Picking" 
 * in the @B Inventor Toolmaker @b Volume 1.
 *
 *   Sets: SoPickRayElement, SoViewportRegionElement
 *
 * @EXAMPLE
 * Do picking using an SoEventCallback node. @BR
 * Note: In this case, you could do picking by simply calling the SoHandleEventAction's
 * \if_dotnet GetPickedPoint() \else getPickedPoint() \endif method, which applies an
 * SoRayPickAction internally. However it may be necessary to use SoRayPickAction
 * directly in order to set specific options.
 * \if_cpp
 *   \code
 *   void mouseBtnCB(void *userData, SoEventCallback *node )
 *   {
 *     const SoEvent* evt = node->getEvent();
 *     // If this is a mouse button1 press event
 *     if (SO_MOUSE_PRESS_EVENT( evt, BUTTON1 )) {
 *       SoHandleEventAction* action = node->getAction();
 *       // Do picking using event's cursor position
 *       SoRayPickAction rayPick( action->getViewportRegion() );
 *       rayPick.setPoint( evt->getPosition() );
 *       rayPick.setRadius( 10 ); // Optional: Use larger pick radius
 *       rayPick.apply( action->getPickRoot() );
 *       const SoPickedPoint* pickedPt = rayPick.getPickedPoint();
 *       if (pickedPt != NULL) {
 *         SoFullPath* pickedPath = (SoFullPath*)pickedPt->getPath();
 *         SoNode*     pickedNode = pickedPath->getTail();
 *       }
 *     }
 *   }
 *   \endcode
 * \endif
 * \if_dotnet
 *   \code
 *   private void mouseBtnCB(SoEventCallback node, Object data)
 *   {
 *       SoEvent evt = node.GetEvent();
 *       // If this is a mouse button1 press event
 *       if (SoMouseButtonEvent.IsButtonPressEvent( evt, SoMouseButtonEvent.Buttons.BUTTON1 )) {
 *           SoHandleEventAction action = node.GetAction();
 *           // Do picking using event's cursor position
 *           SoRayPickAction rayPick = new SoRayPickAction( action.GetViewportRegion() );
 *           rayPick.SetPoint( evt.GetPosition() );
 *           rayPick.SetRadius( 10 ); // Optional: Use larger pick radius
 *           rayPick.Apply( action.GetPickRoot() );
 *           SoPickedPoint pickedPt = rayPick.GetPickedPoint();
 *           if (pickedPt != null) {
 *               SoPath pickedPath = pickedPt.GetPath();
 *               SoNode pickedNode = pickedPath.FullPath.GetTail();
 *           }
 *       }
 *   }
 *   \endcode
 * \endif
 * \if_java
 *   \code
 *   class MouseBtnCB extends SoEventCallbackCB {
 *       @Override
 *       public void invoke( SoEventCallback node ) {
 *           SoEvent evt = node.getEvent();
 *           // If button 1 was pressed
 *           if (SoMouseButtonEvent.isButtonPressEvent( evt, SoMouseButtonEvent.Buttons.BUTTON1 ))
 *           {
 *               SoHandleEventAction action = node.getAction();
 *               SoRayPickAction rayPick = new SoRayPickAction( action.getViewportRegion() );
 *               rayPick.setPoint( evt.getPosition() );
 *               rayPick.setRadius( 10 ); // Optional: Use larger pick radius
 *               rayPick.apply( action.getPickRoot() );
 *               SoPickedPoint pickedPt = rayPick.getPickedPoint();
 *               if (pickedPt != null) {
 *                   SoPath pickedPath = pickedPt.getPath();
 *                   SoNode pickedNode = pickedPath.full.getTail();
 *               }
 *           }
 *	    }
 *   }
 *   \endcode
 * \endif
 * 
 * @SEE_ALSO
 *    SoPickedPoint,
 *    SoPickedPointList,
 *    SoPickStyle
 * 
 * 
 */
class INVENTOR_API SoRayPickAction : public SoPickAction {

  SO_ACTION_HEADER(SoRayPickAction);

 public:

  /** 
   * Tells ray pick action in which view the pick occurs.
   * When stereo mode is active, user can choose between left, right, 
   * or normal view to perform the action. Only applicable when stereo
   * is active. Default is LEFT_VIEW.
   */
  static void setStereoMode(SoCamera::StereoMode stereoMode);

  /** Returns the view used to perform pick when stereo is active.
   */
  static SoCamera::StereoMode getStereoMode();

  /**
   * Constructor takes viewport region to use for picking. Even though the picking
   * operation may not involve a window per se, some nodes need this information to
   * determine their size and placement.
   */
  SoRayPickAction(const SbViewportRegion &viewportRegion);

  /** 
  * Clears the picked point list.
  *
  * The picked point list is automatically cleared
  * when the action is destroyed or re-applied.
  * However it may be useful to clear the list explicitly
  * in order to remove references to picked node(s).
  */
  virtual void clearApplyResult();

  // Destructor
#ifndef HIDDEN_FROM_DOC
  virtual ~SoRayPickAction();
#endif // HIDDEN_FROM_DOC

  //////////////////////////////////////////////////////////////////
  //
  //  Setting up the action before it is applied:
  //

  enum PickingMode {
    /** In the default mode, Inventor computes the intersection of the
     * pick ray with geometry nodes (face, line, point, volume, mesh, etc).
     * SoPickedPoint::getPoint() returns the coordinate of the intersection
     * and SoPickedPoint::getDetail() returns (usually) an SoDetail class
     * specific to the picked geometry.
     */
    DEFAULT,

    /** In this mode, Inventor finds all the vertices inside the pick radius.
     * However this is only supported for SoBufferedShape and classes derived
     * from SoIndexedShape.  This mode can be much faster because, for example,
     * it does not need to check for intersection with the triangles of an
     * SoIndexedFaceSet. For pick radius see #setRadius().  Specific different 
     * behaviors include:
     * - If no vertex is inside the pick radius, no picked point is returned.
     * - If a vertex is inside the pick radius, SoPickedPoint->getPoint() returns
     *   the coordinate of the vertex, not the intersection with the geometry.
     * - SoPickedPoint->getNormal() returns 0,0,0 
     * - SoPickedPoint->getTextureCoords() returns 0,0,0,0
     * - SoPickedPoint->getDetail normally returns an SoPointDetail instead of an
     *   SoFaceDetail (or whatever would normally be returned for that geometry).
     */
    POINT_PICKING

    // EDGE_PICKING,
    // FACE_PICKING
  };

  /**
   * Sets the picking mode.
   * @useenum{PickingMode}.
   * Default value is PickingMode::DEFAULT
   */
  void setPickingMode(PickingMode pickingMode);

  /**
   * Returns the #PickingMode used for the ray pick action.
   */
  PickingMode getPickingMode() const;

  /**
   * Sets the viewport point through which the ray passes, starting at the
   * camera position. Viewport coordinates range from (0,0) at the lower left to
   * (width-1,height-1) at the upper right. Default is (0,0).
   *
   * NOTE: You can use this method or #setNormalizedPoint or #setRay.
   * Whichever method you call last is the one that takes effect.
   */
  virtual void setPoint(const SbVec2s &viewportPoint);

  /**
   *  Float version of #setPoint. It can be used when a desktop is magnified
   *  on a wall of screens using ScaleViz with a tracker
   *  device calibrated for this wall.
   *
   * NOTE: You can use this method or #setNormalizedPoint or #setRay.
   * Whichever method you call last is the one that takes effect.
   */
  virtual void setPoint(const SbVec2f &viewportPoint);

  /**
   * Gets the viewport point in pixels
   * (returns the last value passed to #setPoint).
   */
  const SbVec2s&  getPoint() const       { return VPPoint; }

  /**
   *  Float version of #getPoint(). It can be used when a desktop is magnified
   *  on a wall of screens using ScaleViz with a tracker
   *  device calibrated for this wall.
   */
  const SbVec2f&  getPointFloat() const   { return VPPointFloat; }

  /**
   * Sets the viewport point in normalized coordinates, which range from (0,0) at the
   * lower left to (1,1) at the upper right.
   *
   * NOTE: You can use this method or #setPoint or #setRay. 
   * Whichever method you call last is the one that takes effect.
   */
  virtual void setNormalizedPoint(const SbVec2f &normPoint);

  /**
   * Gets the viewport point in normalized coordinates [0..1]
   * (returns the last value passed to #setNormalizedPoint).
   *
   * A point is inside the viewport if its coordinates are in the range [0, 1].
   *
   * If no point has been set or the setRay method was called instead of setXXXPoint,
   * this method returns an undefined coordinate (NaN, NaN).
   */
  const SbVec2f       getNormalizedPoint() const
    { return normVPPoint; }

  /**
   * Sets the radius around the point. The radius is defined in pixels when defining a ray
   * using the #setPoint or #setNormalizedPoint method, and is defined in world coordinates, 
   * when the ray is defined using the #setRay method. By default, for the setPoint and
   * setNormalizedPoint method the radius is 5 pixels. 
   *
   * By default, the radius is not taken into account for triangle
   * based shapes, only for points and lines. To enable this use the #enableRadiusForTriangles method.
   * When radius is taken into account, the ray is extended in 3D space.
   * For perspectivecameras, the ray is extended to be a cone.
   * For orthographic cameras, the ray is extended to be a cylinder. 
   *
   * Specifying a radius of 0 may give better performance. In particular, some shapes
   * like MoMeshSkin implement a fast GPU picking algorithm
   * that can only be used when radius is 0.
   *
   * \if_cpp
   * See also SoQtViewer::setPickRadius(), SoWinViewer::setPickRadius(), SoXtViewer::setPickRadius.
   * \endif
   */
  void                setRadius(float radius);

  /** 
   * Gets the radius (in pixels) around the point.
   */
  float               getRadius() const
    { return VPRadius; }

  /** 
   * Sets a world-space ray along which to pick. The ray is defined as a world space
   * starting point and direction vector. The direction vector will be normalized
   * automatically. The last two arguments specify optional near and far plane 
   * clipping during the pick operation. These values are distances from the start 
   * point along the direction vector, similar to nearDistance and farDistance in 
   * SoCamera. A negative distance (such as the default values) means disable 
   * clipping to that plane. 
   *
   * The ray-picking is orthogonal.  This means the pick volume
   * is a rectangular prism with a square base having edges of length #setRadius * 2
   * or else a cylinder having radius #setRadius if #enableConicPickVolume is set
   * to TRUE.
   *
   * NOTE: You can use this method or the #setPoint / #setNormalizedPoint 
   * Whichever method you call last is the one that takes effect.
   */
  virtual void setRay(const SbVec3f &start, const SbVec3f &direction,
                      float nearDistance = -1.0,
                      float farDistance = -1.0);

  /** 
   * Sets a world-space ray along which to pick in the the same way as the other
   * version of #setRay(), but allows you to set a view angle value. The ray is defined as a world space
   * starting point and direction vector. The direction vector will be normalized
   * automatically. The last two arguments specify optional near and far plane 
   * clipping during the pick operation. These values are distances from the start 
   * point along the direction vector, similar to nearDistance and farDistance in 
   * SoCamera. A negative distance (such as the default values) means disable 
   * clipping to that plane. 
   *
   * If fovy is non-zero, perspective ray-picking is used.  This means the pick 
   * volume is a frustum intersecting the plane
   * passing through the point specified as start argument and having normal vector specified
   * by the direction argument and whose base is a square having edges of length #setRadius * 2
   * or else a circle of radius #setRadius if #enableConicPickVolume is set to TRUE.
   *
   * NOTE: You can use this method or the #setPoint / #setNormalizedPoint 
   * Whichever method you call last is the one that takes effect.
   */
  virtual void setRay(float fovy, const SbVec3f &start, const SbVec3f &direction,
                      float nearDistance = -1.0,
                      float farDistance = -1.0);

  /**
   * Sets whether the action will return all objects intersected or just the
   * closest one.  Default is FALSE (only closest intersection).
   */
  void setPickAll(SbBool flag)                
    { pickAll = flag; }

  /**
   * Returns whether the action will return all objects intersected or just the
   * closest one.
   */
  SbBool isPickAll() const            
    { return pickAll; }

  //////////////////////////////////////////////////////////////////
  //
  //  Examining results after the action is applied:
  //

  /**
   * Returns list of picked points.
   *
   * This results in a copy of the picked point
   * list and a copy of every SoPickedPoint object in
   * the list.
   */
  const SoPickedPointList &getPickedPointList() const;
  
 /** 
  * Returns the indexed picked point from the list.
  * Returns NULL if index is larger than the number of picked points.
  *
  * \if_cpp 
  * Note: Each SoPickedPoint object contains an SoPath
  * with the path to the picked node; therefore the
  * picked node's ref count has been incremented and
  * will remain incremented until the SoPickedPoint
  * object is destroyed.
  *
  * The SoPickedPoint object is owned by the pick action.
  * It will be destroyed when the action is destroyed or
  * when either apply() or clearApplyResult() is
  * called.
  * \endif
  */
  SoPickedPoint *getPickedPoint(int index = 0) const;

  /** 
  * Clears the picked point list.
  *
  * The picked point list is automatically cleared
  * when the action is destroyed or re-applied.
  * However it may be useful to clear the list explicitly
  * in order to remove references to picked node(s).
  */
  SoDEPRECATED_METHOD(9600, "Use clearApplyResult() method")
  void clearPickedPointList()
  {
    clearApplyResult();
  }

  /** 
   * Enable radius for triangle-based shape
   *
   * If TRUE, the radius of the ray specified by #setRadius
   * is taken in account when checking for a ray intersection with 
   * triangle-based shapes (e.g., SoCylinder).
   * Otherwise, the pick radius for these shapes is 1 pixel regardless of
   * the specified pick radius.  Default is FALSE for performance.
   */
  void enableRadiusForTriangles(SbBool flag);
 
  /**
   * Returns whether the pick radius specified by #setRadius
   * is taken into account for picking on triangle-based shapes.
   */
  SbBool isRadiusEnableForTriangles();

  /**
   * 
   * Enables culling of triangles relative to the ray frustum.
   * Enabling culling improves performance for shapes containing a large number of triangles.
   * Default is FALSE.
   */
  static void enableTriangleCulling(SbBool flag) ;


  /**
   * Returns whether triangle culling is enabled
   */
  static SbBool isTriangleCulling()
    { return s_triangleCullingEnabled; }
  
  /**
   * Enables generation of texture coordinates for picked points.
   * Default is FALSE for performance (unless environment variable 
   * OIV_PICK_GENERATE_ALL_PROPERTIES envvar is set to TRUE).
   */
  void enableTexCoordsGeneration(const SbBool enable);

  /**
   * Enables generation of normal vectors for picked points.
   * Default is TRUE.
   */
  void enableNormalsGeneration(const SbBool enable);

  /** 
   * Returns whether texture coordinate generation is enabled for picked points.
   * See #enableTexCoordsGeneration().
   */
  SbBool isTexCoordsGenerationEnabled() const;

  /**
   * Returns whether generation of normal vectors is enabled for picked points.
   * See #enableNormalsGeneration().
   */
  SbBool isNormalsGenerationEnabled() const;

   /**
   * Controls the pick volume shape for picking with #setRay().
   * The default is FALSE, meaning that the picking volume is a rectangular
   * shape, either a prism or a frustum (depending on which version of
   * setRay was called).
   * When enableConicPickVolume is TRUE the picking volume
   * is a conic shape, either a cylinder or a cone (depending on which
   * version of setRay was called).
   *
   * Setting enableConicPickVolume to TRUE ensures that the entities picked
   * using #setRay() will be the same as picking using an equivalent call
   * to #setPoint(), but this mode is slightly more costly than frustum picking.
   */
  void enableConicPickVolume(SbBool flag);

  /**
   * Returns TRUE if the picking volume is a conic shape (not a frustum).
   */
  inline SbBool isConicPickVolume()
  { return m_conicPickVolume; }

  /** @copydoc SoAction::apply(SoNode*) */
  virtual void apply( SoNode* node );

  /** @copydoc SoAction::apply(SoPath*) */
  virtual void apply( SoPath* path );

  /** @copydoc SoAction::apply(const SoPathList&,SbBool) */
  virtual void apply(const SoPathList &pathList, SbBool obeysRules = FALSE) { SoAction::apply(pathList, obeysRules); }


 SoEXTENDER public:

  // If a ray was not defined with setRay(), this causes the world
  // space pick ray to be computed from the screen space point and
  // radius, using the current view specification from the state.
  // This is typically done when a camera is encountered during
  // traversal.
  void computeWorldSpaceRay();

  // This returns TRUE if the action has had a world space ray set
  // or computed
  SbBool hasWorldSpaceRay() const;

  // This is called by shapes to set up object space picking. It
  // uses the current state matrices to determine how to map between
  // world and object spaces. It should be called before calling any
  // of the intersection methods.
  // The second form takes a matrix to concatenate with the current
  // objToWorld matrix. It can be used, for example, if a shape has
  // sizing or positioning info built into it.
  //
  // These methods are also in charge of resetting the cached picked point
  // shape path, so the application must call the appropriate one once for
  // any shape that redefines the rayPick method before calling addIntersection().
  void                setObjectSpace();
  void                setObjectSpace(const SbMatrix &matrix);

  // These intersect the current object-space ray with a variety of
  // primitives: triangle, line, point, bounding-box. Intersection
  // with a triangle uses only the ray, while intersection with a
  // line or point uses the cone or cylinder around the ray. The
  // intersection with a bounding-box uses the cone/cylinder also,
  // since the contents of the box may be lines or points. NOTE: you
  // must call setObjectSpace() before calling any of these.

  // Triangle: returns intersection point, barycentric coordinates,
  // and whether the front side (defined by right-hand-rule) was hit.
  SbBool intersect(const SbVec3f &v0,
                   const SbVec3f &v1,
                   const SbVec3f &v2,
                   SbVec3f &intersection, SbVec3f &barycentric,
                   SbBool &front) ;

  // Line:
  SbBool intersect(const SbVec3f &v0, const SbVec3f &v1,
                   SbVec3f &intersection) const;

  // Point:
  SbBool intersect(const SbVec3f &point) const;

  /**
   * Bounding box: just return whether the ray intersects it. If
   * useFullViewVolume is TRUE, it intersects the picking view
   * volume with the box. Otherwise, it uses just the picking ray,
   * which is faster.
   */
  inline SbBool intersect(const SbBox3f &box, SbBool useFullViewVolume = TRUE)
  {
    SbXfBox3f xbox(box);
    return intersect(xbox, useFullViewVolume);
  }

  /**
   * Bounding box: just return whether the ray intersects it. If
   * useFullViewVolume is TRUE, it intersects the picking view
   * volume with the box. Otherwise, it uses just the picking ray,
   * which is faster.
   */
  SbBool intersect(const SbXfBox3f &box, SbBool useFullViewVolume = TRUE);


  // Returns an SbViewVolume that represents the object-space ray to
  // pick along. The projection point of the view volume is the
  // starting point of the ray. The projection direction is the
  // direction of the ray. The distance to the near plane is the
  // same as the distance to the near plane for the ray. The
  // distance to the far plane is the sum of the near distance and
  // the depth of the view volume.
  const SbViewVolume &getViewVolume() const
    { return objVol; }

  // Returns SbLine that can be used for other intersection tests.
  // The line's position is the starting point and the direction is
  // the direction of the ray. Given an intersection with this ray,
  // you can call isBetweenPlanes() to see if the intersection is
  // between the near and far clipping planes.
  const SbLine &getLine() const
    { return objLine; }

  // Returns TRUE if the given object-space intersection point is
  // between the near and far planes of the object-space view
  // volume, as well as any clipping planes that have been defined.
  // This test can be used to determine whether the point of
  // intersection of the ray with an object is valid with respect to
  // the clipping planes.
  SbBool isBetweenPlanes(const SbVec3f &intersection) const;

  /**
  * Adds an SoPickedPoint instance representing the given object
  * space point to the current list and returns it. If
  * pickAll is TRUE, this inserts the instance in correct sorted
  * order. If it is FALSE, it replaces the one instance in the list
  * only if the new one is closer; if the new one is farther away,
  * no instance is created and NULL is returned, meaning that no
  * more work has to be done to set up the SoPickedPoint.
  */
  SoPickedPoint *addIntersection(const SbVec3f &objectSpacePoint);

  // Adds an SoPickedPoint instance representing the given object
  // space point to the current list and returns a pointer to it. If
  // pickAll is TRUE, this inserts the instance in correct sorted
  // order. If it is FALSE, it replaces the one instance in the list
  // only if the new one is closer; if the new one is farther away,
  // no instance is created and NULL is returned, meaning that no
  // more work has to be done to set up the SoPickedPoint.
  SoPickedPoint *addIntersection_ ( const SbVec3f &objectSpacePoint, PickedPointListImpl* ppimplptr );
  SoPickedPoint *addIntersection_( SoPickedPoint *pp, PickedPointListImpl* ppimplptr );
  SoPickedPoint *addIntersection_( SoPickedPoint *pp );


  // Get PickedPoints during a traversal without sorting them by distance
  // this method must be used instead of the getPickedPointList
  // that should be used instead only AFTER the rayPick traversal is completed
  SoPickedPoint *getUnsortedPickedPoint( int i ) const;

  // Use preferably this method to get just the number of picked points
  // instead of getPickedPointList()->getLenght(), for performance reason
  int getPickedPointsListLength() const;

 protected:
  // Initiates action on graph
  virtual void beginTraversal(SoNode *node);

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  // return whether OIV_PICK_OPTIM is set to TRUE
  static SbBool isPickOptimSet()
  { return s_useAlternateIntersection; }

  // utility method to test if culling occurs because of clipping planes
  // worldCoord is used to use world coordinates instead of object space coordinates
  static bool isPtCulledByClippingPlanes( const SbVec3f &worldPt, SoState *state, bool worldCoord = true );

  // utility method to test if culling occurs because of clipping planes
  // worldCoord is used to use world coordinates instead of object space coordinates
  static bool isBboxCompletelyCulledByClippingPlanes( const SbBox3d& worldBBox, SoState* state, bool worldCoord = true );
  // worldCoord is used to use world coordinates instead of object space coordinates
  static bool isBboxCompletelyCulledByClippingPlanes( const SbBox3f& worldBBox, SoState* state, bool worldCoord = true );
  // worldCoord is used to use world coordinates instead of object space coordinates
  static bool arePointsCulledByAClippingPlane( const std::vector< SbVec3f>& pointVector, SoState* state, bool worldCoord = true );

  /** 
   * Properties that can be disabled during the generation of SoPickedPoint to the application.
   * Allows to speedup picking action.
   * Default is ALL but TEXCOORDS.
   */
  enum PickedProperties
  {
    /** normals **/
    NORMALS = 1 << 0,
    /** texture **/
    TEXCOORDS = 1 << 1,
    
    /** All properties */
    ALL = ~0
  };

  /** Allows to speedup picking action by removing some uneeded properties generation when possible.
   * @useenum{PickedProperties}.
   */
  void enablePickedProperties(const enum PickedProperties pickedProperty, const SbBool enable);

  /** Checks is a picking properties is currently required or not.
   * @useenum{PickedProperties}.
   */
  SbBool isPickedPropertiesEnabled(const enum PickedProperties pickedProperty) const;

  // Returns TRUE if the two SoPickedPoint are the same.
  SbBool isEqual(const SoPickedPoint *pp0, const SoPickedPoint *pp1) const;

  // Returns TRUE if the first intersection point is closer to the
  // starting point of the ray than the second.
  SbBool isCloser(const SoPickedPoint *pp0, const SoPickedPoint *pp1) const;
  SbBool isCloser(const SbVec3f& p0, const SbVec3f& p1) const;

  // get depth for ray picking sorting
  double getDepth(const SbVec3f& p0) const;

  //interface exposed to share SoPath among SoPickedPoints
  void resetCurrPathCache();

  enum PointPositionClassification
  {
    INSIDE = 0,
    BEHIND_FAR = 1,
    BEFORE_NEAR = 2,
    BESIDE_LEFT = 3,
    BESIDE_RIGHT = 4,
    UNDER_BOTTOM = 5,
    ABOVE_TOP = 6
  };

  PointPositionClassification homogeneSpaceIntersection( const SbVec3f &point, SbVec4f &pPoint ) const;
  SoPath *getClonedCurrPath();

  static bool isMTPickingActive();

  // get VPradius in pixels (even if it was provided in world coords)
  float getVPRadiusPixel() const ;

  /* Because picking points should take care of pointSize without modifying
   * the internal projection matrix,
   * a distosion depending of the size of viewPortRegion is introduced
   * This distorsion can be computed from pointSize
   * By default pointSize is (0, 0)
   * Normal usage should be setPointSizeForPicking( WidthInPixels, HeightInPixels)
   * WARNING: Don't forget to restore previous pointSize after your traversal to avoid side effect
   */
  void setPointSizeForPicking( float pointWidth, float pointHeight);

  // Same as previous with an SbVec2f
  void setPointSizeForPicking( const SbVec2f & pointSize)
  { setPointSizeForPicking(pointSize[0], pointSize[1]); }

  // Return current pointSize for picking points
  SbVec2f getPointSizeForPicking()
  { return SbVec2f( m_pointWidthForPicking, m_pointHeightForPicking ); }

  struct GPUPickInfo
  {
    bool isValid = false;
    SbVec3f modelPosition;
    uint32_t shapeId = 0;
    uint32_t faceId = 0;
    float distToPickCenter = 0.f;
  };

  bool canGPUPick() const;

  /**
   * @brief Collect current path for GPU picking.
   */
  void collectGPUPickingPath();

  // Render the scene with GPU Picking data attached and
  // retrieve the picking data from the GPU
  // The data can be accessed via getGPUPickingInfo()
  void renderGPUPickingScene();

  // Return the picking informations (faceId, shapeId, Position)
  // for the selected point.
  const std::vector<GPUPickInfo>& getGPUPickInfo() const;

  PickedPointListImpl* getPickedPointListImpl() const;

 protected:
  SbBool lineWasSet;  // TRUE if a world-space line was set
  SbBool rayWasComputed;      // TRUE if ray computed by camera
  SbBool pickAll;     // Pick all objects or just closest
  SbVec2s VPPoint;    // Point in viewport coordinates
  SbVec2f VPPointFloat; // Point in viewport coordinates
  SbVec2f normVPPoint;        // Point in normalized vp coordinates
  SbBool normPointSet;        // TRUE if setNormalizedPoint called
  float VPRadius;     // Radius in viewport space pixels
  SbMatrix objToWorld;        // Object-to-world space matrix
  SbMatrix worldToObj;        // World-to-object space matrix

  // The ray is defined as an SbViewVolume as in the
  // SoPickRayElement, and is usually stored in an instance of the
  // element. This stores the ray if it is set using setRay().
  SbViewVolume worldVol;

  // Users can specify negative near and far distances to indicate
  // that picks should not be clipped to those planes. These flags
  // store that info, since the distances in the view volume can't
  // be negative.
  SbBool clipToNear, clipToFar;

  // These store the object-space ray info as a view volume and a
  // line. See the comments on getViewVolume() and getLine().
  SbLine  objLine;     // Line representing ray
  SbLined objLineD;
  SbViewVolume objVol;                // View volume representing ray

  // If the caller passes a matrix to setObjectSpace(), the inverse
  // of it is stored here so the object-space angle can be computed
  // correctly later. The extraMatrixSet flag is set to TRUE in this
  // case.
  SbBool extraMatrixSet;
  SbMatrix extraMatrix;

  // Computes matrices to go between object and world space
  void computeMatrices();

  // Computes object-space view volume and line
  void computeObjVolAndLine();

  // updates RayBBox and WMat
  void setUpRayBBoxAndWMat( const SbVec3f &centerPt );

  void evalCenterPts( SbVec3f &centerPt, SbVec3d &centerPtD, SbVec3d &projPtD );

  // get VPRadius in world coordinates (even if it was provided in pixels)
  float getVPRadiusWorld() const ; 

  // Computes distance t of a point along a ray: point = start + t * direction.
  // The point has to be on the ray for this to work
  static float        rayDistance(const SbVec3f &start,
                                  const SbVec3f &direction,
                                  const SbVec3f &point);

  // Set commons properties of the action (called by the setRay method)
  void setRayCommonProperties (const SbVec3f &start, const SbVec3f &direction, 
                               float fovy, float aspectRatio = 1.0f, float nearDistance=-1.0f, float farDistance=-1.0f);

  // Initialize some members variable when a new point in viewport has been set
  void initializeMembersAfterVPPointSet(bool normalizedPointSet);



  // check if the intersection is inside conic picking volume
  inline SbBool isInConicPickingVolume( const SbVec3f &intersection ) const;

  void initSetRayData ( const SbVec3f &start, const SbVec3f &direction,
                        float fovy, float aspectRatio = 1.0f, 
                        float nearDistance = -1.0f, float farDistance = -1.0f );
  
  // Store the current ray bounding box for triangle culling
  SbBool m_canUseTriangleCulling;
  SbBox3f  m_rayBBox ; 
  

  // store the current viewvolume matrix and near far in homogene space
  // to optimize intersection computation 
  SbMatrixd m_wmat;
  double m_wnear;
  double m_wfar;
  HomTransfData *m_htdta;

  // Indicate if triangle are culled by ray frustum.
  SbBool canUseTriangleCulling() const
  { return (s_triangleCullingEnabled && m_canUseTriangleCulling); }

  static SbBool  s_triangleCullingEnabled ;

  // generate all picked point properties? (OIV_PICK_GENERATE_ALL_PROPERTIES)
  static SbBool s_generateAllPickedProperties;

  //
  static SoCamera::StereoMode s_stereoMode;

  // Use faster intersection method (OIV_PICK_OPTIM)
  static SbBool s_useAlternateIntersection;

  // Force the raypick action to take ray radius in account
  SbBool m_bEnableRadiusForTriangles;

  // Store ray properties
  SbVec3f m_direction;
  SbVec3f m_start;

private:
  /** Contains common code part between apply(SoNode) and apply(SoPath).
   * Note that this template is just here to factorize the 2 apply method. That'a why template
   * argumet should be only SoNode or SoPath. */
  template<typename SO_NODE_OR_SO_PATH> void commonApply(SO_NODE_OR_SO_PATH* nodeOrPath);

  float getCorrectedVPRadius() const;

  int m_pickedPropertiesMask;
  PickingMode m_pickingMode;

  int m_VPRadiusState;      // how VPRadius has been initialized

  // data buffered for the lineWasSet case
  float m_fovy;             
  float m_nearDistance;
  float m_farDistance;
  float m_aspectRatio;

  // to force pick volume to be a cone (perspective pick) or a cylinder (orthogonal pick)
  SbBool m_conicPickVolume;

  PickedPointListImpl *m_pickedPointListlImpl;

  // Only computed by setPointSizeForPicking()
  // Only used in homogeneSpaceIntersection()
  // Store picking error depending of viewPort
  float m_pointPickingErrorX;
  float m_pointPickingErrorY;
  // Store extended pick trap depending of pointSize
  float m_projSpaceWidth;
  float m_projSpaceHeight;
  // Store pointSize for picking point
  float m_pointWidthForPicking;
  float m_pointHeightForPicking;

  /// Collected paths to be GPU picked
  SoPathList m_gpuPickingCollectionPaths;
  std::vector<SoRayPickAction::GPUPickInfo> m_gpuPickingInfos;

  /**
   * @brief Fill the GPU picking infos from updated GPU picking textures.
   */
  void fillGPUPickInfos();
};

inline 
void SoRayPickAction::enableRadiusForTriangles(SbBool flag)
{
  m_bEnableRadiusForTriangles = flag;  
}

inline
SbBool SoRayPickAction::isRadiusEnableForTriangles()
{
  return m_bEnableRadiusForTriangles;
}
#endif /* _SO_RAY_PICK_ACTION_ */

