/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_WRITE_ACTION_
#define  _SO_WRITE_ACTION_

#include <Inventor/actions/SoAction.h>
#include <Inventor/actions/SoSubAction.h>
#include <Inventor/SoOutput.h>

/**
 * Writes a scene graph to a file.
 * 
 * @ingroup actions
 * 
 * @DESCRIPTION
 *   This class is used for writing scene graphs in Open Inventor (.iv) format. 
 *   SoWriteAction traverses the scene graph and uses an instance of SoOutput
 *   to write each node. SoOutput methods can be called to specify what file or 
 *   memory buffer to write to. SoOutput supports both ASCII (default) and 
 *   binary formats and provides some convenience functions for opening and 
 *   closing files. See SbFileHelper for more convenience functions.
 *   Since Open Inventor 8.1, SoOutput can write compressed data in the lossless 
 *   Zlib (gzip) format.  Both ASCII and binary format files may be compressed.
 *
 *   STL @BR
 *   Open Inventor can also export geometry to an STL (.stl) format file.
 *   See SoSTLWriteAction.
 *
 *  @EXAMPLE
 *   An instance of SoOutput is automatically created by SoWriteAction.
 *   This is typically the only instance needed. For example:
 *   \if_cpp
 *     \code
 *     SoWriteAction writeAction;
 *     writeAction.getOutput()->openFile( "output.iv" );
 *     writeAction.getOutput()->setBinary( TRUE );  // Optional: write binary format
 *     writeAction.apply( root );
 *     writeAction.getOutput()->closeFile();
 *     \endcode
 *   \endif
 *   \if_dotnet
 *     \code
 *     SoWriteAction writeAction = new SoWriteAction();
 *     writeAction.GetOutput().OpenFile( "output.iv" );
 *     writeAction.GetOutput().SetBinary( true );  // Optional: write binary format
 *     writeAction.Apply( root );
 *     writeAction.GetOutput().CloseFile();
 *     \endcode
 *   \endif
 *   \if_java
 *     \code
 *     SoWriteAction action = new SoWriteAction();
 *     SoOutput      output = action.getOutput();
 *     output.openFile( "output.iv" );
 *     output.setBinary( true );  // Optional: write binary format
 *     action.apply( root );
 *     output.closeFile();
 *     \endcode
 *   \endif
 * 
 * @SEE_ALSO
 *    SoOutput, SoSTLWriteAction
 */
class INVENTOR_API SoWriteAction : public SoAction {

  SO_ACTION_HEADER(SoWriteAction);

 public:
  /**
   * Constructor.
   */
  SoWriteAction();

  /**
   * Constructor that takes an SoOutput to use for output.
   */
  SoWriteAction(SoOutput *out);

  // Destructor
#ifndef HIDDEN_FROM_DOC
  virtual ~SoWriteAction();
#endif // HIDDEN_FROM_DOC

  /**
   * Returns reference to SoOutput instance in action.
   */
  SoOutput *          getOutput() const               { return output; }

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

  // Continues write action on a graph or path
  void                continueToApply(SoNode *node);
  void                continueToApply(SoPath *path);

  /**
   * Internal helper method to write a scene graph to a file
   * @param outputFilename output filename.
   * @param scenegraph scene graph to save to outputFilename
   * @param binaryFormat save as a binary file if true (default is false)
   * @param compressedFormat save as a compressed file if true (default is false)
   * @return TRUE on success.
   */
  static bool writeFile(const SbString& outputFilename, SoNode* scenegraph, const bool binaryFormat=false, const bool compressedFormat=false);

 protected:
  // Initiates action on graph
  virtual void        beginTraversal(SoNode *node);

  // Override method to return FALSE, since this action applies
  // itself to each path separately, so it doesn't need the extra
  // overhead of compacting the list.
  virtual SbBool      shouldCompactPathLists() const;

 private:
  SoOutput            *output;        // Output info
  SbBool              createdOutput;  // TRUE if output created by action
  SbBool              continuing;     // TRUE only if continueToApply()
  // was used to apply action
  SbBool              doOneStage;     // TRUE if only supposed to do 1 stage
  SbPList             savedLists;     // Path lists saved for later apply

  // Performs traversal on a path list, which is a little tricker
  void                traversePathList(SoNode *node);
};

#endif /* _SO_WRITE_ACTION_ */

