/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#if !defined(SO_FXAAPARAMETERS_H)
#define SO_FXAAPARAMETERS_H

#include <Inventor/antialiasing/SoAntialiasingParameters.h>

/** 
 * @VSGEXT Antialiasing parameters class for the FXAA algorithm.
 *
 * @ingroup Antialiasing
 *
 * @DESCRIPTION
 * This class provides the different parameters that can be set to 
 * configure the FXAA antialiasing algorithm. These objects are used with the
 * setAntialiasing() method in SoSceneManager and some viewer classes (e.g. SoWinGLWidget).
 *
 * @SEE_ALSO
 *   SoAccumulationAntialiasingParameters,
 *   SoFullSceneAntialiasingParameters
 */
class INVENTOR_API SoFXAAParameters : public SoAntialiasingParameters
{
public:

  /**
   * Constructor which takes a value for each possible parameter.
   *
   * @param qualityPreset Quality value in the range [10;15] , [20;29] or equal to 39.
   * @param subPixQuality Chooses the amount of sub-pixel aliasing removal (1.0 upper limit = softer, 0.0 = off).
   * @param edgeThresholdQuality The minimum amount of local contrast required to apply algorithm (0.333 = too little but faster, 0.063 = overkill and slower).
   * @param edgeThresholdMin Trims the algorithm from processing darks (0.0833 = upper limit the start of visible unfiltered edges, 0.0625 = high quality and faster, 0.0312 = visible limit but slower).
   */
  SoFXAAParameters(int qualityPreset, float subPixQuality, float edgeThresholdQuality, float edgeThresholdMin);

  /** Default destructor. */
  virtual ~SoFXAAParameters();

  /**
   * Returns the quality preset.
   */
  int getQualityPreset() const;

  /**
   * Returns the subPixQuality parameter.
   */
  float getSubPixQuality() const;

  /**
   * Returns the edgeThresholdQuality parameter.
   */
  float getEdgeThresholdQuality() const;

  /**
   * Returns the edgeThresholdMinimum parameter.
   */
  float getEdgeThresholdMin() const;

  bool operator==( const SoFXAAParameters& other ) const
  {
    return m_qualityPreset == other.m_qualityPreset &&
      m_subPixQuality == other.m_subPixQuality &&
      m_edgeThresholdQuality == other.m_edgeThresholdQuality &&
      m_edgeThresholdMin == other.m_edgeThresholdMin;
  }

  bool operator!=( const SoFXAAParameters& other ) const
  {
    return !operator==( other );
  }

  /** Copy to a new instance */
  virtual SoAntialiasingParameters* copy() const
  {
    return new SoFXAAParameters( *this );
  }

private:

  int m_qualityPreset;
  float m_subPixQuality;
  float m_edgeThresholdQuality;
  float m_edgeThresholdMin;
};

#endif // SO_FXAAPARAMETERS_H
