/*=================================================================================
*** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),        ***
***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
***                                                                            ***
***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
***                                                                            ***
***                        RESTRICTED RIGHTS LEGEND                            ***
***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
***                                                                            ***
***                   COPYRIGHT (C) 1996-2019 BY FEI S.A.S,                    ***
***                              BORDEAUX, FRANCE                              ***
***                            ALL RIGHTS RESERVED                             ***
=================================================================================*/

#pragma once

#include <Inventor/bundles/SoBundle.h>
#include <Inventor/elements/SoTangentElement.h>
#include <Inventor/elements/SoTangentBindingElement.h>
#include <Inventor/misc/SoTangentGenerator.h>

class SoTangentCache;

/**
 * Class: SoTangentBundle
 *
 * Bundle that allows shapes to deal with tangents and tangent bindings
 * more easily. This class provides a fairly simple interface to
 * tangent handling, including tangent generation.
 *
 * [OIV-WRAPPER-CLASS NO_WRAP]
 */
SoEXTENDER class INVENTOR_API SoTangentBundle : public SoBundle
{
public:
  SoTangentBundle( SoAction* action );

  virtual ~SoTangentBundle();

  // Returns true if tangents need to be generated.
  bool shouldGenerate( int numNeeded );

  // Initializes generation in the case where shouldGenerate() is
  // not called. (shouldGenerate() does this automatically).
  void initGenerator( int initialNum = 100 );

  //////////////////////
  //
  // If shouldGenerate() returns TRUE, these methods can be used by
  // shapes to specify the geometry to generate tangents for. They
  // are front-ends to methods on the SoTangentGenerator class:
  //

  // Send a line strip's worth of vertices. Begin a strip, send as
  // many vertices as you want, and then end the strip.
  void beginLineStrip()
  {
    m_generator->beginLineStrip();
  }

  void lineStripVertex( const SbVec3f& point )
  {
    m_generator->lineStripVertex( point );
  }

  void endLineStrip()
  {
    m_generator->endLineStrip();
  }

  // See SoTangentGenerator for details.
  void setupIndexedVertexArray( const SbVec3f* pointArray )
  {
    m_generator->setupIndexedVertexArray( pointArray );
  }

  void lineStripVertex( const int32_t index )
  {
    m_generator->lineStripVertex( index );
  }

  void setupApproxShapeSize( const int32_t numVert )
  {
    m_generator->setupApproxShapeSize( numVert );
  }

  // Calculates the tangents once all vertices have been sent. The
  // tangents are stored by the bundle so the caller does not have to
  // deal with them directly.  The startIndex argument specifies the
  // index at which the generated tangents will begin-- it can be
  // used by shapes that allow the coordinates and tangents to be
  // offset (non-indexed shapes).  If addToState is TRUE, the
  // state will be pushed, the tangents will be added to the
  // state; otherwise, you can just use
  // getGeneratedTangents/getNumGeneratedTangents to get the generated
  // tangents.
  //
  // The tangent bundle DOES NOT FREE the generated tangents!  It is
  // expected that tangents will be added to a tangent cache, and the
  // tangent cache will free the tangents.
  //
  void generate( int startIndex = 0, bool addToState = true );

  // Returns the generated tangents.
  const SbVec3f* getGeneratedTangents() const
  {
    return m_generator->getTangents();
  }

  int getNumGeneratedTangents() const
  {
    return m_generator->getNumTangents();
  }

  SoTangentBindingElement::Binding getTangentBinding() const
  {
    return m_generator->getBinding();
  }


  // This allows shapes that generate their own tangents (for
  // efficiency) to store the resulting tangents in the state. The
  // bundle takes care of pushing/popping the state.
  // Note:  This method only adds the tangents to the state, it does
  // NOT store a pointer to them (so you cannot call the
  // getGenerated() routines)
  void set(
    int32_t numTangents,
    const SbVec3f* tangents,
    SoTangentBindingElement::Binding binding /*= SoTangentBindingElement::PER_VERTEX */,
    int startIndex = 0
  );

  // to set the SoTangentBundle from a SoTangentCache
  void set( SoTangentCache* tc );

  // Returns indexed tangent. This can be used for primitive
  // generation or during rendering
  const SbVec3f& get( int index ) const
  {
    return m_tangElt->getNum() != 0 ? m_tangElt->get( index - m_startIndex ) : s_zVers;
  }

  int getStartIndex() const
  {
    return m_startIndex;
  }

SoINTERNAL public:
    SoTangentGenerator* m_generator;

private:
  // Tangent elements:
  const SoTangentElement* m_tangElt;
  bool m_pushedState; // We pushed state to set tangents
  SoNode* m_currentNode; // Node that created the bundle
  static const SbVec3f s_zVers;
  int m_startIndex; // this the offset used in indexing the tangents WITHOUT allocating useless memory
  bool m_tangSetOrGen;
};
