/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Patrick Vigneras (Nov 1999)
**=======================================================================*/

#ifndef  _SO_TEXTURE_COORDINATE3_BUNDLE
#define  _SO_TEXTURE_COORDINATE3_BUNDLE

#include <Inventor/bundles/SoBundle.h>
#include <Inventor/elements/SoTextureCoordinate3Element.h>
#include <Inventor/misc/SoState.h>
#include <memory>

class TexGenEvaluatorBundle3;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoTextureCoordinate3Bundle
//
//  Bundle that allows shapes to deal with texture coordinates more
//  easily. This class provides a fairly simple interface to texture
//  coordinate handling, including default texture coordinate
//  generation. This can be used during either rendering or primitive
//  generation.
//
//  This class can be used during either rendering or primitive
//  generation. For primitive generation, there are two cases,
//  distinguished by the flag returned by isFunction(). If this
//  flag is TRUE, the texture coordinates are to be generated using
//  the get(point, normal) method, which uses a software texture
//  coordinate function. (This process is also used for texture
//  coordinates that are generated by default when necessary - in this
//  case, the function does a linear map across two sides of the
//  bounding box of the shape.) If the isFunction() flag is FALSE, the
//  coordinates are accessed directly from the element using the
//  get(index) method.
//
//  For GL rendering, there is an additional case. If
//  needCoordinates() returns FALSE, no texture coordinates need to be
//  sent at all, and the bundle does not have to be used for anything
//  else. Otherwise, send(index) should be used.
//
//////////////////////////////////////////////////////////////////////////////
/** [OIV-WRAPPER-CLASS NO_WRAP] */
SoEXTENDER class INVENTOR_API SoTextureCoordinate3Bundle : public SoBundle {

public:
  // Constructor - takes the action the bundle is used for and a
  // flag to indicate whether the bundle is being used for
  // rendering. If this is TRUE, the bundle can be used to send
  // texture coordinates to GL. If it is FALSE, the setUpDefault
  // flag (default TRUE) indicates whether to set up a texture
  // coordinate function if the binding is DEFAULT. Shapes can pass
  // FALSE here if they are picking and want to delay computation of
  // the texture coordinates until an intersection is found.
  SoTextureCoordinate3Bundle(SoAction *action, SbBool forRendering,
                             SbBool setUpDefault = TRUE);

  // This constructor is used for meshes such as QuadMesh.
  // The constructor calls different methods to insure
  // that the texture doesn't get mapped edgeon. The last argument is
  // just to make the methods unique.
  SoTextureCoordinate3Bundle(SoAction *action, SbBool forRendering,
                             SbBool setUpDefault, SbBool forMesh);
  // Destructor
  ~SoTextureCoordinate3Bundle();

  // Returns TRUE if texture coordinates are needed at all
  SbBool              needCoordinates() const { return needCoords; }

  // return value to determine which get() to use.
  SbBool              isFunction() const      { return isFunc; }

  // Returns texture coordinate computed by function during
  // primitive generation or rendering
  SbVec4f             get(const SbVec3f &point, const SbVec3f &normal) const
  { return texCoordElt->get(point, normal); }

  // Returns indexed texture coordinate during primitive generation
  // or rendering
  SbVec4f             get(int index) const
  { if (tCoords) return(SbVec4f(tCoords[index][0],tCoords[index][1],
                                0.0, 1.0)); 
  else return texCoordElt->get4(index); }

  // Sends indexed texture coordinate to GL during rendering
  void                send(int index) const;

private:
  // TextureCoordinate3 elements:
  const SoTextureCoordinate3Element     *texCoordElt;
  const SoTextureCoordinate3Element   *GLTexCoordElt;

  SbBool              needCoords;     // Texture coordinates are needed
  SbBool              isFunc;         // Coordinates generated by function
  SbBool              isRendering;    // Bundle being used for rendering
  SbBool              setFunction;    // We set default coord func in state

  // These indicate the dimensions used for S, T and R for default
  // texture coordinate generation
  int                 coordS, coordT, coordR;
  // These hold the vectors used for default texture coordinate generation
  SbVec4f             sVector, tVector, rVector;
  // This holds the texture coords from a vertexProperty node:
  const SbVec3f *     tCoords;

  // Sets up bundle for primitive generation or rendering
  void                setUpForPrimGen(SoAction *action,
                                      SbBool setUpDefault);
  void                setUpForGLRender(SoAction *action);

  // Sets up for default texture coordinate generation
  void                setUpDefaultCoordSpace(SoAction *action);

  // Sets up bundle for primitive generation or rendering for meshes
  void                setUpForPrimGenMesh(SoAction *action,
                                          SbBool setUpDefault);
  void                setUpForGLRenderMesh(SoAction *action);

  // Sets up for default texture coordinate generation for meshes
  void                setUpDefaultCoordSpaceMesh(SoAction *action);
  
  void setupTextureCoordinateElement(SoState* state, SoNode* currentNode) const;
  
  std::unique_ptr<TexGenEvaluatorBundle3> m_texGenEvaluator;
};

#endif /* _SO_TEXTURE_COORDINATE3_BUNDLE */


