/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_TEXTURE_COORDINATE_BUNDLE
#define  _SO_TEXTURE_COORDINATE_BUNDLE

#include <Inventor/bundles/SoBundle.h>
#include <Inventor/elements/SoTextureCoordinateElement.h>
#include <Inventor/misc/SoState.h>
#include <memory>

class TexGenEvaluatorBundle;

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoTextureCoordinateBundle
//
//  Bundle that allows shapes to deal with texture coordinates more
//  easily. This class provides a fairly simple interface to texture
//  coordinate handling, including default texture coordinate
//  generation. This can be used during either rendering or primitive
//  generation.
//
//  This class can be used during either rendering or primitive
//  generation. For primitive generation, there are two cases,
//  distinguished by the flag returned by isFunction(). If this
//  flag is TRUE, the texture coordinates are to be generated using
//  the get(point, normal) method, which uses a software texture
//  coordinate function. (This process is also used for texture
//  coordinates that are generated by default when necessary - in this
//  case, the function does a linear map across two sides of the
//  bounding box of the shape.) If the isFunction() flag is FALSE, the
//  coordinates are accessed directly from the element using the
//  get(index) method.
//
//  For GL rendering, there is an additional case. If
//  needCoordinates() returns FALSE, no texture coordinates need to be
//  sent at all, and the bundle does not have to be used for anything
//  else. Otherwise, send(index) should be used.
//
//////////////////////////////////////////////////////////////////////////////
/** [OIV-WRAPPER-CLASS NO_WRAP] */
SoEXTENDER class INVENTOR_API SoTextureCoordinateBundle : public SoBundle {

 public:
  // Constructor - takes the action the bundle is used for and a
  // flag to indicate whether the bundle is being used for
  // rendering. If this is TRUE, the bundle can be used to send
  // texture coordinates to GL. If it is FALSE, the setUpDefault
  // flag (default TRUE) indicates whether to set up a texture
  // coordinate function if the binding is DEFAULT. Shapes can pass
  // FALSE here if they are picking and want to delay computation of
  // the texture coordinates until an intersection is found.
  // texUnits is the eventual list of texture units where texture
  // coordinates should be generated. If not set the current texture unit is used.
  SoTextureCoordinateBundle(SoAction *action, SbBool forRendering,
                            SbBool setUpDefault = TRUE, const SbIntList *texUnits=NULL);

  // This constructor is used for meshes such as VEMLElevationGrid or
  // QuadMesh. The constructor calls different methods to insure
  // that the texture doesn't get mapped edgeon. The last argument is
  // just to make the methods unique.
  // texUnits is the eventual list of texture units where texture
  // coordinates should be generated. If not set the current texture unit is used.
  SoTextureCoordinateBundle(SoAction *action, SbBool forRendering,
                            SbBool setUpDefault, SbBool forMesh,
                            const SbIntList *texUnits=NULL);
  // Destructor
  ~SoTextureCoordinateBundle();

  // Returns TRUE if texture coordinates are needed at all
  SbBool              needCoordinates() const { return needCoords; }

  // return value to determine which get() to use.
  SbBool              isFunction() const      { return isFunc; }

  // Returns texture coordinate computed by function during
  // primitive generation or rendering
  SbVec4f             get(const SbVec3f &point, const SbVec3f &normal) const
  { return texCoordElt->get(point, normal, m_curUnit); }

  // Returns indexed texture coordinate during primitive generation
  // or rendering
  SbVec4f             get(int index) const
  { if (tCoords) 
      return(SbVec4f(tCoords[index][0],tCoords[index][1],
                     0.0, 1.0)); 
    else 
      return texCoordElt->get4(index, m_curUnit); 
  }

  // Sends indexed texture coordinate to GL during rendering
  void                send(int index) const;

 private:
  // TextureCoordinate elements:
  const SoTextureCoordinateElement    *texCoordElt;
  const SoTextureCoordinateElement  *GLTexCoordElt;

  SbBool              needCoords;     // Texture coordinates are needed
  SbBool              isFunc;         // Coordinates generated by function
  SbBool              isRendering;    // Bundle being used for rendering
  SbBool              m_statePushed ;  // Indicate if the state has been pushed

  // These indicate the dimensions used for S and T for default
  // texture coordinate generation
  int                 coordS, coordT;
  // These hold the vectors used for default texture coordinate generation
  SbVec4f             sVector, tVector;
  // This holds the texture coords from a vertexProperty node:
  const SbVec2f *     tCoords;

  // The eventual list of texture units where texture
  // coordinates should be generated. If not set the current texture unit is used.
  const SbIntList *m_texUnits ;

  // First texture image unit
  int              m_unitImage ;

  // Current texture unit
  int              m_curUnit ;

  // Sets up bundle for primitive generation or rendering
  void                setUpForPrimGen(SoAction *action, SbBool setUpDefault);
  void                setUpForGLRender(SoAction *action);

  // Sets up for default texture coordinate generation
  void                setUpDefaultCoordSpace(SoAction *action);

  // Sets up bundle for primitive generation or rendering for meshes
  void                setUpForPrimGenMesh(SoAction *action, SbBool setUpDefault, SbBool forMesh);
  void                setUpForGLRenderMesh(SoAction *action, SbBool forMesh);

  // Sets up for default texture coordinate generation for meshes
  void                setUpDefaultCoordSpaceMesh(SoAction *action, SbBool forMesh);

  // Update texture units used
  void updateTexUnits(const SbIntList *texUnits) ;

  /** Set texgen on state */
  void setupTextureCoordinateElement(SoState* state, SoNode* currentNode) const;
  
  std::unique_ptr<TexGenEvaluatorBundle> m_texGenEvaluator;
};

#endif /* _SO_TEXTURE_COORDINATE_BUNDLE */


