/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/



#ifndef  _SO_BOUNDING_BOX_CACHE
#define  _SO_BOUNDING_BOX_CACHE

#include <Inventor/SbBox.h>
#include <Inventor/caches/SoCache.h>

////////////////////////////////////////////////////////////////////////
//
// @ingroup Caches
//
//  Class SoBoundingBoxCache:
//
//  A bounding box cache stores a bounding box and center point. It
//  also stores a flag indicating whether the box contains any line or
//  point objects, meaning that intersection tests with the box should
//  use a little extra leeway.
//
////////////////////////////////////////////////////////////////////////

SoEXTENDER class INVENTOR_API SoBoundingBoxCache : public SoCache {

 public:
  // Constructor and destructor
  SoBoundingBoxCache(SoState *state);
  virtual ~SoBoundingBoxCache();

  // Stores a bounding box and center point in the cache
  void                set(const SbXfBox3f &boundingBox,
                          SbBool centerSet,
                          const SbVec3f &centerPoint);
  /** Returns the current bbox value */
  const SbXfBox3f& getBox() const
  { return m_box; }

  /** Returns BBox in local space */
  SbBox3f getProjectedBox() const
  { return m_box.project(); }

  // Returns TRUE if the center is valid.
  SbBool isCenterSet() const
  { return m_data.m_centerSet; }


  // Returns the center (in object space)
  const SbVec3f &     getCenter() const       { return m_center; }

  // Sets the hasLinesOrPoints flag to TRUE in all open bounding box
  // caches in the given state. (The flag is FALSE by default.)
  static void         setHasLinesOrPoints(SoState *state);

  // Sets the has screen space bounding box flag.
  // This is used by some nodes to indicate that if the action does not
  // ignore the real ScreenSpace boudingBox then the cache is invalid.
  static void         setHasScreenSpaceBoundingBoxes(SoState *state, bool flag);

  // Returns the hasLinesOrPoints flag.
  SbBool hasLinesOrPoints() const
  { return m_data.m_hasLOrP; }

  // Returns the hasScreenSpaceBoundingBoxes flag.
  bool hasScreenSpaceBoundingBoxes() const
  { return m_data.m_hasScreenSpaceBoundingBoxes; }

  virtual SbBool isValid(const SoState *state) const;

 SoINTERNAL public:
 
  // Sets the been generated with Ignore screen space flag.
  // This is used by some nodes to indicate the value of
  // SoGetBoundingBoxAction::ignoreScreenSpaceBoundingBoxes
  // when the cache was generated.
  void
  setBeenGeneratedWithIgnoreScreenSpaceBb( bool flag )
  {
    m_data.m_beenGeneratedWithIgnoreScreenSpaceBb = flag;
  }

 private:

  bool isValid_(const SoState *state) const;

  SbXfBox3f           m_box;            // Bounding box
  SbVec3f             m_center;         // Center point

  struct Flags
  {
    unsigned char m_centerSet                   : 1; // 1 if the center was set
    unsigned char m_hasLOrP                     : 1; // 1 if contains lines or points
    unsigned char m_hasScreenSpaceBoundingBoxes : 1; // 1 if contains screenspace bbox
    unsigned char m_beenGeneratedWithIgnoreScreenSpaceBb : 1; // 1 if the cache has been initialized with ignoreScreenSpaceBb
  };
  Flags m_data;
};

#endif /* _SO_BOUNDING_BOX_CACHE */


