/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#ifndef _SO_BUFFER_OBJECT_CACHE_H_
#define _SO_BUFFER_OBJECT_CACHE_H_

#include <Inventor/SbBase.h>
#include <Inventor/STL/map>
#include <Inventor/STL/list>

class SoBufferObject;
class SoCpuBufferObject;
class SbThreadSpinlock;

#ifdef _WIN32
#pragma warning( push )
#pragma warning(disable:4251)
#endif

/**
 * @VSGEXT CPUBufferObject cache management class.
 *
 * @ingroup Caches
 * 
 * @DESCRIPTION
 *
 *  A base class used to manage a simple cache of SoCpuBufferObject objects.
 *
 *  SoBufferObject creates an instance of this class that is primarily used
 *  for the LDM tile cache (see SoLDMGlobalResourceParameters for more information).
 *  The buffer object cache can be accessed using the SoBufferObject static method
 *  getBufferObjectCache.  Default size is 50.
 *
 *  This class was added in Open Inventor 8.5.
 *
 */
SoINTERNAL class INVENTORBASE_API SoCpuBufferObjectCache
{
 public:
   
  /** 
   * Remove all unused buffers (ref count = 0) from the cache.
   */
  virtual void flush();

  /**
  * Resize cache to the given size.
  * @param newSize new size for the cache (number of buffers). @BR
  * Default is 50.
  */
  virtual void resize( const size_t newSize );

 /**
  * Return the max size of the cache (number of buffers).
  */
  virtual size_t getSize( );

 SoINTERNAL public:
  /**
   * Constructor.
   */
  SoCpuBufferObjectCache();

  /** Destructor */
  virtual ~SoCpuBufferObjectCache();
  
  /**
  * Get a pointer to a buffer in the cache and add a reference to that buffer,
  * indicating the buffer is in use. @BR
  * Returns NULL if bufferId does not match any buffer in the cache.
  * If bufferId is valid, increments the buffer's ref count and returns its address.
  * Therefore a call to this function guarantees that, if present in cache, the 
  * buffer cannot be removed from the cache until releaseBufferRef() has been called
  * and the ref count reaches zero.  To get a point to a buffer without incrementing
  * the ref count use the getBuffer() method.
  *
  * @param bufferId of buffer to retrieve.
  */
  virtual SoCpuBufferObject* getBufferRef( const uint64_t bufferId );

  /** 
   * Add the given buffer to the cache. @BR
   * If the buffer was successfully inserted in the cache, returns a unique
   * non-zero id that can be used with the other methods in this class and 
   * initializes the buffer's ref count to 1.  Therefore the buffer cannot
   * be removed from the cache until releaseBufferRef() has been called. @BR
   * Returns 0 if the insertion failed.
   * Put it's id at the end of the LRU list. If the cache is full, the least
   * recently used is deleted (first one in the list).
   *
   * @param buffer object to be added in the cache
   */
  virtual uint64_t addInCache( SoCpuBufferObject* bufferObject );

  /**
  * Release a reference to a buffer in the cache, indicating the buffer is no longer
  * in use. @BR
  * Returns false if the specified buffer was not found in the cache or its ref count
  * is already zero, else returns true and decrements the buffer's ref count. A buffer
  * cannot be removed from the cache until its ref count reaches zero (addInCache sets
  * the initial ref count to 1). Even when its ref count reaches zero, the buffer will
  * remain in the cache until internal management removes it, for example because the
  * space is needed for a new buffer.  This method must be called when the buffer is
  * no longer in use, otherwise it will never be removed from the cache.
  *
  * @param bufferId of buffer to release.
  */
  virtual bool releaseBufferRef ( const uint64_t bufferId );

  /**
  * Get a pointer to a buffer in the cache without incrementing the ref count. @BR
  * Returns NULL if the buffer Id does not match any buffer in cache. Warning: This 
  * method must be called between getBufferRef/addInCache and releaseBufferRef 
  * to guarantee the buffer is valid
  *
  * @param bufferId of buffer to retrieve.
  */
  SoCpuBufferObject* getBuffer(const uint64_t bufferId);

  // internal perf counters access
  void getInternalCounter(  uint64_t &nbCacheAccess, uint64_t& nbCacheDefault);
  void resetInternalCounter( );

  // Force a delete from cache (mainly used as a double checked in SoCpuBufferCache destructor
  // costly because we have to search for the bufferId first.
  virtual bool deleteFromCache( const uint64_t bufferId );

  // Retrieve bufferId from its signature. 
  uint64_t getMappingIdfromSig(const uint64_t& cacheSig);

private:

  // implementation of getBuffer for getBuffer and getBufferRef
  // notes it is upd to the caller to lock the cacheList mutex
  SoCpuBufferObject* getBufferInternal(const uint64_t bufferId);

  // internal cache management
  typedef std::map<uint64_t, SoCpuBufferObject*> cacheListType;
  typedef std::map<uint64_t, uint64_t> bufferSignatureMap;
  typedef std::list<uint64_t> lruBufferIdsType;

  cacheListType m_cacheList;
  bufferSignatureMap m_bufferIdToSigMap;
  bufferSignatureMap m_bufferSigToIdMap;
  // List of sorted ids. The last one is the last used one.
  lruBufferIdsType lruBufferIds;

  // Max size of the cache
  size_t m_maxCacheSize;

  // Uniq Id management
  uint64_t getUniqueId();
  uint64_t m_lasUniqId;

  // Remove the least recently used buffer.
  void makeRoomInCache();
  
  // Spinlock used to protect cache access
  SbThreadSpinlock *m_cacheListLock;

  // perf counter manager
  uint64_t m_perfNbCacheAccess ;
  uint64_t m_perfNbCacheDefault;
};

#ifdef WIN32
#pragma warning( pop )
#endif

#endif //_SO_BUFFER_OBJECT_CACHE_H_


