/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
 **=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#pragma once

#include <Inventor/components/stereo/SoBaseStereo.h>
#include <Inventor/ViewerComponents/SoInterlacedStereoParameters.h>

class SoStereoViewer;

/**
 * @VSGEXT Class for interlaced stereo support.
 *
 * @ingroup Stereo
 *
 * @DESCRIPTION
 *   This class implements the interlaced stereo type. Using the interlaced mode, the
 *   stereo effect is achieved using special devices (projectors,...) that translate
 *   an interlaced image into a stereoscopic view. Alternating scan lines display
 *   images for the left and right eyes.
 *
 *   The interlaced mode can work in two modes: fast or best. This is simply a matter
 *   of quality. Due to the interlacing, lines close to horizontal can appear as
 *   dotted lines. In best mode, those parts of the model are drawn as if they were
 *   thicker. By default, the mode is fast.
 *
 * @SEE_ALSO
 *    SoStereoViewer,
 *    SoBaseStereo,
 *    SoWinViewer
 *
 * [OIVJAVA-WRAPPER-CLASS NO_WRAP]
 */
class INVENTORGUI_API SoInterlacedStereo : public SoBaseStereo
{
public:
  /** Interlaced modes */
  enum InterlacedMode
  {
    /** Horizontal (default) */
    HORIZONTAL = 0, // default
    /** Vertical */
    VERTICAL
  };

  /**
   * Constructor.
   */
  SoInterlacedStereo( SoStereoViewer* stereoVwr = NULL, SoGuiViewer* guiViewer = NULL );

  ~SoInterlacedStereo();

  /**
   * Sets interlaced mode. Default is HORIZONTAL.
   */
  void setInterlacedMode( InterlacedMode mode ) { m_parameters->setInterlacingMode( (SoInterlacedStereoParameters::InterlacedMode)mode ); }

  /**
   * Queries interlaced mode.
   */
  InterlacedMode getInterlacedMode() const { return (SoInterlacedStereo::InterlacedMode)m_parameters->getInterlacingMode(); }

  /**
   * Sets screen interlacing.
   * If screen is TRUE (default), the line interlacing is done according to the
   * screen (if the window is moving, the first line might be either an even or an
   * odd line). Otherwise, do the interlacing according to the window itself (first
   * line is always even).
   */
  void setScreenInterlacing( SbBool screen ) { m_parameters->setReverseInterlacing( screen ); }

  /**
   * Queries screen interlacing.
   */
  SbBool isScreenInterlacing() const { return m_parameters->isReverseInterlacing(); }

  /**
   * Sets fast interlacing mode. Default is TRUE.
   */
  void setFastInterlacing( SbBool fast = TRUE ) { m_parameters->setQualityRendering( fast == FALSE ); }

  /**
   * Queries fast interlacing mode.
   */
  SbBool isFastInterlacing() const { return !m_parameters->getQualityRendering(); }

  /**
   * Always returns SoBaseStereo::INTERLACED_STEREO.
   */
  virtual StereoViewType getStereoViewType() { return INTERLACED_STEREO; }

  /**
   * Enable image adjustment specific to Sharp LCD displays.
   */
  void enableGreenShift( SbBool enable ) { m_parameters->setGreenShift(enable); }

  /**
   * Query green shift setting
   */
  SbBool isGreenShiftEnabled() const { return m_parameters->getGreenShift(); }

SoINTERNAL public:
  void doLeftEyeSettings( SoGLRenderAction* action );
  void doRightEyeSettings( SoGLRenderAction* action );
  void doMonoscopicSettings( SoGLRenderAction* action );

private:
  SoInterlacedStereoParameters* m_parameters;
};
