/*=======================================================================
** VSG_COPYRIGHT_TAG
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Aug 2013)
**=======================================================================*/

#if !defined(_SO_AUTO_MAP_H_)
#define _SO_AUTO_MAP_H_

#include <Inventor/devices/SoLocalCpuBufferObject.h>

/**
* @VSGEXT buffer object mapping helper class.
*
* @ingroup CpuDevice
*
* @DESCRIPTION
*
* This class provides an safe and simplified way to access data through typed a cpu array.
* It allows to use the following syntax while app wants to read or write some data from an SoBufferObject.
* Example:
* \par
* \code
*    void updateABufferObject(SoBufferObject* bufferObject);
*    {
*      SoAutoMap<SbVec3f> vec3array(bufferObject,SoBufferObject::SET);
*      for (int i=0;i<vec3array.getSize();++i)
*        vec3array[i] = SbVec3f(0.f,0.f,0.f);
*    }
*
*    // which replace and add safety to the follwoing code
*    void updateABufferObject(SoBufferObject* bufferObject);
*    {
*      SoRef<SoCpuBufferObject> cpuBufferObject = new SoCpuBufferObject;
*      bufferObject->map(cpuBufferObject,SoBufferObject::SET);
*      SbVec3f* vec3array = static_cast<SbVec3f*>(cpuBufferObject->map(SoBufferObject::SET));
*      size_t arraySize=cpuBufferObject.getSize()/sizeof(SbVec3f);
*      for (int i=0;i<arraySize;++i)
*        vec3array[i] = SbVec3f(0.f,0.f,0.f);
*      cpuBufferObject->unmap();
*      bufferObject->unmap(cpuBufferObject);
*    }
* \endcode
*
* [OIV-WRAPPER-NO-WRAP]
*/
template<typename T>
class SoAutoMap
{
public:

  /** Constructor. In charge of setting mapping */
  SoAutoMap(SoBufferObject* bufferObject, SoBufferObject::AccessMode accessMode)
    :m_bufferObject(bufferObject)
  {
    m_bufferObject->map(&m_cpuBufferObject,accessMode);
    m_pointer = static_cast<T*>(m_cpuBufferObject.map(accessMode));
    m_arraySize = bufferObject->getSize()/sizeof(T);
  }

  /** Destructor. In charge of releasing mapping */
  ~SoAutoMap()
  {
    m_cpuBufferObject.unmap();
    m_bufferObject->unmap(&m_cpuBufferObject);
  }

  /** accessor to T element at index i */
  T& operator[](const size_t i)
  {
    assert(i<m_arraySize);
    return (m_pointer[i]);
  }

  /** const accessor to T element at index i */
  const T& operator[](const size_t i) const
  {
    assert(i<m_arraySize);
    return (m_pointer[i]);
  }

  /** return the number of T element in the mapped array */
  size_t getSize() const
  { return m_arraySize; }

private:
  // private default constructor to avoid copy by reference
  SoAutoMap(){}

  // cpu bufferobject to keep track of the current mapping
  SoLocalCpuBufferObject m_cpuBufferObject;

  // keep track of the source mapped buffer object
  SoBufferObject* m_bufferObject;

  // keep track of the pointer target of the mapping
  T* m_pointer;
  
  // keep track of the buffer size in number of element
  size_t m_arraySize;
};

#endif // _SO_AUTO_MAP_H_
