/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#pragma once

#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoCpuContext.h>

SO_PIMPL_PUBLIC_DECLARATION(SoCpuBufferObject)

/**
 * @VSGEXT CPU buffer object class
 *
 * @ingroup CpuDevice
 *
 * @DESCRIPTION
 *
 * This class provides management functions for CPU memory buffers.
 *
 * By default memory allocations have the maximum possible alignment to allow use
 * with (for example) SSE instructions.
 *
 * Memory is managed using the new/delete operators. On Microsoft Windows platforms
 * it is possible to use VirtualAlloc/VirtualFree instead by setting
 * OIV_BUFFER_USE_VIRTUAL_ALLOC (see SoPreferences).
 *
 * See SoBufferObject for general information about buffer objects.
 *
 * See SoBufferedShape for an example of storing vertices in a CPU buffer.
 *
 * @if_cpp
 * @B NOTE: @b @BR
 * SoBufferObject classes are reference counted objects (just like nodes, 
 * paths, etc).  Therefore you cannot create an SoBufferObject on the 
 * stack. And to delete an SoBufferObject you must ref and unref the 
 * object.  The SoRef smart pointer is a safe and convenient way to
 * create buffer objects. See examples in SoBufferObject.
 * \endif
 *
 * @EXAMPLE
 * Load data into a buffer object from an array in memory:
 * \if_cpp
 *   \code
 *   const float vertices[][3] = {
 *        1,0.5,0, 0,1,0, -1,0.5,0,
 *       -1,-1,0, 1,-1,0, 1,0,0, -1,0,0,
 *       -1,-1.5,0, 1,-1.5,0
 *   };
 *   const int NUM_COORDS = sizeof(vertices) / sizeof(SbVec3f);
 *   \endcode
 *   Wrap coordinate array in a CPU buffer object:
 *   \code
 *   SoRef<SoCpuBufferObject> vertBuf = 
 *       new SoCpuBufferObject( (void*)vertices, NUM_COORDS * sizeof(SbVec3f) );
 *   \endcode
 *   Or allocate memory and copy data into buffer:
 *   \code
 *   SoRef<SoCpuBufferObject> vertBuf = new SoCpuBufferObject();
 *   vertBuf->setSize( NUM_COORDS * sizeof(SbVec3f) );
 *   SbVec3f* vertPtr = (SbVec3f*)vertBuf->map( SoBufferObject::SET );
 *       memcpy( vertPtr, vertices, NUM_COORDS * sizeof(SbVec3f) );
 *   vertBuf->unmap();
 *   \endcode
 * \endif
 * \if_dotnet
 *   \par
 *   \code
 *   float[] vertices = new float[9 * 3] {
 *        1.0f, 0.5f,0.0f, 0.0f, 1.0f,0.0f, -1.0f,0.5f,0.0f,
 *       -1.0f,-1.0f,0.0f, 1.0f,-1.0f,0.0f,  1.0f,0.0f,0.0f, -1.0f,0.0f,0.0f,
 *       -1.0f,-1.5f,0.0f, 1.0f,-1.5f,0.0f
 *   };
 *
 *   // Wrap coordinate array in a CPU buffer object
 *   SbNativeArray<float> vertArray = new SbNativeArray<float>(vertices);
 *   SoCpuBufferObject vertBuf = new SoCpuBufferObject( (SbNativeArray<byte>)vertArray );
 *   \endcode
 * \endif
 * \if_java
 *   \par
 *   \code
 *   float[] vertices = {
 *           1.0f, 0.5f,0.0f, 0.0f, 1.0f,0.0f, -1.0f,0.5f,0.0f,
 *          -1.0f,-1.0f,0.0f, 1.0f,-1.0f,0.0f,  1.0f,0.0f,0.0f, -1.0f,0.0f,0.0f,
 *          -1.0f,-1.5f,0.0f, 1.0f,-1.5f,0.0f
 *   };
 *
 *   // Create a CPU buffer object and set its size (allocate memory)
 *   SoCpuBufferObject cpuBuffer = new SoCpuBufferObject();
 *   cpuBuffer.setSize( vertices.length * Float.SIZE/8 );
 *
 *   // Copy vertex data into the buffer object
 *   FloatBuffer vertData = cpuBuffer.map( SoBufferObject.AccessModes.SET ).asFloatBuffer();
 *   vertData.put(vertices);
 *   cpuBuffer.unmap();
 *   \endcode
 * \endif
 *
 * @EXAMPLE
 * Access data stored in a buffer object.
 * \par
 * \if_cpp
 *   \code
 *   unsigned char* data  = (unsigned char*)cpuBuffer->map( SoBufferObject::READ_ONLY );
 *   unsigned char  value = data[0];
 *       . . .
 *   cpuBuffer->unmap();
 *   \endcode
 * \endif
 * \if_dotnet
 *   \code
 *   SbNativeArray<byte> data = (SbNativeArray<byte>)cpuBuffer.Map( SoBufferObject.AccessModes.READ_ONLY );
 *   byte value = data[0];
 *       . . .
 *   cpuBuffer.Unmap();
 *   \endcode
 * \endif
 * \if_java
 *   \code
 *   ByteBuffer data = cpuBuffer.map( SoBufferObject.AccessModes.READ_ONLY );
 *   byte value = data.get(0);
 *       . . .
 *   cpuBuffer.unmap();
 *   \endcode
 * \endif
 *
 *
 * @SEE_ALSO
 * SoGLBufferObject
 *
 * [OIVNET-WRAPPER-CLASS CUSTOM_DISPOSE]
 */
class INVENTORBASE_API SoCpuBufferObject : public SoBufferObject
{
  SO_TYPED_CLASS_HEADER();
  SO_PIMPL_PUBLIC_HEADER(SoCpuBufferObject)

public:

  /**
   * Default constructor.
   * The contents of the buffer are initially empty.
   */
  SoCpuBufferObject();

  /**
   * Constructor that takes an existing block of CPU memory. @BR
   * This memory is owned, and should be freed by, the application.
   * \if_java
   * \par
   * The specified byte buffer must be a direct buffer with native
   * byte order else an IllegalArgumentException is thrown.
   * See below for an example of direct buffer allocation.
   * \code
   *   ByteBuffer buffer = ByteBuffer.allocateDirect(numBytes);
   *   buffer.order(ByteOrder.nativeOrder());
   *   buffer.put(bytes);
   *   SoCpuBufferObject cpuBuffer = new SoCpuBufferObject(buffer);
   * \endcode
   * @param buffer the buffer to manage.
   * \else
   * @param buffer Pointer to the buffer to manage.
   * @param size Size of buffer in bytes.
   * \endif
   * [OIVJAVA-WRAPPER-ARG ARRAY&NO_ALLOC_REF,NO_WRAP{buffer != null ? buffer.limit() : 0}]
   * [OIVJAVA-WRAPPER HELPER_END{refBuffer(buffer)}]
   */
  SoCpuBufferObject(void *buffer, size_t size);

  /**
   * Request that the buffer object manage an existing block of memory. @BR
   * The application still owns this memory and is responsible for releasing
   * the memory (when no SoCpuBufferObjects have a reference on it).
   *
   * We recommend to use the most aligned memory pointer possible to enable
   * optimized algorithm usage.
   * \if_java
   * \par
   * The specified byte buffer must be a direct buffer with native
   * byte order else an IllegalArgumentException is thrown.
   * See below for an example of direct buffer allocation.
   * \code
   *   ByteBuffer buffer = ByteBuffer.allocateDirect(numBytes);
   *   buffer.order(ByteOrder.nativeOrder());
   *   buffer.put(bytes);
   *   cpuBuffer.setBuffer(buffer);
   * \endcode
   * \endif
   *
   * NOTE: If another buffer object is currently mapped into another buffer, the
   * other buffer is automatically unmapped and its contents are undefined.
   * [OIVNET-WRAPPER CUSTOM_CODE]
   * [OIVJAVA-WRAPPER-ARG ARRAY&NO_ALLOC_REF,NO_WRAP{buffer != null ? buffer.limit() : 0}]
   * [OIVJAVA-WRAPPER HELPER_END{refBuffer(buffer)}]
   */
  void setBuffer(void *buffer, size_t size);

  /**
    * Create a new buffer with the same properties as the current one. @BR
    * The new instance will have the same context
    * or device properties, but no memory is allocated.
    */
  virtual SoBufferObject* createInstance() const;

  /**
   * This function clears the content of the buffer, it frees the memory if the mode was COPY.
   * [OIVNET-WRAPPER HELPER_END{OnClearInstance()}]
   */
  virtual void clearInstance();

SoINTERNAL public:

  /** Return pointer to readonly data. Data must not be changed */
  const void* getReadOnlyPointer() const;

  /**
   * Copy data from this buffer object to the specified CPU buffer object. @BR
   */
  virtual void copyToCpuBuffer( SoCpuBufferObject* targetBufferObject, size_t destOffset = 0, size_t sourceOffset = 0, size_t copySize = SO_BUFFER_SIZE_ALL );

  /**
   * This function indicates if the buffer memory was allocated by the object
   * or if it was set using the setBuffer funcion.
   */
  bool isBufferOwner() const;

  static void memcpyPtr2Ptr(void* dest, void* data, size_t dataSize, size_t count);

protected:
  /**
   * Destructor. @BR
   * Any memory allocated by the buffer object (e.g. by setSize()) is freed.
   */
  virtual ~SoCpuBufferObject();

};
