/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/
#if !defined SOGLCONTEXT_H
#define SOGLCONTEXT_H

#include <Inventor/devices/SoDeviceContext.h>
#include <Inventor/devices/SoGLFormat.h>
#include <Inventor/devices/SoGLGlew.h>
#include <Inventor/STL/vector>
#include <Inventor/STL/list>
#include <Inventor/STL/map>
#include <Inventor/helpers/SbGlContextHelper.h>
#include <Inventor/threads/SbThreadLocalStorage.h>
#include <Inventor/threads/SbThreadSpinlock.h>
#include <Inventor/errors/SoError.h>
#include <Inventor/image/SbRasterImage.h>
#include <Inventor/threads/SbThreadLocalStorage.h>


#include <Inventor/SbPList.h> // For SbStringList
#include <Inventor/STL/stack>

#include <Inventor/helpers/SbGPUCapabilities.h>

#if defined(_WIN32)
#include <windows.h>
#endif


#ifdef _MSC_VER
#pragma warning( push )
#pragma warning(disable:4251)
#endif

class SbThreadMutex;
struct GLEWContextStruct;

/*! \cond PRIVATE */
class FunctionRedirector;
/*! \endcond */

/**
* @VSGEXT OpenGL context management class.
*
* @ingroup GLDevice
*
* @DESCRIPTION
*
* This class provides functions to manage OpenGL device contexts.
*
* The application can control sharing of OpenGL contexts.
* When you share OpenGL contexts, any context in that shared group can use GPU resources
* (such as textures, buffers, etc.) created in any other context in the group.
* This saves time and memory when rendering into multiple windows, buffers, etc.
* We recommend using context sharing when scene graphs are shared between multiple viewers.
* This approach saves memory on the GPU and improves rendering performance.
* To create a shared gl context, use a SoGLContext constructor with the bool shared argument, and specify true.
*
* In a typical Open Inventor application, the viewer class automatically
* creates an OpenGL device context and manages it for rendering and other GPU-related operations.
* However, this context is only made current when the viewer is actively using it,
* for example, to render the scene graph.
*
* Do not assume that any OpenGL context is current. If you need a current
* OpenGL context, for example to create an SoGpuBufferObject, bind() and
* unbind() an SoGLContext object.
* 
* If your application uses an Open Inventor viewer, the viewer's rendering context
* can be made current using the viewer's bindNormalContext() 
* method and released using the viewer's unbindNormalContext() method. This
* context can also be queried using the viewer's getNormalSoContext() method.
*
* If you do not have (or do not have access to) a viewer, you can create a
* context by creating an SoGLContext object. In almost all cases, calling the
* constructor with a true value will create a context that is in a share group
* with the rendering context. In other words, resources such as textures and
* buffer objects are shared between these contexts and can be used in any of
* the contexts in the share group. For example:
* \if_cpp
*   \par
*   \code
*    SoRef<SoGLContext> glContext = new SoGLContext(true);
*    glContext->bind();
*        SoRef<SoGpuBufferObject>  gpuBuffer = new SoGpuBufferObject( SoGpuBufferObject::STATIC, SoGpuBufferObject::SHARED );
*    glContext->unbind();
*   \endcode
* \endif
* \if_dotnet
*   \par
*   \code
*   SoGLContext glContext = new SoGLContext(true);
*   glContext.Bind();
*       SoGpuBufferObject gpuBuffer = new SoGpuBufferObject( SoGpuBufferObject.BufferAccessFrequencies.STATIC, SoGpuBufferObject.BufferAccessNatures.SHARED );
*   glContext.Unbind();
*   \endcode
* \endif
* \if_java
*   \par
*   \code
*   SoGLContext glContext = new SoGLContext( true );
*   glContext.bind();
*       SoGpuBufferObject gpuBuffer = new SoGpuBufferObject( SoGpuBufferObject.BufferAccessFrequencies.STATIC, SoGpuBufferObject.BufferAccessNatures.SHARED );
*   glContext.unbind();
*   \endcode
* \endif
*
* If the OpenGL context is created and managed by the application or a third-
* party library, the application should create an SoGLContext object to wrap 
* the actual current context, then bind() and unbind() this object around the Open
* Inventor render call. Note that this method will return null if there is no
* current OpenGL context. Also note that the SoGLContext object should be told
* not to manage (specifically not to delete) the OpenGL context since its lifespan
* is managed by other software. For example:
* \if_cpp
*   \par
*   \code
*   SoGLContext* context = SoGLContext::getCurrent( true );
*   if (context != NULL)
*       context->setNoGLContextDelete(); // Context is managed by someone else
*   else
*       context = new SoGLContext( true );
*   context->bind();
*       . . .
*   context->unbind();
*   \endcode
* \endif
* \if_dotnet
*   \par
*   \code
*   SoGLContext context = SoGLContext::GetCurrent( true );
*   if (context != null)
*       context.SetNoGLContextDelete(); // Context is managed by someone else
*   else
*       context = new SoGLContext( true );
*   context.Bind();
*       . . .
*   context.Unbind();
*   \endcode
* \endif
* \if_java
*   \par
*   \code
*   if (context != null)
*       context.setNoGLContextDelete(); // Context is managed by someone else
*   else
*       context = new SoGLContext( true );
*   context.bind();
*       . . .
*   context.unbind();
*   \endcode
* \endif
*
* \if_cpp
* @B Reference counting:@b 
* - If an SoGLContext is bound when an SoGLBufferObject is created,
*   the buffer object will ref() the context (and unref() it when the buffer
*   object is destroyed).
* \endif*
*/
class INVENTORGL_API SoGLContext : public SoDeviceContext
{
public:

  /**
   * Sharing Policy. @BR
   * User can change sharing policy by defining the OIV_SHARE_LISTS environment variable.
   */
  enum SharedGroupPolicy
  {
    /** Disables context sharing (same behavior as v2.6 and older) */
    DISABLED,
    
    /**
     * Conservative sharing (not available on Unix) @BR
     * Only add context to a share group containing contexts with the
     * constraint (ex: same pixel format for SoGLContext objects).
     * This mode matchs the official behavior of the wglShareLists() function.
     * (Please refer to Microsoft documentation for more details.)
     */
    CONSERVATIVE,
    
    /**
     * Aggressive sharing (default) @BR
     * Tries to share contexts, independent of their sharing constraint.
     * When sharing succeeds, shared contexts are part of the same sharedGroup,
     * else, a new sharedGroup is created.
     */
   AGGRESSIVE
  };

  /**
   *  Constructor which creates an SoGLContext based on the attributes of the current context. @BR
   *  If the current context is NULL, the function tries to create a context on a temporary window.
   *  If this doesn't work, the resulting SoGLContext will not be valid!
   *
   *  @param shared Indicates if we want to share the context. If TRUE, context is shared with
   *  current context, depending on the currently defined SharedGroupPolicy. If FALSE, context is
   *  explicitly not shared.
   */
  SoGLContext( bool shared );

  /** 
   * Constructor which creates an SoGLContext based on the specifed parameters.
   *
   * On Microsoft Windows platforms the Display, VisualInfo and GLContext parameters
   * cannot be NULL. @BR
   * On Linux platforms the Display, Drawable, VisualInfo and GLContext parameters cannot
   * be NULL. @BR
   * On Mac platforms the Drawable, VisualInfo and GLContext parameters cannot
   * be NULL. @BR
   * Display is equal to zero for the first display, 1 for the second, ...
   */
  SoGLContext( SbGlContextHelper::Display dpy,
               SbGlContextHelper::VisualInfo vis,
               SbGlContextHelper::Drawable drawable,
               SbGlContextHelper::GLContext ctx );

  /** 
   * Constructor which creates a SoGLContext based on the specified parameters.
   *
   * The specified format must have been acticated before using it with this contructor.
   * The GLContext paramater cannot be NULL as long as this constructor is not supposed 
   * to create an OpenGL context but use an existing one. If the constructor must create a
   * context, you must use the constructor which takes only a SoGLFormat or the one which takes
   * a SoGLContext and a SoGLFormat.
   */
  SoGLContext( const SoGLFormat& format, SbGlContextHelper::GLContext ctx, SbGlContextHelper::Drawable drawable = 0 );

  /**
   * Constructor which creates a SoGLContext based on the specifed format. @BR
   * Note: Context will be shared with other contexts, depending on the currently
   * defined SharedGroupPolicy
   *
   * The specified format must have been activated before using it with this contructor.
   */
  SoGLContext( const SoGLFormat& format, SbGlContextHelper::Drawable drawable = 0 );

  /** 
   * Constructor which creates a SoGLContext based on the specified parameters. @BR
   * Note: Context will be shared with other contexts, depending on the currently defined
   * SharedGroupPolicy.
   *
   * On Microsoft Windows platforms the Display and VisualInfo parameters
   * cannot be NULL. @BR
   * On Linux platforms the Display, Drawable and VisualInfo parameters cannot
   * be NULL. @BR
   * On Mac platforms the Drawable and VisualInfo parameters cannot
   * be NULL. @BR
   * Display is equal to zero for the first display, 1 for the second, ...
   *
   * @param dpy The display/device context.
   * @param vis The visual/pixel format descriptor for the rendering context.
   * @param drawable The drawable (must be null under Windows).
   * @param shared Indicates if we want to share the context. If TRUE, context is shared with
   * one of the already created contexts, depending on the currently defined SharedGroupPolicy.
   * If FALSE, context is explicitly not shared.
   */
  SoGLContext( SbGlContextHelper::Display dpy,
               SbGlContextHelper::VisualInfo vis,
               SbGlContextHelper::Drawable drawable,
               bool shared = true );

  /**
   * Constructor which creates an SoGLContext based on the attributes of the specified context. 
   *
   * If @I shared @i is true then the created OpenGL context will be shared with the given context,
   * depending on the current SharedGroupPolicy.
   */
  SoGLContext( SoGLContext* context, bool shared );

  /**
   * Constructor which creates a SoGLContext shared with a specific context, but attached to
   * the display specified in the SoGLFormat.
   *
   * @param context The SoGLContext to be share with. Cannot be NULL.
   * @param format The format used for the drawable used by the new context.
   * @param drawable The drawable on which the context is created. Use the one from the context parameter if NULL is provided.
   */
  SoGLContext( SoGLContext* context, const SoGLFormat& format, SbGlContextHelper::Drawable drawable = 0 );
  
  /**
   * Bind the OpenGL context to the current thread.
   * This call should be followed by a call to unbind() once the context is no longer used.
   */
  virtual void bind();


  /**
   * Try to bind the OpenGL context to the current thread. @BR
   * Returns false if already bound, else calls bind() and returns true.
   */
  virtual bool tryBind();

  /**
   * Unbind the OpenGL context from the current thread.
   */
  virtual void unbind();

  /**
   * Returns true if the specified context and this context are shared.
   *
   * @return true if the two contexts are shared.
   */
  bool isSharedWith( const SoDeviceContext* context ) const;

  /** @copydoc SoDeviceContext::isValid */
  virtual bool isValid() const;

  /**
   * Set this context as invalid so it won't be used anymore.
   * This is mainly useful when the SoGLContext has been created from
   * an OpenGL context managed by the application or a third-party library (e.g.: Qt, JOGL...).
   */
  void invalidate();

  /**
   * Returns the internal id for this context.
   * The internal id is unique, it's not possible to get two contexts with the same id.
   *
   * @return The internal id of the context.
   */
  int getId() const;

  /**
   * Returns true if the specified context and this one are compatible.
   * Compatible means that using a buffer from one of these contexts inside
   * the other will not require copying the data.
   * If two OpenGL contexts are compatible, they either have the same id or they are shared.
   *
   * @param context The context to compare with this one.
   * @return True if the specified context is compatible with this one.
   */
  virtual bool isCompatible( SoDeviceContext* context ) const;

  /** 
   * Returns true if this context is valid and currently active.
   * If the current context is shared with this context then it returns true.
   */
  virtual bool isCurrent() const;

  /**
   * Returns the current active OpenGL context (if any).
   *
   * This method returns the current active OpenGL context from
   * the Open Inventor state. However it can also get the actual
   * current context from OpenGL.
   * It may be useful if Open Inventor should render using an OpenGL context
   * created by the application or a third-party library. In that case,
   * Open Inventor will create a new SoGLContext using contextId returned
   * by system getCurrentContext() method.
   *
   * @param checkGLState This param is no longer used.
   *
   * @return The current OpenGL context (or NULL if there isn't one).
   */
  static SoGLContext* getCurrent( bool checkGLState = false );

  /**
   * Returns the OpenGL rendering context of this SoGLContext.
   *
   * @return The OpenGL rendering context.
   */
  const SbGlContextHelper::GLContext& getGLContext() const;

  /** 
   * Returns the first SoGLContext associated with the specified OpenGL context.
   *
   * @return The found SoGLContext or NULL if none found.
   */
  static SoGLContext* findGLContext( SbGlContextHelper::GLContext );

  /** 
   * Returns an SoGLContext that matches the specified SharedGroupDescription.
   * On Windows, SharedGroupDescription is a PixelFormat id. @BR
   * On Unix, SharedGroupDescription is a display pointer.
   */
#if defined(_WIN32)
  typedef int SharedGroupDescription;
#else
  typedef SbGlContextHelper::Display SharedGroupDescription;
#endif
  static SoGLContext* findSharedContext( SharedGroupDescription );

  /**
   * Returns the Display of this SoGLContext.
   *
   * @return The Display.
   */
  const SbGlContextHelper::Display& getDisplay() const;

  /**
   * Returns the VisualInfo of this SoGLContext.
   *
   * @return The VisualInfo.
   */
  const SbGlContextHelper::VisualInfo& getVisualInfo() const;

  /**
   * Returns the SoGLFormat associated to the SoGLContext.
   *
   * @return the SoGLFormat.
   */

  const SoGLFormat& getFormat() const;

  /**
   * Returns the context corresponding to an internal id.
   *
   * @param id The internal id used to search for the context.
   * @return The context which has the specified id.
   */
  static SoGLContext* getContextFromId( int id );

  /** 
   * Returns the first context that belongs to the specified sharedIdGroup.
   *
   * @param sharedIdGroup The internal sharedIdGroup used to search for the context.
   * @return The first context that belongs to the specified sharedIdGroup.
   */
  static SoGLContext* getContextFromSharedId( int sharedIdGroup );

  /**
   * Returns true if the context is the current active context or
   * if it is shared with the current active context.
   */
  bool isValidForCurrent() const;

  /**
   * Assert this context and the current active context are the same.
   */
  void assertContext() const;

  /**
  * Swaps the buffers with the value stored in the SoGLFormat, which is set to the main plane by default.
  */
  bool swapBuffers();

  /** 
   * Returns an SbString containing the supported extensions for this context.
   *
   * @B Note:@b SbStringList is a list of pointers, not a list of objects, so
   * the destructor for this class does @I not@i free the memory associated with
   * the SbString objects.  The application is responsible for deleting each object
   * in the list.
   */
  SbStringList getSupportedExtensions();

  /** Returns the current sharedGroupPolicy */
  static SharedGroupPolicy getSharedGroupPolicy();

  /**
   * Prevent deletion of the native OpenGL context by SoGLContext.
   * 
   * Note: Can be useful when the OpenGL context is created and managed by
   *       the application or a third-party library such as Qt.
   */
  void setNoGLContextDelete() { m_noOGLContextDelete = true; }

  /**
   * Retrieve graphics capabilities from this context.
   * Make sure that this context is valid, otherwise capabilities will be wrong.
   */
  const SbGPUCapabilities& getContextGraphicsCapabilities();

  /**
   * Retrieve graphics capabilities from the current bound context, if any.
   * A tempory context is created with default values if no context is bound.
   */
  static const SbGPUCapabilities& getGraphicsCapabilities();

SoINTERNAL public:

  SB_THREAD_TLS_HEADER();

  static void initClass();
  static void exitClass();

  /** GPU type, used by getGPUVendorType() */
  enum GPUVendorType
  {
    GPU_TYPE_NOT_INITIALIZED = -1,
    GPU_NVIDIA = 0,
    GPU_ATI = 1,
    GPU_INTEL = 2,
    GPU_FIREPRO_MAC = 3
  };

  /* Returns the GPU vendor type */
  static GPUVendorType getGPUVendorType();

  /** 
   * Returns the current active OpenGL context from the OpenGL state. @BR
   * This function can be used to ensure a valid state when makeCurrent() 
   * calls are made by a third party API.
   * 
   * @param supposedContext This parameter is used to speed up the process,
   * when the context is supposed to be current. Mainly used by the getCurrent()
   * function to ensure that the context on the internal stack is the actual one.
   *
   * @return The actual SoGLContext according to the OpenGL state.
   */
  static SoGLContext* getActualCurrentContext( SoGLContext* supposedContext );

  /** This functions sets a user data. This function
    * should be used only by the viewers to store platform specific
    * data. */
  void setUserData( void* data )
  { m_userData = data; }

  /** Returns previously stored user data pointer */
  void* getUserData() const
  { return m_userData; }

  /** Adds the shared context to interna list of shared contexts and update the 
    * sharedId with the one from sharedContext */
  bool setSharedWith( SoGLContext* sharedContext );

  // internally used by PBuffer to change the display.
  void forceDisplay( const SbGlContextHelper::Display& display );

  // Used internally in remote to update the drawable (Linux only)
  void forceDrawable( const SbGlContextHelper::Drawable& drawable );

  // Used internally to avoid the destruction of the context
  // in some cases, like Qt because Qt destroy the context.
  void forceContext( const SbGlContextHelper::GLContext& context )
  { m_context = context; }

  //Allow to change the no delete flag since the public method is one way only
  inline void forceNoGLContextDelete(bool flag) { m_noOGLContextDelete = flag; }

  // Used internally to get the drawable of this context
  const SbGlContextHelper::Drawable& getDrawable() const
  { return m_drawable;}

  virtual SbString getInfos();

  //Provides the name of the graphic card used by the context.
  SbString getDeviceName() const;

  /** Prints out the information about the context. 
    * Available on X11 Only.
    */
  void printInformation();

  /** Prints information about the current context.
    * Available on X11 Only.
    */
  static void printCurrentContextInformation();

  /** Prints information about each context.
    * Available on X11 Only.
    */
  static void printContextsInformation();

  /**
   * This function is used to indicates to the SoGLContext which use the specified
   * drawable that it is not valid anymore. This prevent unexpected bind when there
   * is no valid drawable for the context.
   */
  static void invalidateDrawable(SbGlContextHelper::Drawable drawable);

  /**
   * This function is used to indicates to the SoGLContext which use the specified
   * display that it is not valid anymore. This prevent unexpected bind when there
   * is no valid display for the context.
   */
  static void invalidateDisplay(SbGlContextHelper::Display display);

  /**
   *  Set if the OpenGL buffer swap should be synchronized to the vertical refresh of the screen.
   *  Default is normally true, but may be controlled through system settings on some machines.
   *  See #isVSyncEnabled().
   *
   *  VSync prevents rendering artifacts commonly called "tearing" caused by
   *  the GPU updating the frame buffer with the next frame image before the video hardware
   *  has finished updating the screen. However this also limits performance to the screen 
   *  refresh rate, which is typically 60 frames per second.  Higher "frames per second" for
   *  benchmarks may be possible with VSync disabled.
   *  This feature needs the WGL_EXT_swap_control extension on Windows and the GLX_SGI_video_sync extension on X11.
   *
   *  The driver must not force ON or OFF the vertical sync.
   *
   *  Returns :
   *  - true if the vertical sync has been correctly enabled/disabled.
   *  - false otherwise.
   */
  bool setVSyncEnabled(bool SO_UNUSED_PARAM(on));

  /**
   *  Returns the VSync enabled state.
   *  Default is normally true, but may be controlled through system settings on some machines.
   *  See #setVSyncEnabled().
   */
  bool isVSyncEnabled();

  /** Return true if context is currently binded for current thread. */
  bool isBinded();

  static GLEWContextStruct* glewGetCurrentContext();
  static SoGLGlew::GlewContextNative* glewGetCurrentContextNative();

protected:
  /**
   * Destructor (used by ref/unref)
   */
  virtual ~SoGLContext();

  /**
   * Remove all content of sharedGroup.
   * This is usefull for derived class which manage the context like SoPBuffer.
   * Such class need/want to remove attached objects before deleting the corresponding context.
   */
  void removeSharedObjects();

private:

  /** Display a warning if refcount < 1 */
  void checkRefCount() const;
  
  virtual bool setSharedWith( SoDeviceContext* sourceCtx );

  /**
   * Set the swap interval for the window associated with this context.
   * A value of 0 disable the VSync while a value of 1 enable it.
   * A value of -1 means use the default value.
   */
  void setSwapInterval(int interval);

  // Class static thread local storage
  struct MTstruct
  {
    typedef std::vector<SoGLContext*> ContextVec;
    /** This list keep an eye on the binded contexts for each thread. */
    ContextVec* m_bindedContexts;
  };

  bool initContext( SoGLContext* sourceContext, bool shared );
  void preInitContext();
  bool postInitContext();

  void checkContextParameters( const char* function );

  SoGLContext* findSharedContext() const;

  typedef std::list< SoGLContext* > SoGLContextList;

  /** Static internal list of available contexts. */
  static SoGLContextList s_contexts;

  /** Static variable which indicates the first available id for the contexts. */
  static int s_firstAvailableId;

  /** Static data mutex. */
  static SbThreadSpinlock s_contextsMutex;

  /** The internal id of the context. */
  int m_id;

  SbGlContextHelper::GLContext m_context;
  SbGlContextHelper::Drawable m_drawable;

  /**
   * When m_context is set to 0 during invalidate(), we need to keep the old
   * value to allow commonDeleteContextHook() to unregister the context.
   */
  SbGlContextHelper::GLContext m_savedContextForDelete;

  SoGLFormat m_format;

  void* m_userData;

  static GPUVendorType m_GpuVendorType;

  bool m_noOGLContextDelete;

  bool m_drawableIsValid;
  bool m_displayIsValid;

  static int s_checkContext;

  static int s_oivCompatibilityMode;

  /** Current global sharedGroupPolicy. */
  static SharedGroupPolicy s_sharedGroupPolicy;

  // True if all content of shared group is removed.
  bool m_sharedObjectsRemoved;

  /** If true displays warnings about ref in the bind method */
  bool m_displayWarning;

  static SoGLContext* findSoGLContext(SbGlContextHelper::GLContext ctx);
  void pushInBindedList();
  static void updateBindedList();
  static void updateBindedList(SbGlContextHelper::Display display, SbGlContextHelper::Drawable drawable, SbGlContextHelper::GLContext context);
  static SoGLContext* popBindedList();

  bool m_thirdPartyContext;
  bool m_isBeingDestroyed;
  SbGPUCapabilities* m_graphicCapabilities;
  static FunctionRedirector s_deleteContextRedirector;
  
  static bool callOriginalDeleteContext(SbGlContextHelper::Display display, SbGlContextHelper::GLContext context, int& returnValue);
  static int commonDeleteContextHook(SbGlContextHelper::Display dpy, SbGlContextHelper::GLContext ctx);

#if defined(WIN32)
  static int _stdcall deleteContextHook(SbGlContextHelper::GLContext context);
#elif defined(__APPLE__)
  static int deleteContextHook(SbGlContextHelper::GLContext context);
#else
  static int deleteContextHook(SbGlContextHelper::Display dpy, SbGlContextHelper::GLContext ctx);
#endif
    
  static MTstruct::ContextVec& getBindedContextsList();
};

#ifdef _MSC_VER
#pragma warning( pop )
#endif


#endif //SOGLCONTEXT_H

