/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Martial PAPILLON (June 2013)
**=======================================================================*/

#ifndef SO_DEVICE_SETTINGS
#define SO_DEVICE_SETTINGS

#include <Inventor/devices/SoDevice.h>
#include <Inventor/lists/SbStringList.h>

/**
* @VSGEXT Device settings management class
*
* @ingroup Devices
*
* @DESCRIPTION
*
* This class provides functions to query and modify information about OpenGL device settings.
* A setting is a vendor defined collection of device dependent parameters like "Stereo - Enable",
* "Antialiasing ? Mode" or "Vertical Sync".  NVIDIA calls this a "driver profile".
* Parameters can be set as a group using \if_dotnet #SetSettingName(), \else #setSettingName(), \endif
* with a predefined name, or one by one using \if_dotnet #SetParameter(). \else #setParameter(). \endif
*
* Applications should only create an SoDeviceSettings object using the SoGLDevice method
* \if_dotnet GetDeviceSettings() \else getDeviceSettings() \endif (see below).
*
* Device settings should be changed before any rendering is done, i.e. before the OpenGL render context is created.
*
* Setting and parameter changes do not take effect until applySettings() is called.
*
* Changes should only affect the current running application (but this may depend on the
* graphics board vendor and driver installed).
*
* The settings and parameters that can be queried through this interface may not correspond to the
* parameters displayed by the graphics board vendor's "control panel" application.  This depends on
* the vendor and the driver installed.
*
* @EXAMPLE
* Set a predefined driver setting (profile) for an NVIDIA graphics board.
* \if_cpp
* \code
*   #include <Inventor/devices/SoGLDevice.h>
*
*   SoGLDevice* gldevice = SoGLDevice::getDevice(0);
*   SoGLDeviceSettings* devSettings = gldevice->getDeviceSettings();
*   if (devSettings)
*   {
*     devSettings->setSettingName( "Workstation App - Dynamic Streaming" );
*     devSettings->applySettings();
*   }
* \endcode
* \endif
* \if_dotnet
* \code
*   SoGLDevice gldevice = SoGLDevice.GetDevice( 0 );
*   SoGLDeviceSettings devSettings = gldevice.GetDeviceSettings();
*   if (devSettings != null)
*   {
*     devSettings.SetSettingName( "Workstation App - Dynamic Streaming" );
*     devSettings.ApplySettings();
*   }
* \endcode
* \endif
* \if_java
* \code
*   SoGLDevice gldevice = SoGLDevice.getDevice( 0 );
*   SoGLDeviceSettings devSettings = gldevice.getDeviceSettings();
*   if (devSettings != null) {
*     devSettings.setSettingName( "Workstation App - Dynamic Streaming" );
*     devSettings.applySettings();
*   }
* \endcode
* \endif
*
* @SEE_ALSO
*   SoGLDevice
*/
class INVENTORGL_API SoGLDeviceSettings
{
public:
  /**
   * Initializes SoDeviceSettings.
   */
  static void initClass();

  /**
   * Cleans SoDeviceSettings.
   */
  static void exitClass();

  /**
   * Returns number of parameters in current setting.
   */
  virtual int getParameterCount() const = 0;

  /**
   * Returns the name of the parameter with the specified index in the current setting.
   * Use #getParameterCount() to get the number of parameters in the current setting.
   * For example 
   * "Ambient Occlusion", "Anisotropic filtering mode", "Anisotropic filtering setting",
   * "Antialiasing - Behavior Flags", "Preferred OpenGL GPU", etc.
   */
  virtual SbString getParameterIdbyIndex(const unsigned int index) const = 0;

  /**
   * Returns current value of specified parameter in current setting.
   * The value is converted to a string.
   */
  virtual SbString getParameter(const SbString& pParameterId) const = 0;

  /**
   * Set the value of a parameter in the current setting.
   * @return TRUE if success, FALSE otherwise
   */
  virtual bool setParameter(const SbString& pParameterId, const SbString& pParameterValue) = 0;

  /**
   * Returns the name of the current setting (vendor defined list of parameters).
   */
  virtual SbString getSettingName() const = 0;

  /**
   * Set a setting (vendor defined list of parameters) by name.
   * See #getSettingsCount and #getSettingNameByIndex to retrieved predefined list of settings.
   * @return TRUE if success, FALSE otherwise
   */
  virtual bool setSettingName(const SbString &pPredefinedSettingName) = 0;

  /**
   * Returns the number of vendor defined settings.
   * Depends on the graphics board and driver version.
   */
  virtual int getSettingsCount() const = 0;

  /**
   * Returns the name of the specified setting.
   * See #getSettingsCount to get the number of settings available.
   */
  virtual SbString getSettingNameByIndex(const unsigned int index) const = 0;

  /**
   * Apply changes to the setting and/or parameters.
   */
  virtual bool applySettings() const = 0;

  /**
   * Prints information about the device settings.
   */
  INVENTORGL_API friend std::ostream& operator << ( std::ostream& os, const SoGLDeviceSettings &profile );

  // ----------------------- Protected usage --------------------------------------
protected:

  /**
   * Protected constructor.
   */
  SoGLDeviceSettings();

  /**
   * Protected destructor.
   */
  ~SoGLDeviceSettings();

  /**
   * Protected method facilities to print Profile information
   */
  virtual SbString printProfile() const { return SbString("No Profile information"); };

  // Refs counter
  static int s_initRefCount;

  // ----------------------- Private usage --------------------------------------
private:

  /**
   * Private copy constructor.
   */
  SoGLDeviceSettings(const SoGLDeviceSettings &pOrigin);
  const SoGLDeviceSettings& operator = (const SoGLDeviceSettings& copy);

};

#endif // SO_DEVICE_SETTINGS
