/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/

#ifndef SO_GPU_BUFFER_OBJECT_H
#define SO_GPU_BUFFER_OBJECT_H

#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoCpuBufferObject.h>

/*! \cond PRIVATE */
namespace inventor
{
namespace renderer
{
template<typename ResourceType> struct Resource;
}
}
/*! \endcond */

SO_PIMPL_PUBLIC_DECLARATION(SoGpuBufferObject)

/**
 * @VSGEXT GPU buffer object class
 *
 * @ingroup CpuDevice
 *
 * @DESCRIPTION
 *
 * This class provides management functions for GPU memory buffers.
 *
 * Note: Since Open Inventor 10.4, applications should use SoGPUBufferObject,
 * instead of SoGLBufferObject, in most cases. For example to store vertex
 * data in GPU memory for use with SoBufferedShape or SoVertexShaderParameterBufferObject.
 *
 * See SoBufferObject for general information about buffer objects.
 *
 * NOTES:
 * @if_cpp
 * - Creating a GPU buffer requires a current GPU context (like OpenGL). @BR
 *   For example, regarding the historical viewers the method SoWinGLWidget::bindNormalContext()
 *   must be called before using a SoGpuBufferObject. Concerning the viewer components,
 *   the window containing the RenderArea must be exposed before using a SoGpuBufferObject.
 * @endif
 * @if_java
 * - Creating a GPU buffer requires a current GPU context (like OpenGL). @BR
 * @endif
 * @if_dotnet
 * - Creating a GPU buffer requires a current GPU context (like OpenGL). @BR
 * @endif
 * - Modifying the properties of an GPU buffer require a current GPU context. @BR
 *   This includes mapping and unmapping the buffer (map()/unmap()),
 *   allocating memory (setSize()) and copying data into the buffer
 *   (e.g. memcpy()).
 *
 * - The same instance of SoGpuBufferObject can not be used inside two GPU contexts
 *   that are not shared. For example, if an instance of SoBufferedShape contains
 *   SoGpuBufferObjects in its fields, this instance can be used in multiple
 *   viewers only if their contexts are shared.
 *
 * @if_cpp
 * - SoBufferObject classes are reference counted objects (just like nodes, 
 *   paths, etc).  Therefore you cannot create an SoBufferObject on the
 *   stack. And to delete an SoBufferObject you must ref and unref the 
 *   object.  The SoRef smart pointer is a safe and convenient way to
 *   create buffer objects. See examples in SoBufferObject.
 * @endif
 *
 * See SoBufferObject for general information about buffer objects.
 *
 * See SoBufferedShape for an example of storing vertices in a GPU buffer.
 *
 * @EXAMPLE
 * Load data into a buffer object.
 *
 * Create a GPU buffer object and load data from an array in memory:
 * \if_cpp
 * \code
 * const float vertices[][3] = {
 *      1,0.5,0, 0,1,0, -1,0.5,0,
 *     -1,-1,0, 1,-1,0, 1,0,0, -1,0,0,
 *     -1,-1.5,0, 1,-1.5,0
 * };
 * const int NUM_COORDS = sizeof(vertices) / sizeof(SbVec3f);
 *
 * // Wrap coordinate array in a CPU buffer object
 * SoRef<SoCpuBufferObject> cpuBuffer = new SoCpuBufferObject( (void*)coords, NUM_COORDS * sizeof(SbVec3f) );
 *
 * // Create a GPU buffer and load data from CPU buffer
 * // Ensure that a context is created and bound before this step !
 * //   - If using the historical viewers call the SoWinGLWidget::bindNormalContext() method here
 * //   - If using viewer components, make sure the window containing the RenderArea is already exposed
 * SoRef<SoGpuBufferObject> gpuBuffer = new SoGpuBufferObject( SoGpuBufferObject::STATIC, SoGpuBufferObject::SHARED );
 * gpuBuffer->setSize( cpuBuffer->getSize() ); // Set the buffer size (allocate memory)
 * gpuBuffer->memcpy( cpuBuffer.ptr() );      // Copy data into the buffer
 * \endcode
 * \endif
 * \if_dotnet
 * \code
 * float[] vertices = new float[9 * 3] {
 *      1.0f, 0.5f,0.0f, 0.0f, 1.0f,0.0f, -1.0f,0.5f,0.0f,
 *     -1.0f,-1.0f,0.0f, 1.0f,-1.0f,0.0f,  1.0f,0.0f,0.0f, -1.0f,0.0f,0.0f,
 *     -1.0f,-1.5f,0.0f, 1.0f,-1.5f,0.0f
 * };
 *
 * // Wrap coordinate array in a CPU buffer object
 * SbNativeArray<float> vertArray = new SbNativeArray<float>(vertices);
 * SoCpuBufferObject vertBuf = new SoCpuBufferObject( (SbNativeArray<byte>)vertArray );
 *
 * // Create a GPU buffer and load data from CPU buffer
 * // Ensure that a context is created and bound before this step !
 * //   - If using the historical viewers call the SoWinGLWidget::bindNormalContext() method here
 * //   - If using viewer components, make sure the window containing the RenderArea is already exposed
 * SoGpuBufferObject gpuBuffer = new SoGpuBufferObject( SoGpuBufferObject.BufferAccessFrequencies.STATIC, SoGpuBufferObject.BufferAccessNatures.SHARED );
 * gpuBuffer.SetSize( cpuBuffer.GetSize() );  // Set the buffer size (allocate memory)
 * gpuBuffer.Memcpy ( cpuBuffer );            // Copy data into the buffer
 * \endcode
 * \endif
 *
 * \if_java
 * \code
 * float[] vertices = {
 *         1.0f, 0.5f,0.0f, 0.0f, 1.0f,0.0f, -1.0f,0.5f,0.0f,
 *        -1.0f,-1.0f,0.0f, 1.0f,-1.0f,0.0f,  1.0f,0.0f,0.0f, -1.0f,0.0f,0.0f,
 *        -1.0f,-1.5f,0.0f, 1.0f,-1.5f,0.0f
 * };
 *
 * // Create a GPU buffer and allocate memory
 * // Ensure that a context is created and bound before this step !
 * // Make sure the window containing the RenderArea is already exposed
 * SoGpuBufferObject gpuBuffer = new SoGpuBufferObject( SoGpuBufferObject.BufferAccessFrequencies.STATIC, SoGpuBufferObject.BufferAccessNatures.SHARED );
 * gpuBuffer.setSize( vertices.length * Float.SIZE/8 ); // Set the buffer size (allocate memory)
 *
 * // Copy data into the buffer object
 * FloatBuffer vertData = gpuBuffer.map( SoBufferObject.AccessModes.SET ).asFloatBuffer();
 *     vertData.put(vertices);
 * gpuBuffer.unmap();
 *
 * \endcode
 * \endif
 *
 *
 * @EXAMPLE
 * Access data stored in a GPU buffer object.
 * \par
 * \if_cpp
 * \code
 * float* data  = (float*)gpuBuffer->map( SoBufferObject::READ_ONLY );
 * float  value = data[0];
 * value = data[1];
 *     . . .
 * gpuBuffer->unmap();
 * \endcode
 * \endif
 *
 * \if_dotnet
 * \code
 * SbNativeArray<float> data = (SbNativeArray<float>)gpuBuffer.Map( SoBufferObject.AccessModes.READ_ONLY );
 * float value = data[0];
 * value = data[1];
 *     . . .
 * gpuBuffer.Unmap();
 * \endcode
 * \endif
 *
 * \if_java
 * \code
 * FloatBuffer data = gpuBuffer.map( SoBufferObject.AccessModes.READ_ONLY ).asFloatBuffer();
 * float value = data.get(0);
 * value = data.get(1);
 *     . . .
 * gpuBuffer.unmap();
 * \endcode
 * \endif
 *
 * @SEE_ALSO
 * SoCpuBufferObject
 *
 * <b>NOTE</b>: Class available since Open Inventor 10.0.
 */
class INVENTOR_API SoGpuBufferObject : public SoBufferObject
{
  SO_TYPED_CLASS_HEADER();
  SO_PIMPL_PUBLIC_HEADER(SoGpuBufferObject)

public:

  /**
   * Available values for hints on how frequently the application will be
   * changing the buffer's data.
   */
  enum BufferAccessFrequency
  {
    /** The application will set the data once. */
    STATIC,

    /** The application will set the data multiple times. */
    DYNAMIC
  };

  /**
   * Available values for hints on the nature of the application's access to
   * the buffer.
   * They indicate whether the application will be directly reading or writing
   * the buffer's data.
   */
  enum BufferAccessNature
  {
    /**
     * The application will be writing data to the buffer,
     * but will not read it.
     */
    SHARED,

    /**
     * The application will be neither writing nor reading the data.
     * Only GPU commands will interact with the buffer.
     * This is typically used when a buffer object is used to pass data from
     * one place to another on GPU side.
     */
    DEVICE
  };

  /**
   * Constructor.
   *
   * @param accessFrequency The intended access frequency of this buffer. @useenum{BufferAccessFrequency}
   * @param accessNature The intended access nature of this buffer. @useenum{BufferAccessNature}
   */
  SoGpuBufferObject( BufferAccessFrequency accessFrequency = STATIC, BufferAccessNature accessNature = SHARED );

  /**
   * @copydoc SoBufferObject::setSize
   */
  virtual bool setSize( size_t size );

  /**
   * Create a new buffer with the same properties as the current one. @BR
   * The new instance will have the same context
   * or device properties, but no memory is allocated.
   */
  virtual SoBufferObject* createInstance() const;

  /**
   * This function clears the content of the buffer, it frees the memory if the mode was COPY.
   */
  virtual void clearInstance();

  /** Returns the current buffer access frequency hint for this buffer object */
  BufferAccessFrequency getAccessFrequency() const;

  /** Returns the current buffer access nature hint for this buffer object */
  BufferAccessNature getAccessNature() const;

SoINTERNAL public:

    void copyToCpuBuffer(SoCpuBufferObject* targetBufferObject, size_t destOffset = 0, size_t sourceOffset = 0, size_t copySize = SO_BUFFER_SIZE_ALL);

protected:
  /**
   * Destructor. @BR
   * Any memory allocated by the buffer object (e.g. by setSize()) is freed.
   */
  virtual ~SoGpuBufferObject();

  friend struct inventor::renderer::Resource<SoBufferObject>;
};

#endif // SO_GPU_BUFFER_OBJECT_H
