/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Thibaut Andrieu (Apr 2011)
**=======================================================================*/

#if !defined _SO_RECTANGLE_SCREEN_DRAWER_H_
#define _SO_RECTANGLE_SCREEN_DRAWER_H_


#include <Inventor/drawers/SoPolyLineScreenDrawer.h>
#include <Inventor/fields/SoSFEnum.h>

/**
 * @VSGEXT Interactively draws a rectangle in normalized screen space.
 *
 * @ingroup drawers
 *
 * @DESCRIPTION
 *   This class can be used to interactively draw a rectangle on screen.
 *
 *   When the rectangle is finished, an event is raised to notify the application
 *   (see SoPolyLineScreenDrawer::onFinish ).  Points are returned as normalized
 *   coordinates in the range -1..1, which is conveniently the default view volume
 *   for an SoOrthographicCamera node.
 *
 *   The line color and simplification threshold are specified by
 *   fields in the parent class SoPolyLineScreenDrawer.
 *
 *   Please see SoPolyLineScreenDrawer for general notes and code example.
 *
 *   Rectangle specific notes:
 *
 *   - The rectangle can be specified by 2 corner points (default) or by
 *     a center point and one corner point (see #CreationMethod).
 *
 *   - The rectangle points are always delivered upper-left corner first,
 *     then counter-clockwise.
 *
 *   - The #simplificationThreshold field does not apply to this node. 
 *     A rectangle always has four points.
 *
 *   - The #isClosed field does apply to this node (but FALSE is not very useful).
 *
 *   - If user finishes a rectangle, then the next click will automatically clear
 *     the polyline (begin a new rectangle).
 *
 *   - SbEventHandlers:
 *     - OnStart : Triggered on LeftMouseDown input event.
 *     - OnMove : Triggered on MouseMove input events (after Start).
 *     - OnFinish : Triggered on LeftMouseUp input event.
 *
 *   \if_cpp
 *   [C++] @BR Screen drawer classes must be initialized by calling SoInteraction::init(). High level
 *   viewer init methods, like SoWin::init(), automatically do this. However, if the application
 *   is calling SoDB::init() directly, then it is usually necessary to also call SoInteraction::init()
 *   before using the screen drawer classes.
 *   \endif
 *
 * @USAGE
 * - Press left mouse and drag to draw the rectangle.
 * - Release to finish it.
 * - Press Escape to cancel.
 *
 * @FILE_FORMAT_DEFAULT
 *    RectangleScreenDrawer {
 *    @TABLE_FILE_FORMAT
 *       @TR point        @TD []
 *       @TR color        @TD 1 0 0
 *       @TR simplificationThreshold      @TD 0
 *       @TR isClosed     @TD TRUE
 *       @TR method       @TD CORNER_CORNER
 *    @TABLE_END
 *    }
 *
 * @SEE_ALSO
 *    SoEllipseScreenDrawer,
 *    SoLassoScreenDrawer,
 *    SoPolyLineScreenDrawer,
 *    SoPolygonScreenDrawer,
 */
class INVENTOR_API SoRectangleScreenDrawer : public SoPolyLineScreenDrawer 
{
  SO_NODE_HEADER( SoRectangleScreenDrawer );

public:
  /** Constructor */
  SoRectangleScreenDrawer();

  /** 
   * Methods to create rectangle. 
   */
  enum CreationMethod
  {
    /** Initial point is a corner, current point is a corner. */
    CORNER_CORNER,
    /** Initial point is the center, current point is a corner. */
    CENTER_CORNER
  };

  /** Method to create rectangle.
   *  @useenum{CreationMethod}. Default is CORNER_CORNER. 
   */
  SoSFEnum method;

  /** @copydoc SoPolyLineScreenDrawer::reset() */
  virtual void reset();

SoINTERNAL public:

  /** Register in database */
  static void initClass();

  /** Unregister from database */
  static void exitClass();

SoEXTENDER_Documented protected:

  /** Finish rectangle. */
  virtual void onKeyDown( SoHandleEventAction* action );

  /** Update rectangle. */
  virtual void onMouseDragging( SoHandleEventAction* action );

  /** Start creation of new rectangle. */
  virtual void onMouseDown( SoHandleEventAction* action );

  /** Finish creation of rectangle. */
  virtual void onMouseUp( SoHandleEventAction* action );

  /** Is a rectangle creating. */
  inline bool isCreating() const { return m_initPoint[0] != -1 && m_initPoint[1] != -1; }

protected:

  /** Destructor. */
  virtual ~SoRectangleScreenDrawer();

private:

  /** 
   * Initial point of rectangle (usually the mouseDown event). See #CreationMethod.
   * If (-1, -1), rectangle is not creating 
   */
  SbVec2f m_initPoint;

};


#endif // _SO_RECTANGLE_SCREEN_DRAWER_H_


