/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_CACHE_ELEMENT_
#define  _SO_CACHE_ELEMENT_

#include <Inventor/SbBasic.h>
#include <Inventor/elements/SoSubElement.h>
#include <Inventor/threads/SbThreadLocalStorage.h>
#include <Inventor/misc/SoRef.h>
#include <Inventor/caches/SoCache.h>
#include <Inventor/SoDB.h>
#include <unordered_map>
#include <bitset>
#include <stack>

class SoCache;

/**
*   Stores the most recently opened cache.
*
* @ingroup elements
*
*   @DESCRIPTION
*   This element stores the most recently opened cache.
*   @SEE_ALSO
*   SoSeparator, SoRenderList
*/

SoEXTENDER_Documented class INVENTOR_API SoCacheElement : public SoElement {

  SO_ELEMENT_HEADER(SoCacheElement);

 public:
  /**
  *  Sets cache in element accessed from state.
  */
  static void         set(SoState *state, SoCache *cache);

  /**
  *  Returns the cache stored in an instance. This may be NULL.
  */
  SoCache *           getCache() const { return cache.ptr(); }

  /**
  *  Returns TRUE if any cache is currently open in the state.
  */
  static SbBool       anyOpen(SoState *state);

  /**
  *  Closes any open cache.
  */
  static void         closeAnyOpen(SoState* state);

  /**
  *  Invalidate any open caches.  This is called by nodes that
  *  should not be cached.
  */
  static void         invalidate(SoState *state);

  /**
  *  Overrides this method to unref cache.
  */
  virtual void        pop(SoState *state, const SoElement *prevTopElement);

  /**
  *  Returns the next cache element in the stack.
  */
  SoCacheElement *    getNextCacheElement() const
    { return (SoCacheElement *) getNextInStack(); }

  /** push to save dependencies list */
  virtual void push(SoState *state);

protected:

  /** Initializes element */
  virtual void init(SoState *state);

  /**
  *  Overrides this method to print an error message and return
  *  FALSE. Cache elements should never be compared, since they
  *  never appear in the elements-used list of caches!
  */
  virtual SbBool matches(const SoElement* elt) const;

  /**
  *  Copy method prints error and returns NULL; see comment above.
  */
  virtual SoElement* copyMatchInfo() const;

 SoINTERNAL public:
  // Initializes the SoCacheElement class
  static void initClass();
  static void exitClass();
  
  typedef std::bitset<SO_ELEMENT_MAX_KEY> ElementBitList;

  /**
   *  Invalidate any open caches.  This is called by nodes that
   *  should not be cached.
   */
  void invalidate() const;

  /** 
   * Push/pop only given element from the set of setters dependencies.
   */
  static void pushElementSetters(SoState* state, SoElementKeyType keyToPush);
  static void popElementSetters(SoState* state, SoElementKeyType keyToPop);

  /** add a delimeter for indentifying node traversal start in the setters */
  static void addNodeTraversalStart(SoState* state);

  /** Save the node (if any) which setted the given element */
  static void trackDependencies(SoState *state, const SoElement *elt);

  /** Save the node (if any) which has done a get on the given element */
  static void trackGetterDependencies(SoState *state, const SoElement *elt);

  /** clear list of getters/setters */
  static void clearGettersSetters(SoState* state);

  // Adds the given element to the elements used lists of all
  // currently open caches in the state
  static void         addElement(SoState *state, const SoElement *elt);

  // Adds a dependency on the given cache to all currently open
  // caches in the state
  static void         addCacheDependency(SoState *state, SoCache *cache);

  // Sets invalidated bit, and returns its old value.  Used by
  // SoGLCacheList so auto-caching doesn't cache too much.
  static SbBool       setInvalid(SbBool newValue);

  // returns the current cache, from the top of the stack.  Does not
  // cause a cache dependence like getConstElement().
  static SoCache *    getCurrentCache(SoState *state)
  {return state->getElementNoPush<SoCacheElement>()->cache.ptr();}

  /** @copydoc SoCache::flagDelayedPath */
  static void flagDelayedPath(SoState* state);

  static SbBool listIsOpen(SoState* state);

  /**
   * Return the dependency scene graph which contains the hierarchy of the list of nodes
   * needed to compute the given elements. Involved nodes are stored in the dependencies vector.
   */
  SoRef<SoGroup> getDependencies(const ElementBitList& elementKeys, std::vector<SoNode*>& dependencies) const;

  /** List of lazy evaluation state as bitfield mask */
  enum LazyEvaluationPolicy
  {
    DISABLED  = (0),      // Lazy evaluation is disabled
    ENABLED   = (1<<0),   // Lazy evaluation is enabled
    INHERITED = (1<<1)    // Lazy evaluation policy is propagated to children
  };

  inline void setLazyEvaluationPolicy(int32_t flag)
  {
    m_lazyEvaluationPolicy = flag;
    if (!hasLazyEvaluation())
      m_lazyEvaluationPolicy = DISABLED;
    if (!isLazyEvaluationEnabled())
      invalidate();
  }

  inline bool isLazyEvaluationEnabled() const
  {
    return (m_lazyEvaluationPolicy & ENABLED) > 0;
  }

  inline bool isLazyEvaluationInherited() const
  {
    return (m_lazyEvaluationPolicy & INHERITED) > 0;
  }

  /** 
   * True if rendercache must do lazy eval 
   * Will be false if caching is disabled on the whole path leading to the current shape.
   * It avoid doing useless lazy eval if the scene is retraversed at each frame
   */
  bool needLazyEval() const;

  /**
   *  Sets lazy evaluation policy.
   */
  static void setLazyEvaluationPolicy(SoState *state, int32_t lazyEvaluationPolicyFlag);

  /**
   *  Sets cache in element accessed from state with lazy evaluation policy.
   */
  static void set(SoState *state, SoCache *cache, int32_t lazyEvaluationPolicyFlag);
  
/** Check if state and element need tracking */
  static bool needTracking(SoState *state, const SoElement* elt);

  /** Check if state needs tracking */
  static bool needTracking(SoState *state);

  /** Check if lazy evaluation is active */
  static bool hasLazyEvaluation();

  /** Keep track of which node setted a given element */
  struct ElementInfos
  {
    enum Type { PUSH, POP, ELEMENT, NODE_TRAVERSAL_START } ;
    SoElementKeyType elementKey;
    SoNode* node;
    Type type;

    ElementInfos(){}
    ElementInfos(SoElementKeyType elementKey, SoNode* node, Type type)
    {
      this->elementKey = elementKey;
      this->node = node;
      this->type = type;
    }
  };

  /**
   * When a separator is traversed, SoCacheElement::push() is called and it should call this class'
   * push() method. This stores current data list size so that, when calling pop(), data list is restored
   * at value when entering separator.
   */
  template<typename Data>
  struct ListContainer : public SoRefCounter
  {
    ListContainer()
    {}

    void push()
    {
      m_lastSize.push( m_list.size() );
    }

    void pop()
    {
      size_t lastSize = m_lastSize.top();
      m_list.resize( lastSize );
      m_lastSize.pop();
    }

    std::stack<size_t> m_lastSize;
    std::vector<Data> m_list;
  };

  /** Struct used to accumulate element set by nodes from top separator to current separator. */
  typedef ListContainer<ElementInfos> ElementInfosListContainer;

  typedef std::vector<ElementInfos> ElementInfosList;

  /** Return list of element that have been set by which node. */
  const ElementInfosList& getElementSetters() const { return m_elementSetters->m_list; }

  /** Struct used to accumulate nodes that set ShaderState elements */
  typedef ListContainer<SoNode*> LazyEvalNodesListContainer;

  /**
   * Return the list of hashes corresponding to the list of nodes that set ShaderState elements for each Shader State.
   *
   * List {
   *   SHADER_STATE_1: <node1, node2, node3>        => HASH1,
   *   SHADER_STATE_2: <node4, node5>               => HASH2,
   *   SHADER_STATE_3: <node6, node7, node8, node9> => HASH3,
   *   ...
   * }
   */
  static const std::vector<uint64_t>& getLazyEvalNodesHashes( const SoState* state );

  struct MTstruct
  {
    bool invalidated;
  };

  SB_THREAD_TLS_HEADER();

 protected:
  virtual ~SoCacheElement();

 private:

  /** 
   * Return the node at the tail of the current path. 
   * Return NULL if we don't need to track dependencies (lazy evaluation disabled, not a render action...)
   */
  static SoNode* getSetterGetterNode(SoState *state, const SoElement *elt);
  
  SoRef<SoCache> cache; // Stores pointer to cache

  /** Keep the list of node/element pair that have been set/get from top separator until this level.
   * => Top down propagation */
  SoRef<ElementInfosListContainer> m_elementSetters;

  /**
   * Keep the list of nodes that set ShaderState elements from top separator until this level,
   * for each ShaderState.
   * => Top down propagation
   */
  std::vector<SoRef<LazyEvalNodesListContainer> > m_lazyEvalNodesLists;

  /** Keep a cache of hashes for the list of LazyEval nodes (see #m_lazyEvalNodesLists). */
  std::vector<uint64_t> m_lazyEvalNodesHashes;

  /** 
   * See LazyEvaluationPolicy for usage.
   * It avoid doing useless lazy eval if the scene is retraversed at each frame
   */
  int32_t m_lazyEvaluationPolicy;

  friend class SoElement;
};

#endif /* _SO_CACHE_ELEMENT_ */


