/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/
#ifndef  _SO_CLIP_PLANE_ELEMENT
#define  _SO_CLIP_PLANE_ELEMENT

#include <Inventor/SbLinear.h>
#include <Inventor/elements/SoAccumulatedElement.h>

/**
*   Stores the current set of clipping planes, specified
*   as SbPlanes.
* 
* @ingroup elements
* 
*   @DESCRIPTION
*   This element stores the current set of clipping planes, specified
*   as SbPlanes.
* 
*   When a plane is added, this element gets the current model matrix
*   from the state and stores it in the instance. This allows the
*   get() method to return the clip plane in object space (the plane
*   as originally defined) or world space (after being transformed by
*   the model matrix).
* 
*   @SEE_ALSO
*     SoClipPlane, SoOrthoSlice
*
*/

SoEXTENDER_Documented class INVENTOR_API SoClipPlaneElement : public SoAccumulatedElement
{

  SO_ELEMENT_HEADER(SoClipPlaneElement);

public:
  /** Adds a clip plane to the current set in the state. */
  static void add(SoState *state, SoNode *node, const SbPlane &plane);

  /** Overrides push() method to copy values from next instance in the stack. */
  virtual void push(SoState *state);

  /** Overrides pop() method to free up planes that were added. */
  virtual void pop(SoState *state, const SoElement *prevTopElement);

  /** Returns the top (current) instance of the element in the state. */
  static const SoClipPlaneElement * getInstance(SoState *state);

  /** Returns the number of planes in an instance. */
  int getNum() const;

  /**
  *  Returns the indexed plane an element as an SbPlane. The plane
  *  can be returned in object or world space.
  */
  const SbPlane& get(int index, SbBool inWorldSpace = TRUE) const;

  /**
   * Returns the maximum number of concurrent clipping planes
   * supported by current implementation.
   */
  static int getMaxPlanes();

  /** Prints element (for debugging). */
  virtual void print(FILE *fp) const;

protected:

  /** @copydoc SoElement::init */
  virtual void init(SoState *state);

SoINTERNAL public:
  
  /** @copydoc SoElement::initClass */
  static void initClass();
  /** @copydoc SoElement::exitClass */
  static void exitClass();

  /** Spaces in which to express clip plane */
  enum Space
  {
    /** Model space when plane was pushed on state. It correspond to plane equation
     * given in SoClipPlane::plane field */
    MODEL,
    /** World space when plane was pushed on state */
    WORLD,
    /** View space. You generally should this space when dealing with clip plane. */
    VIEW
  };

  /**
  *  Returns the indexed plane an element as an SbPlane. The plane
  *  can be returned in object, world space or view space.
  * warning: their may be transforms of camera changes between the time ClipPlane was
  * pushed on state and the time you get it.
  * That's why you generally wish to return it to view space.
  */
  const SbPlane& get(int index, Space space) const;

  /** Enable or disable clip planes */
  static void enable( SoState* state, SbBool activate );

  /** True if clip planes are activated */
  static SbBool isEnabled(SoState* state);

protected:

  /** List of plane structures */
  SbPList planes; 

  /** Index of 1st plane created in this instance */
  int startIndex;

  /** is enabled or not */
  SbBool m_enabled;

  /**
   * Adds the clipping plane to an instance. Takes the new plane and
   * the current model matrix
   */
  virtual void        addToElt(const SbPlane &plane, const SbMatrix &modelMatrix, const SbMatrix &viewMatrix);

  /** Default constructor */
  virtual ~SoClipPlaneElement();
};

#endif /* _SO_CLIP_PLANE_ELEMENT */


