/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (Apr 2009)
**=======================================================================*/

#ifndef _SO_DEPTH_BUFFER_ELEMENT_
#define _SO_DEPTH_BUFFER_ELEMENT_

#include <Inventor/SbBasic.h>
#include <Inventor/SbVec.h>
#include <Inventor/elements/SoSubElement.h>
#include <Inventor/SbEnums.h>

class SoDepthBuffer;

/**
 * @VSGEXT Stores the current depth buffer parameters.
 *
 * @ingroup elements
 *
 * @DESCRIPTION
 * This element stores the current depth buffer parameters used for rendering,
 * including:
 * - Depth buffer test enabled,
 * - Depth buffer writing enabled,
 * - Depth buffer range, and
 * - Depth comparison function.
 *
 * @SEE_ALSO
 *   SoDepthBuffer
 */
class INVENTOR_API SoDepthBufferElement : public SoElement
{
  SO_ELEMENT_HEADER(SoDepthBufferElement);

public:
  /**
   * Depth function mode as defined by OpenGL.
   */
  enum DepthWriteFunction {
    /** Never passes. */
    NEVER = SbEnums::COMP_FUNC_NEVER,
    /** Passes if the	incoming depth value is	less than the stored depth value. */
    LESS = SbEnums::COMP_FUNC_LESS,
    /** Passes if the	incoming depth value is	equal to the stored depth value. */
    EQUAL = SbEnums::COMP_FUNC_EQUAL,
    /** Passes if the	incoming depth value is	less than or equal	to the stored depth value. */
    LEQUAL = SbEnums::COMP_FUNC_LESS_EQUAL,
    /** Passes if the	incoming depth value is greater than the stored depth	value. */
    GREATER = SbEnums::COMP_FUNC_GREATER,
    /** Passes if the	incoming depth value is	not equal	to the stored depth value. */
    NOTEQUAL = SbEnums::COMP_FUNC_NOT_EQUAL,
    /** Passes if the	incoming depth value is greater than or equal	to the stored depth value. */
    GEQUAL = SbEnums::COMP_FUNC_GREATER_EQUAL,
    /** Always passes. */
    ALWAYS = SbEnums::COMP_FUNC_ALWAYS
  };

  /**
  * Sets the depth buffer parameters.
  * This method will (indirectly) call glDepthMask, glDepthFunc, glDepthRange
  * with the new values
  */
  static void set(SoState* state, const SbBool test, const SbBool write, const DepthWriteFunction function, const SbVec2f range);

  /**
  * Sets the depth buffer parameters.
  * This method will (indirectly) call glDepthMask, glDepthFunc, glDepthRange
  * with the new values
  */
  static void set(SoState* state, SoDepthBuffer* depthBuffer,  const SbBool test, const SbBool write, const DepthWriteFunction function, const SbVec2f range);

  /**
  *  Returns current depth buffer parameters.
  */
  static void get(SoState* state, SbBool& test, SbBool& write, DepthWriteFunction& function, SbVec2f& range);

  /**
  *  Prints element (for debugging).
  */
  virtual void print(FILE *fp) const;

  /**
  *  Override push() method to keep GL up to date.
  */
  virtual void push(SoState *state);

  /**
  *  Override pop() method to keep GL up to date.
  */
  virtual void pop(SoState *state, const SoElement *prevTopElement);

protected:

  /** Initializes element. */
  virtual void init(SoState *state);

  /**
  *  Returns TRUE if depth parameters match.
  */
  virtual SbBool matches(const SoElement* elt) const;

  /**
  *  Create and return a copy of this element.
  */
  virtual SoElement* copyMatchInfo() const;


SoINTERNAL public:

  /** register in database. */
  static void initClass();

  /** unregister from database. */
  static void exitClass();

  /** write enable accessor */
  static SbBool getWriteMask(SoState * state);

  static void setWriteMask(SoState* state, const SbBool writeMask);

  /** function accessor */
  static DepthWriteFunction getFunction(SoState * state);

  static void setFunction( SoState* state, const DepthWriteFunction function );

  /** test accessor */
  static SbBool getTest( SoState * state );

  static void setTest( SoState* state, const SbBool test );

  /** range accessor */
  static SbVec2f getRange(SoState * state);

  /** dethbuffer accessor */
  static SoDepthBuffer* getDepthBuffer(SoState * state);

  /** Default depth write function */
  static DepthWriteFunction getDefaultFunction();

  /** Performs a depth test using the given test function. */
  static bool depthTest(DepthWriteFunction func, double currentDepth, double newDepth);

  /** Performs a depth test using the current depth buffer parameters on state. */
  static bool depthTest(SoState* state, double currentDepth, double newDepth);

protected:

  /** Destructor */
  virtual ~SoDepthBufferElement();

  /**
   * Sets the depth buffer parameter in an instance. Has GL side effects.
   * This method will enable or not GL_DEPTH_TEST mode and call glDepthRange/glDepthMask
   * with the new values.
   */
  virtual void setElt(SoDepthBuffer* depthBuffer, const SbBool enable, const SbBool write, const DepthWriteFunction function, const SbVec2f range);

protected:
  /** depth test is enable or not */
  SbBool test;
  /** depth write is enbale or not */
  SbBool write;
  /** write mode functions */
  DepthWriteFunction function;
  /** current depth range */
  SbVec2f range;

  /** current depth buffer */
  SoDepthBuffer* m_depthBuffer;
};

#endif /* _SO_DEPTH_BUFFER_ELEMENT_ */


