/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Alan Norton (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef _SO_GL_LAZY_ELEMENT
#define _SO_GL_LAZY_ELEMENT


#include <Inventor/elements/SoLazyElement.h>
#include <Inventor/actions/SoGLRenderAction.h>
class SoGLRenderCache;
class SoGLLazyState;


/**
*  Manages the GL state for the SoLazyElement.
*
* @ingroup elements
*
*   @DESCRIPTION
*   This element manages the GL state for the SoLazyElement.
*
*   Note that this class relies on SoLazyElement to store the
*   Open Inventor color(s), etc. in the instance.
*   This element keeps GL shadow copies of colors.
*
*   @SEE_ALSO
*   SoLazyElement
*/

SoEXTENDER_Documented class INVENTOR_API SoGLLazyElement : public SoLazyElement
{

  SO_ELEMENT_HEADER(SoGLLazyElement);

 public:
  /**
  *  Overrides push() method to maintain GL state.
  */
  virtual void push(SoState *state);
  /**
  *  Overrides pop() method to maintain GL state.
  */
  virtual void pop(SoState *state, const SoElement *prevTopElement);

  /**
  * Static send, always send top-of-stack. Intended for extender use.
  */
  static void sendAllMaterial(SoState *state) {
    SoGLLazyElement *le = getInstance(state);
    if ((le->invalidBits)||(state->isCacheOpen()))
      le->reallySend(state, ALL_MASK|PATTERN_MASK);
  }

  /**
  * Static send, always send top-of-stack. Intended for extender use.
  */
  static void sendNoMaterial(SoState *state) {
    SoGLLazyElement *le = getInstance(state);
    if ((NO_COLOR_MASK & le->invalidBits)||(state->isCacheOpen()))
      le->reallySend(state, NO_COLOR_MASK);
  }

  /**
  * Static send, always send top-of-stack. Intended for extender use.
  */
  static void sendOnlyDiffuseColor(SoState *state) {
    SoGLLazyElement *le = getInstance(state);
    if ((DIFFUSE_ONLY_MASK & le->invalidBits)||(state->isCacheOpen()))
      le->reallySend(state, DIFFUSE_ONLY_MASK);
  }

  /**
  *  Sends diffuse color and transparency, by index.
  *  Included for compatibility with SoMaterialBundle.
  *  To be used by a shape to send additional colors after first send().
  *  Should NEVER be the first send of a shape!
  */
  void sendDiffuseByIndex(int index) const;

  /**
  *  Destroys knowledge of GL state.
  */
  void reset(SoState* state, uint32_t bitmask) const;

  /**
  *  Returns the top (current) instance of the element in the state
  *  Note it does NOT cause cache dependency!
  *  It also casts away the const.
  */
  static SoGLLazyElement * getInstance(const SoState *state) {
    return const_cast<SoGLLazyElement*>(state->getConstElement<SoGLLazyElement>());
  }

protected:

  /** @copydoc SoElement::commonInit() */
  virtual void commonInit();

  /** Initializes element */
  virtual void init(SoState *state);

  /** @copydoc SoLazyElement::matches() */
  virtual SbBool matches(const SoElement *) const;

  /** @copydoc SoLazyElement::copyMatchInfo() */
  virtual SoElement* copyMatchInfo() const;

  /** @copydoc SoLazyElement::registerRedundantSet */
  virtual void registerRedundantSet(SoState *, uint32_t bitmask);

 SoINTERNAL public:

  typedef SoGLLazyState GLLazyState;

  // Initializes the SoGLLazyElement class
  static void initClass();
  static void exitClass();

  // Sends indicated component(s) to GL:
  // Only sends if value is not already in GL.
  // note: has side effects, cannot really be const.
  // however will not necessarily cause cache dependency.
  void send(const SoState *state, uint32_t mask) const {
    if ((mask & invalidBits)||(state->isCacheOpen()))
      (const_cast<SoGLLazyElement*>(this))->reallySend(state, mask);
  }

  // Replace matches() for this element:
  // matches "this" element GL or IV state with values in eltInState
  SbBool lazyMatches(uint32_t checkGLFlag, uint32_t checkIVFlag,
                     const SoGLLazyElement *eltInState) {
    if (checkGLFlag || checkIVFlag ||
        ivState.transpType != eltInState->ivState.transpType)
      return fullLazyMatches(checkGLFlag, checkIVFlag, eltInState);
    else
      return TRUE;
  }
  
  /** Specify if blending can be set or none. If setOverrideBlending(true);
   * any call to setBlending() will be ignored.
   * If setOverrideBlending(false), setBlending() will behave as expected.
   * It can be usefull to override blending for example in some transparency algorithm that
   * require blending to be forced to some value and prevent node to change it. */
  void setOverrideBlending(bool value){ m_overrideBlending = value; }

  //
  bool isColorMaterialMode() const;

  /** Return the current GL material state */
  const GLLazyState& getGLState() const;

  // Replaces copyMatchInfo for this element:
  // makes a GLLazyElement with some initialization.
  // also initializes cacheSetBits to zero.
  SoGLLazyElement *copyLazyMatchInfo(SoState *state) const;

  // method that makes a copy of GL state at cache close().
  // goes from state="this" to cacheGLState
  // only copies if bitmask (GLSend of cache lazy element) is set.
  void getCopyGL(SoGLLazyElement *cacheLazyElement,
                 SoGLLazyElement::GLLazyState& cacheGLState);

  // method that copies GL state back into "this" element
  // after cache has been called.
  // only copies if bit in bitmask is set.
  // also sets invalidBits to FALSE for these components.
  void copyBackGL(SoGLLazyElement *cacheLazyElement,
                  SoGLLazyElement::GLLazyState& cacheGLState) {
    if (cacheLazyElement->GLSendBits)
      reallyCopyBackGL(cacheLazyElement->GLSendBits, cacheGLState);
  }

  // Send a packed color (not in state). Send transparency
  // as stipple if necessary.
  void sendVPPacked(SoState *state, const unsigned char *pcolor);

  // Note: destructor is public, so cache can delete its copy.
  virtual ~SoGLLazyElement();

  void mergeCacheInfo(SoGLRenderCache * childCache,
                      SoGLRenderCache *parentCache,
                      uint32_t doSendFlag,
                      uint32_t checkIVFlag,
                      uint32_t checkGLFlag);

  // Make copy of IV values into cacheLazyElement
  void copyIVValues(uint32_t bitmask, SoGLLazyElement *cacheLazyElement);

  //register a get() with the cache:
  virtual void registerGetDependence(SoState *, uint32_t bitmask);

 private:
  // Make copy of GL values into cacheLazyElement
  void copyGLValues(uint32_t bitmask, SoGLLazyElement *cacheLazyElement);

  // non-inline send when something has to be sent:
  void reallySend(const SoState *state, uint32_t bitmask);

  // private version of lazyMatches, does actual work when needed:
  SbBool fullLazyMatches(uint32_t checkGLFlag, uint32_t checkIVFlag,
                         const SoGLLazyElement* eltInState);

  // Pack the current diffuse and transparency into an SoColorPacker
  void packColors(SoColorPacker *cPacker);
  void packColors2(SoColorPacker *cPacker);

  // virtual set() methods that track GL state
  // as well as inventor state
  virtual void setDiffuseElt(SoNode *node, int32_t numColors,
                             const SbColor *colors,
                             SoColorPacker *cPacker);

  virtual void setTranspElt(SoNode *node, int32_t numTrans,
                            const float *trans,
                            SoColorPacker *cPacker);

  virtual void setTranspTypeElt(int32_t type);
  virtual void setPatternFlagElt(SbBool flag);
  virtual void setPackedElt(SoNode *node,
                            int32_t numColors,
                            const uint32_t *packedColors,
                            SoColorPacker *cPacker);

  virtual void setAmbientElt(const SbColor *color);
  virtual void setEmissiveElt(const SbColor *color);
  virtual void setSpecularElt(const SbColor *color);
  virtual void setShininessElt(float color);
  virtual void setColorMaterialElt(SbBool value);
  virtual void setBlendingElt(SbBool value);
  virtual void setSmoothingElt(SbBool value);
  virtual void setLightModelElt(SoState *, int32_t model);
  virtual void setMaterialElt(SoNode *, uint32_t bitmask,
                              SoColorPacker *cPacker,
                              const SoMFColor &, const SoMFFloat &,
                              const SoMFColor &, const SoMFColor &,
                              const SoMFColor &, const SoMFFloat &);
  virtual void setMaterialElt(SoNode *, uint32_t bitmask,
                              SoColorPacker *cPacker, SoMFColor *,
                              SoMFFloat *, SoMFColor *,
                              SoMFColor *, SoMFColor *, SoMFFloat *);
  virtual void setLineWidthElt(float lineWidth);
  
  //really does the copying of GL state (invoked by copyBackGL)
  void reallyCopyBackGL(uint32_t bitmask, SoGLLazyElement::GLLazyState &);
  
  // Copy of what has been sent to GL:
  SoGLLazyElement::GLLazyState* glState;

  // BitMap indicating what GL sends have been made:
  uint32_t GLSendBits;


  // Private storage associated with stipple patterns:
  // Holds defined 32x32 bit stipple patterns. Each is defined as 32
  // rows of 4 bytes (32 bits) each.
  static u_char patterns[64+1][32 * 4];

  // Indicates whether patterns were created and stored yet
  static SbBool patternsCreated;

  // Holds flags to indicate whether we defined a display list for
  // the corresponding stipple pattern
  static SbBool patternListDefined[64+1];

  // Stores base display list index for patterns
  static int patternListBase;

  // Stores cache context in which display lists were created
  static int patternListContext;

  // Set to TRUE in getInstance()??? if a cache is currently being
  // built, meaning we can't build a new display list
  SbBool cacheOpen;

  // Creates and sends the stipple pattern to GL
  void sendStipple(const SoState *state, int transpIndex);

  // Fills in "patterns" arrays with polygon stipples that simulate
  // transparency levels, using a standard dither matrix
  static void createPatterns();

  // hold envvar OIV_NEW_STIPPLE_BEHAVIOR
  static bool s_useNewStippleBehavior;

  // Sends the line width to GL
  void sendLineWidth(const SoState *state);

  // All glstate's nodeid (*not* ivstate's nodeid) must
  // be initialized with this value
  static uint64_t GLSTATE_INVALID_NODEID;

  // see SoPrefrences OIV_LINE_WIDTH var
  static bool s_forceLineWidth;

  /** If true, Blending should not be updated. */
  bool m_overrideBlending;
};

#endif /* _SO_GL_LAZY_ELEMENT */


