/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_SHAPE_HINTS_ELEMENT
#define  _SO_SHAPE_HINTS_ELEMENT

#include <Inventor/elements/SoReplacedElement.h>

/**
*   Stores the current shape hints.
*
* @ingroup elements
*
*   @DESCRIPTION
*   This element stores current shape hints.
*   All hints must be set at the same time; to
*   leave any hint as is, use the "AS_IS" enum value.
*
*   @SEE_ALSO
*   SoShapeHints
*/

SoEXTENDER_Documented class INVENTOR_API SoShapeHintsElement : public SoElement {

  SO_ELEMENT_HEADER(SoShapeHintsElement);

 public:
 /**
 *  Hints about ordering of face vertices.
 */
 enum VertexOrdering {
 /**
 *  No ordering info is known.
 */
 UNKNOWN_ORDERING,
 /**
 *  Vertices are ordered CW around faces.
 */
 CLOCKWISE,
 /**
 *  Vertices are ordered CCW around faces.
 */
 COUNTERCLOCKWISE,
 /**
 *  Indicates to set() to leave as is.
 */
 ORDERING_AS_IS
  };
 /**
 *  Hints about entire shape.
 */
 enum ShapeType {
 /**
 *  Nothing is known about shape.
 */
 UNKNOWN_SHAPE_TYPE,
 /**
 * Shape is known to be solid, and it defines the exterior of the volume it encloses.
 */
 SOLID,
 /**
 * Shape is known to be solid, and it defines the interior of the volume it encloses.
 * @ENUM_SINCE_OIV 2025.1
 */
  INNER_SOLID,
 /**
 *  Indicates to set() to leave as is.
 */
 SHAPE_TYPE_AS_IS
 };

 /**
 *  Hints about faces of shape.
 */
 enum FaceType {
 /**
 *  Nothing is known about faces.
 */
 UNKNOWN_FACE_TYPE,
 /**
 *  Faces are all convex.
 */
 CONVEX,
 /**
 *  Indicates to set() to leave as is.
 */
 FACE_TYPE_AS_IS
  };

 /**
 *  Hints about winding type
 */
 enum WindingType {
 /**
 *  No hole.
 */
    NO_WINDING_TYPE,
    ODD_TYPE,
    NON_ZERO_TYPE,
    POSITIVE_TYPE,
    NEGATIVE_TYPE,
    ABS_GEQ_TWO_TYPE
  };

  /**
  *  Sets the current shape hints in the state. Calls virtual
  *  method that subclasses can override.
  */
  static void         set(SoState *state, VertexOrdering vertexOrdering,
                          ShapeType shapeType, FaceType faceType);

  /**
  *  Sets the current shape hints in the state. Only for the winding type.
  *  Call virtual method that subclasses can override.
  */
  static void         set(SoState *state, WindingType windingType);

  /**
  *  Sets the current shape hints in the state. Only for VBO usage.
  */
  static void         set(SoState *state, SbBool useVBO) ;

  /**
   * Set shape hints by defining all options.
   */
  static void set(SoState *state, VertexOrdering vertexOrdering, ShapeType shapeType, FaceType faceType,
                  WindingType windingType, SbBool useVBO);

  /**
  *  Sets the current shape hints in the state.
  */
  static void         set(SoState *state, SoNode *,
                          VertexOrdering vertexOrdering,
                          ShapeType shapeType, FaceType faceType)
    { set(state, vertexOrdering, shapeType, faceType); }

  /**
  *
  */
  static void         set(SoState *state, SoNode *,WindingType windingType)
    { set(state,windingType); }

  /**
  *  Indicates if VBO is used from the state.
  */
  static void         set(SoState *state, SoNode *, SbBool useVBO)
    { set(state,useVBO); }

  /**
  *  Returns current shape hints from the state.
  */
  static void         get(SoState *state,
                          VertexOrdering &vertexOrdering,
                          ShapeType &shapeType, FaceType &faceType);

  /**
  *  Return current shape winding type from the state.
  */
  static void         get(SoState *state, WindingType &windingType );

  /**
  *  Indicates if VBO is used from the state.
  */
  static SbBool       isVBOUsed(SoState *state) ;

  /**
  *  Returns default hint.
  */
  static VertexOrdering getDefaultVertexOrdering()
  {
    return s_vertexOrderingDefault;
  }
  /**
  *  Returns default hint.
  */
  static ShapeType      getDefaultShapeType()
  {
    return s_shapeTypeDefault;
  }
  /**
  *  Returns default hint.
  */
  static FaceType       getDefaultFaceType()
  {
    return s_faceTypeDefault;
  }
  /**
  *  Returns default hint.
  */
  static WindingType    getDefaultWindingType()
  {
    return s_windingTypeDefault;
  }
  /**
  *  Returns default hint.
  */
  static SbBool         getDefaultVBOUsage()
  {
    return true;
  }

  /**
  *  Push copies the values from the previous element, so set
  *  doesn't have to set them if they are set AS_IS.
  */
  virtual void        push(SoState *state);

  /**
  *  Prints element (for debugging).
  */
  virtual void        print(FILE *fp) const;

protected:

  /** Initializes element */
  virtual void init(SoState *state);

  /**
  *  Returns TRUE if the hints match in both elements.
  */
  virtual SbBool matches(const SoElement* elt) const;

  /**
  *  Create and return a copy of this element.
  */
  virtual SoElement* copyMatchInfo() const;

 SoINTERNAL public:
  // Initializes the SoShapeHintsElement class
  static void         initClass();
  static void         exitClass();

  static bool isTwoSidedLighting(SoState *state);
  static void setForceTwoSidedLighting( SoState* state, bool forceTwoSidedLighting );

  /** Return true if we are doing back culling */
  static bool isFrontCulling( SoState* state );
  /** Return true if we are doing front culling */
  static bool isBackCulling( SoState* state );

  /** Return true if we are doing front culling */
  bool isFrontCulling() const;
  /** Return true if we are doing back culling */
  bool isBackCulling() const;

  /** Return true if we are doing front culling */
  static bool isFrontCulling( VertexOrdering vertexOrdering, ShapeType shapeType );
  /** Return true if we are doing back culling */
  static bool isBackCulling( VertexOrdering vertexOrdering, ShapeType shapeType );

  /** Return true if we are using CCW polygon */
  static bool isCounterClockwise(SoState *state);

  /** Return true if we are using CCW polygon */
  bool isCounterClockwise() const;

protected:
  /** @copydoc SoElement::commonInit() */
  virtual void commonInit();

  // Virtual set methods that subclasses can override.
  virtual void setElt(VertexOrdering vertexOrdering, ShapeType shapeType, FaceType faceType);

  virtual void setElt(WindingType windingType);

  // Set shape hints by defining all options.
  virtual void setElt(VertexOrdering vertexOrdering, ShapeType shapeType, FaceType faceType,
                      WindingType windingType, SbBool useVBO);

  VertexOrdering      vertexOrdering;
  ShapeType           shapeType;
  FaceType            faceType;
  WindingType         windingType;
  SbBool              useVBO ;

  virtual ~SoShapeHintsElement();

private:
  bool isTwoSidedLighting() const;

  bool m_forceTwoSidedLighting;

  static void initVertexOrdering();
  static void initShapeType();
  static void initFaceType();
  static void initWindingType();

  static VertexOrdering      s_vertexOrderingDefault;
  static ShapeType           s_shapeTypeDefault;
  static FaceType            s_faceTypeDefault;
  static WindingType         s_windingTypeDefault;
};

#endif /* _SO_SHAPE_HINTS_ELEMENT */

