/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Patrick Vigneras (Nov 1999)
**=======================================================================*/

#ifndef  _SO_TEXTURE_COORDINATE3_ELEMENT
#define  _SO_TEXTURE_COORDINATE3_ELEMENT

#include <Inventor/SbLinear.h>
#include <Inventor/elements/SoReplacedTextureElement.h>
#include <Inventor/STL/vector>

/**
 * Function that TextureCoordinateFunction nodes register to compute
 * texture coordinates.  Given the point and normal, compute a texture
 * coordinate and return it:
 *
 * @memberof SoTextureCoordinate3Element
 *
 * [OIV-WRAPPER NAME{FunctionCB}]
 */
typedef const SbVec4f &
SoTextureCoordinate3FunctionCB(void *userdata, const SbVec3f &point,
                               const SbVec3f &normal);

class SoTextureCoordinateElement;

/**
 * @VSGEXT Stores the current 3D texture coordinates.
 *
 * @ingroup elements
 *
 * @DESCRIPTION
 * This element stores the current 3D texture coordinates.
 *
 * @SEE_ALSO
 * SoTextureCoordinate3, SoVertexProperty
 */

SoEXTENDER_Documented class INVENTOR_API SoTextureCoordinate3Element : public SoReplacedTextureElement {

  SO_ELEMENT_HEADER(SoTextureCoordinate3Element);

public:
 /**
 *  The choice of values is for compatibility
 *  with Open Inventor 2.0 binary format files.
 */
 enum CoordType {
    EXPLICIT = 1,   // Coordinates stored in state
    FUNCTION = 2    // Coordinates generated by software function
  };

  /**
  *  Sets the current texture coordinates, in any of several ways:
  *  EXPLICIT, no coordinates (shapes will generate):
  */
  static void         setDefault(SoState *state, SoNode *node);

  // FUNCTION:
  static void         setFunction(SoState *state, SoNode *node,
                                  SoTextureCoordinate3FunctionCB *func,
                                  void *userData,
                                  SbBool frcSend = FALSE);
  
  // EXPLICIT:
  /**
   * [OIV-WRAPPER-ARG IN,IN,NO_WRAP{(coords != NULL? coords->Length: 0)},ARRAY,IN]
   */
  static void         set3(SoState *state, SoNode *node,
                           int32_t numCoords, const SbVec3f *coords,
                           SbBool frcSend = FALSE);

  /**
   * [OIV-WRAPPER-ARG IN,IN,NO_WRAP{(coords != NULL? coords->Length: 0)},ARRAY,IN]
   */
  static void         set4(SoState *state, SoNode *node,
                           int32_t numCoords, const SbVec4f *coords,                                                
                           SbBool frcSend = FALSE);

  /**
  *  Returns code indicating what has been set in state/element.
  * [OIVNET-WRAPPER NAME{GetCoordType}]
  */
  static CoordType    getType(SoState *state, int unit=0);
  /** [OIVNET-WRAPPER NAME{GetCoordType}] */
  virtual CoordType   getType(int unit=0) const;

  /**
  *  Returns the top (current) instance of the element in the state.
  *  The get routines are called on an instance because it is more
  *  efficient than calling a static method that looks up the
  *  element in the state for every coordinate.
  */
  static const SoTextureCoordinate3Element * getInstance(SoState *state);

  // Get routine for FUNCTION case:

  /**
  *  Given point and normal, returns texture coordinate. (The r
  *  coordinate will be 0.0 and the q will be 1.0 until we get 3D
  *  texture coordinate functions.)
  */
  const SbVec4f &     get(const SbVec3f &point, const SbVec3f &normal,
                          int unit=0) const;

  // Get routines for EXPLICIT case:

  /**
  *  Returns the number of coordinate points in an instance.
  */
  int32_t             getNum(int unit=0) const ;

  SbBool              is3D(int unit=0) const ;

  SbBool              isForceSending(int unit=0) const ;

  /**
  *  Returns the indexed coordinate from an element as a 3- or
  *  4-vector, converting if necessary.
  */
  const SbVec3f &     get3(int index, int unit=0) const;
  const SbVec4f &     get4(int index, int unit=0) const;

  /**
   *  Prints element (for debugging).
   */
  virtual void        print(FILE *fp) const;

  /**
  *  Overrides push() method to copy values from next instance in the stack.
  */
  virtual void push(SoState *) ;

protected:

  /** Common Initialization. Directly called by the constructor. */
  virtual void commonInit();

  /** Initializes element. */
  virtual void init(SoState *state);

 SoINTERNAL public:
  // Initializes the SoTextureCoordinate3Element class
  static void         initClass();
  static void         exitClass();

  struct SoTexCoord3Data {
    SoTexCoord3Data()
      :whatKind(EXPLICIT),
       numCoords(0),coords3(NULL),coords4(NULL),coordsAre3D(FALSE),forceSend(FALSE)
      {}


    SoTexCoord3Data& operator=(const SoTexCoord3Data& data)
      {
        whatKind = data.whatKind ;
        numCoords = data.numCoords ;
        coords3 = data.coords3 ;
        coords4 = data.coords4 ;
        coordsAre3D = data.coordsAre3D ;
        forceSend = data.forceSend;
        return *this ;
      }

    // members
    // What kind of coordinates will be done:
    CoordType           whatKind;

    // Storage for EXPLICIT:
    int32_t             numCoords;
    const SbVec3f       *coords3;
    const SbVec4f       *coords4;
    SbBool              coordsAre3D;
    SbBool              forceSend;
  } ;
  SoTexCoord3Data& getData(const size_t unit) const;

protected:

  virtual ~SoTextureCoordinate3Element();

private:
  void checkSize(const size_t size) const;
  //Needed for get method which doesn't have state we need
  SoState* m_state;
  mutable std::vector<SoTexCoord3Data> m_texCoord3Data ;


  SbVec3f             convert3;       // To convert from 4-D to 3-D
  SbVec4f             convert4;       // To convert from 3-D to 4-D
};

#endif /* _SO_TEXTURE_COORDINATE3_ELEMENT */


