/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_TEXTURE_COORDINATE_ELEMENT
#define  _SO_TEXTURE_COORDINATE_ELEMENT

#include <Inventor/SbLinear.h>
#include <Inventor/elements/SoReplacedTextureElement.h>
#include <Inventor/SbEnums.h>
#include <memory>

#include <Inventor/STL/vector>

/** Function that TextureCoordinateFunction nodes register to compute
 * texture coordinates. Given the point and normal, compute a texture
 * coordinate and return it:
 *
 * @memberof SoTextureCoordinateElement
 *
 * [OIV-WRAPPER NAME{FunctionCB}]
 */
typedef const SbVec4f &
SoTextureCoordinateFunctionCB(void *userdata, const SbVec3f &point,
                              const SbVec3f &normal);


/**
 * Stores the current texture coordinates.
 *
 * @ingroup elements
 *
 * @DESCRIPTION
 * This element stores the current texture coordinates.
 *
 * @SEE_ALSO
 * SoTextureCoordinate2
 */
SoEXTENDER_Documented class INVENTOR_API SoTextureCoordinateElement : public SoReplacedTextureElement {
  
  SO_ELEMENT_HEADER(SoTextureCoordinateElement);
  
 public:
  /**
  *  The choice of values is for compatibility
  *  with Open Inventor 2.0 binary format files.
  */
  enum CoordType {
    EXPLICIT = 1,     // Coordinates stored in state
    FUNCTION = 2      // Coordinates generated by software function
  };

  /**
  *  Sets the current texture coordinates, in any of several ways:
  *  EXPLICIT, no coordinates (shapes will generate).
  */
  static void         setDefault(SoState *state, SoNode *node);

  // FUNCTION:
  static void         setFunction(SoState *state, SoNode *node,
                                  SoTextureCoordinateFunctionCB *func,
                                  void *userData, 
                                  SbBool frcSend = FALSE);
  // EXPLICIT:
  /**
   * [OIV-WRAPPER-ARG IN,IN,NO_WRAP{(coords != NULL? coords->Length: 0)},ARRAY,IN]
   */
  static void         set2(SoState *state, SoNode *node,
                           int32_t numCoords, const SbVec2f *coords,
                           SbBool frcSend = FALSE);

  /**
   * [OIV-WRAPPER-ARG IN,IN,NO_WRAP{(coords != NULL? coords->Length: 0)},ARRAY,IN]
   */
  static void         set4(SoState *state, SoNode *node,
                           int32_t numCoords, const SbVec4f *coords,
                           SbBool frcSend = FALSE);
    
  /**
  * Returns code indicating what has been set in state/element.
  * [OIVNET-WRAPPER NAME{GetCoordType}]
  */
  static CoordType    getType(SoState *state, int unit=0);
  /** [OIVNET-WRAPPER NAME{GetCoordType}] */
  virtual CoordType   getType(int unit=0) const;
    
  /**
  *  Returns the top (current) instance of the element in the state.
  *  The get routines are called on an instance because it is more
  *  efficient than calling a static method that looks up the
  *  element in the state for every coordinate.
  */
  static const SoTextureCoordinateElement * getInstance(SoState *state);
    
  // Get routine for FUNCTION case:
    
  /**
  *  Given point and normal, returns texture coordinate. (The r
  *  coordinate will be 0.0 and the q will be 1.0 until we get 3D
  *  texture coordinate functions.)
  */
  const SbVec4f &     get(const SbVec3f &point, const SbVec3f &normal, 
                          int unit=0) const;
    
  // Get routines for EXPLICIT case:
    
  /**
  *  Returns the number of coordinate points in an instance.
  */
  int32_t             getNum(int unit=0) const ;
    
  SbBool              is2D(int unit=0) const ;

  SbBool              isForceSending(int unit=0) const ;
    
  /**
  *  Returns the indexed coordinate from an element as a 2- or
  *  4-vector, converting if necessary.
  */
  const SbVec2f &     get2(int index, int unit=0) const;
  const SbVec4f &     get4(int index, int unit=0) const;
    
  /**
   *  Prints element (for debugging).
   */
  virtual void        print(FILE *fp) const;

  /**
  *  Overrides push() method to copy values from next instance in the stack.
  */
  virtual void push(SoState *) ;

protected:

  /** Common Initialization. Directly called by the constructor. */
  virtual void commonInit();

  /** Initializes element. */
  virtual void init(SoState *state);

 SoINTERNAL public:
  /** Class that TextureCoordinateFunction nodes register to compute
   * texture coordinates. Given the point and normal, compute a texture
   * coordinate and return it.
   */
  class TexGenEvaluator
  {
  public:
    virtual const SbVec4f& eval(const SbVec3f &point, const SbVec3f &normal) = 0;
  };

  /** Set the type of corrds used */
  static void setTextureCoordType(SoState *state, SoNode *node, CoordType coordType);
  
  // Initializes the SoTextureCoordinateElement class
  static void         initClass();
  static void         exitClass();

  protected:

   virtual ~SoTextureCoordinateElement();
   
   struct SoTexCoordData 
   {
     // What kind of coordinates will be done:
     CoordType         whatKind;

     // Storage for FUNCTION:
     std::shared_ptr<TexGenEvaluator> m_texGenEvaluatorUserDefined;
     
     // Storage for EXPLICIT:
     int32_t           numCoords;
     const SbVec2f     *coords2;
     const SbVec4f     *coords4;
     SbBool            coordsAre2D;
     SbBool            forceSend;

     SoTexCoordData() :
      whatKind(EXPLICIT),
      numCoords(0),
      coords2(NULL),
      coords4(NULL),
      coordsAre2D(FALSE),
      forceSend(FALSE)
     {
     }

    void initValues( const CoordType &type )
    {
      whatKind = type;
      numCoords = 0;
      coords2 = NULL; 
      coords4 = NULL;
      coordsAre2D = FALSE;
      forceSend = FALSE;
      m_texGenEvaluatorUserDefined.reset();
    }

     SoTexCoordData& operator=(const SoTexCoordData& data)
     {
       if ( &data != this )
       {
         whatKind = data.whatKind ;
         numCoords = data.numCoords ;
         coords2 = data.coords2 ;
         coords4 = data.coords4 ;
         coordsAre2D = data.coordsAre2D ;
         forceSend = data.forceSend;
       }
       return *this ;
     }
   } ;

   SoTexCoordData& getTexCoordData( const int unit ) const;

 private:
  SbVec2f             m_convert2;       // To convert from 4-D to 2-D
  SbVec4f             m_convert4;       // To convert from 2-D to 4-D

  //Neeed for get method which don't have state we need
  SoState* m_state;
  
  typedef std::vector<SoTexCoordData> SoTexCoordDataList;

  mutable SoTexCoordDataList m_texCoordData ;
};


#endif /* _SO_TEXTURE_COORDINATE_ELEMENT */
