/*=================================================================================
 ***     THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),    ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
 ***                                                                            ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
 ***                                                                            ***
 ***                        RESTRICTED RIGHTS LEGEND                            ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
 ***                                                                            ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                    ***
 ***                              BORDEAUX, FRANCE                              ***
 ***                            ALL RIGHTS RESERVED                             ***
 =================================================================================*/

#ifndef  _SO_TEXTURE_ENABLED_ELEMENT
#define  _SO_TEXTURE_ENABLED_ELEMENT

#include <Inventor/elements/SoReplacedTextureElement.h>
#include <Inventor/STL/vector>

/**
 *  Enables/disables textures.
 *
 * @ingroup elements
 *
 *   @DESCRIPTION
 *   This element enables/disables textures. This is implemented as a
 *   separate element from the TextureImageElement so the texture image
 *   can be changed without invalidating caches containing shapes -- shapes
 *   need to know if texturing is going on or not so they can decide to
 *   send down texture coordinates or not.
 *
 *   This element is set by the GLTextureImageElement and the
 *   GLTextureImageElement, but the interaction between the three elements is a
 *   little complicated. Each of the other two elements always sets up
 *   the GL texture state appropriately, however, texturing is only
 *   enabled if both elements agree that it should be enabled (they
 *   check each other's value before calling set).
 *
 *   This element cannot be overridden, but, since the elements that
 *   set it check each other's value, overriding those elements has
 *   the same effect.
 *
 *   @SEE_ALSO
 *   SoTexture2, SoExtTexture2, SoImageBackground,
 *   SoTextureCubeMapEnabledElement, SoTexture3EnabledElement
 */
class INVENTOR_API SoTextureEnabledElement : public SoReplacedTextureElement
{

  SO_ELEMENT_HEADER( SoTextureEnabledElement );

public:
  /**
   *  Used by GLTextureImageElement, set to TRUE when a non-null
   *  image is set, set to FALSE when there is no texture image (the
   *  default).
   */
  static void set( SoState* state, SbBool value, SoNode *node = NULL );

  /**
   *  Used by shapes to figure out if they need to send texture
   *  coordinates.
   */
  static SbBool get( SoState* state, int unit = 0 );

  /**
   *  By default there is no texture.
   */
  static SbBool getDefault() { return FALSE; }

  /**
   *  Overrides push() so we can remember previous element's value and
   *  avoid making GL calls if this element has the same value.
   */
  virtual void push( SoState* state );

  /**
   *  Overrides pop() method so side effects can occur in GL.
   */
  virtual void pop( SoState* state, const SoElement* prevTopElement );

  /**
   * Specify that a texture unit is not used by any shape, so we don't have to invalidate cache when this texture is enable/disable/modified
   */
  static void disableForShapeCache( SoState* state, int unit, bool disable );

  /**
   * Return if that texture unit is not used by any shape, so we don't have to invalidate cache when this texture is enable/disable/modified
   */
  static bool isDisabledForShapeCache( SoState* state, int unit );

SoINTERNAL public:
  // Initializes the SoTextureEnabledElement class
  static void initClass();
  static void exitClass();

  virtual ~SoTextureEnabledElement();

protected:

  /** Common Initialization. Directly called by the constructor. */
  virtual void commonInit();

  /** Initializes element */
  virtual void init( SoState* state );

  bool isEnabled( int unit ) const;

  void setEnabled( int unit, bool enabled );

  typedef std::vector<bool> SoTextureUnitEnabledList;

  // Indicates for each texture unit if texturing is enabled.
  mutable SoTextureUnitEnabledList m_isEnabled;

  // Indicates for each texture init if this texture will be disable in shape cache system
  mutable std::vector<int> m_textureDisableForShapeCache;
};

#endif /* _SO_TEXTURE_ENABLED_ELEMENT */
