/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2025 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/


#ifndef  _SO_TEXTURE_IMAGE_ELEMENT_BASE
#define  _SO_TEXTURE_IMAGE_ELEMENT_BASE

#if defined(_WIN32)
#pragma warning(push)
#pragma warning(disable:4251) // 'identifier' : class 'type' needs to have dll-interface to be used by clients of class 'type2'
#endif

#define MAX_TEXTURE_NBR 6
// Texture1D, texture and texture3D use one texturename
// But we need 6 for textureCubeMap

#include <Inventor/SbColor.h>
#include <Inventor/elements/SoReplacedTextureElement.h>
#include <Inventor/fields/SoSFImage.h>
#include <Inventor/SbPImpl.h>
#include <Inventor/renderer/RendererResourceMacro.h>


#ifndef GL_TEXTURE_CUBE_MAP
#define GL_TEXTURE_CUBE_MAP 0x8513
#endif
#ifndef GL_TEXTURE_2D
#define GL_TEXTURE_2D 0x0DE1
#endif


class SoRenderToTextureProperty;
class SoBufferObject;
class SoGLTexture;
class SoTexture;

SO_PIMPL_BASE_PUBLIC_DECLARATION(SoTextureImageElementBase)

/**
*  Stores the current texture image.
* 
*  @ingroup elements
*
*   @DESCRIPTION
*   This element stores the current texture image.
* 
*   @SEE_ALSO
*   SoTexture2, SoExtTexture2, SoIndexedTexture2, SoTextureCubeMap,
*/

SoEXTENDER_Documented class INVENTOR_API SoTextureImageElementBase : public SoReplacedTextureElement
{ 
  SO_PIMPL_BASE_PUBLIC_HEADER(SoTextureImageElementBase);
 
  SO_ELEMENT_ABSTRACT_HEADER(SoTextureImageElementBase);

 public:
 
    /** Target */
   enum Target { 
    /** TEXTURE2D */
     TEXTURE2D = GL_TEXTURE_2D, 
    /** TEXTURECUBEMAP */
     TEXTURECUBEMAP = GL_TEXTURE_CUBE_MAP
   };
    
  /**
  *  Print info about image for debugging.
  */
  virtual void        print(FILE *fp) const;

  /**
  *  Overrides push() method to copy values from next instance in the stack.
  */
  virtual void push(SoState *state) ;

protected:
  /** Common Initialization. Directly called by the constructor. */
  virtual void commonInit();

  /** Initializes element. */
  virtual void init(SoState *state);

 SoINTERNAL public:
  // Initializes the class
  static void         initClass();
  static void         exitClass();

  SoTextureImageElementBase::Target getTargetTexture(){return m_target;} 

  struct TextureDataConfiguration
  {
    uint32_t xOffset;
    uint32_t yOffset;
    uint32_t zOffset;

    uint32_t width;
    uint32_t height;
    uint32_t depth;
    uint8_t face;

    uint8_t imgFormat;
    SbDataType imgDataType;

    uint32_t rowLength;
    uint32_t imageHeight;

    int mipLevel;
    void* imgData;

    /** 
     * If false, imgData won't be copied before being submitted to renderer.
     * Also, the renderer will become the owner of the pointer and will delete it when
     * consumed.
     */
    bool copyData : 1;

    /**
     * If true, the renderer will become the owner of the pointer and will delete it when
     * consumed.
     */
    bool deleteData : 1;

    TextureDataConfiguration()
    {
      xOffset = 0;
      yOffset = 0;
      zOffset = 0;
      width = 0;
      height = 0;
      depth = 0;
      face = 0;
      rowLength = 0;
      imageHeight = 0;
      mipLevel = 0;
      imgFormat = 0;
      imgDataType = SbDataType::UNKNOWN;
      imgData = NULL;
      copyData = true;
      deleteData = true;
    }

    /** Helper method to compute size in byte of the subimage data */
    size_t subImageSize() const;
  };
    
  class INVENTOR_API SamplerConfiguration : public SoRefCounter
  {
    RENDERER_RESOURCE(SamplerConfiguration);
  public:
    SbVec4f borderColor;
    float maxAnisotropy;

    uint32_t sAddressMode;
    uint32_t tAddressMode;
    uint32_t rAddressMode;

    uint32_t model;

    uint8_t minFilter;
    uint8_t magFilter;
    uint8_t mipmapFilter;

    /** Return filters to use according to the texture quality, using renderengine's enums */
    static uint8_t getAutoMagFilter(float quality);
    static void getAutoMinFilter(uint8_t& minFilter, uint8_t& mipmapFilter, float quality);

    bool isUsingMipmaps() const;
    bool operator== (const SamplerConfiguration& sc) const;
    bool operator!=( const SamplerConfiguration& sc ) const
    {
      return !operator==( sc );
    }

    SamplerConfiguration();
    ~SamplerConfiguration() { destroyRendererResource(); }

  private:
    SamplerConfiguration( const SamplerConfiguration& )
    : SoRefCounter()
    {}
  };
  
  // store information about the texture Image element data
  struct SoTexImageData
  {
    SoTexture* texture;
    SoRef<SamplerConfiguration> samplerConfiguration;
 
    // Constructor
    SoTexImageData();

    uint8_t texType;
    SoGLTexture*          gltexture;
    SbVec3i32             size;
    int                   numComponents;
    const void*           buffer[MAX_TEXTURE_NBR];
    SoBufferObject*       bufferObjects[MAX_TEXTURE_NBR];
    SbColor               blendColor;
    bool                  enableBorder;

    uint8_t imgFormat;
    SoSFImage::DataType   dataType;
    GLint                 GLInternalFormat;
    SoRenderToTextureProperty* renderToTextureProperty;
    bool                  renderToTexturePropertyFirstPass;
    // Does texture image contain any alpha values less than one?
    // -1 = unknown, 0 = no, 1 = yes
    int                   hasTransparency;
    bool                  isCompressed;
    int                   numCompressedMipmaps;
    float                 quality;
    int                   currentTexture;
    
    /** [OIV-WRAPPER-CLASS NO_WRAP] */
    struct SubTexture
    {
      int         textureName;
      SbVec3i32   size;
      SbVec3i32   offset;
      const void* bytes;    
    };
    std::vector<SubTexture> subtextures;

    //True if the texture is active
    bool enabled;

    int32_t wrapS;
    int32_t wrapR;
    int32_t wrapT;
    int32_t model;
    float maxAnisotropy;
    SbVec4f borderColor;
    uint8_t texFormat;

  };

  // return texture Image element data for the given texture unit
  SoTexImageData& getTexImageData(const int unit) const;

 protected:
  SoTextureImageElementBase::Target m_target;

  SbBool hasTransparency(SoState *state, int unit) const;
  void getElt(SoState *state, int &_minFilter,int &_magFilter, float &_maxAnisotropy, int unit) const;

  const void * getElt(SoState *state, SbVec2i32 &_size,
                      int &_numComponents, SoSFImage::DataType &_dataType,
                      GLint &_GLInternalFormat, int &_wrapS, int &_wrapT,
                      int &_model, SbColor &_blendColor, int unit, int currentTexture = -1) const;
  
  virtual void        setElt(const SbVec2i32 &s, int nc,
                             const void *buffer, SoBufferObject* bufferObject, SoSFImage::DataType dataType,
                             GLint GLInternalFormat, SoRenderToTextureProperty* renderToTex,
                             int wrapS, int wrapT, int wrapR, int model,
                             const SbColor &blendColor, float _maxAnisotropy,
                             SbBool _enableBorder, const SbVec4f &_borderColor, 
                             SbBool isCompressed, int numCompressedMipmaps,
                             int hasTransparency = -1 /* default unknown */, 
                             int unit=0, int currentTexture=-1);

  virtual void setElt(SoState *state, SoNode *node, const SoTexImageData& config);
    
  virtual void        setSubElt(const SbVec2i32 &subSize,
                                int xoffset, int yoffset,
                                const void *buffer, int texname,
                                int unit=0);

  virtual void        setFilterElt(int _minFilter, int _magFilter, int currUnit);


  virtual ~SoTextureImageElementBase();

  // This stores the list of node id's as pointers associated to
  // each texture unit.
  typedef std::vector<SoTexImageData> SoTexImageDataList;
  mutable SoTexImageDataList m_texImageDataList;

  friend class SoVRMLAppearance;
};

#ifdef _WIN32
#pragma warning( pop )
#endif

#endif /* _SO_TEXTURE_IMAGE_ELEMENT */


