/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_TEXTURE_MATRIX_ELEMENT
#define  _SO_TEXTURE_MATRIX_ELEMENT

#include <Inventor/SbLinear.h>
#include <Inventor/elements/SoAccumulatedElement.h>
#include <Inventor/STL/vector>

/**
*  Stores the current texture matrix.
* 
*  @ingroup elements
*
*   @DESCRIPTION
*   This element stores the current texture matrix - the cumulative
*   transformation applied to subsequent shapes. Because the matrix is
*   cumulative, this class is derived from SoAccumulatedElement. The
*   set() method replaces the current matrix, while all the others
*   (mult(), translateBy(), etc.) multiply into it. Node id's of the
*   nodes that affect the element are accumulated properly.
* 
*   @SEE_ALSO
*   SoTexture2Transform, SoTexture3Transform, SoTextureMatrix
*/

SoEXTENDER_Documented class INVENTOR_API SoTextureMatrixElement : public SoAccumulatedElement {
  
  SO_ELEMENT_HEADER(SoTextureMatrixElement);
  
 public:
  /**
  *  Overrides push() method to copy values from next instance in the stack.
  */
 virtual void        push(SoState *state);
    
  /**
  *  Sets the texture matrix to the identity matrix.
  */
  static void         makeIdentity(SoState *state, SoNode *node);
    

  /**
  *  Sets the texture transform matrix to the given matrix.
  */
  static void         set(SoState *state, SoNode *node, const SbMatrix &matrix);

  /**
  *  Multiplies the given matrix into the texture matrix.
  */
  static void         mult(SoState *state, SoNode *node,
                           const SbMatrix &matrix);
    
  /**
  *  Multiplies a matrix that performs the specified
  *  transformation into the texture matrix.
  */
  static void         translateBy(SoState *state, SoNode *node,
                                  const SbVec3f &translation);

  /**
  *  Multiplies a matrix that performs the specified
  *  transformation into the texture matrix.
  */
  static void         rotateBy(SoState *state, SoNode *node,
                               const SbRotation &rotation);

  /**
  *  Multiplies a matrix that performs the specified
  *  transformation into the texture matrix.
  */
  static void         scaleBy(SoState *state, SoNode *node,
                              const SbVec3f &scaleFactor);
    
  /**
  *  Returns current matrix from the state.
  */
  static const SbMatrix &     get(SoState *state, int unit=0);
    
  /**
  *  Prints element (for debugging).
  */
  virtual void        print(FILE *fp) const;

protected:

  /** Common Initialization. Directly called by the constructor. */
  virtual void commonInit();

  /** Initializes element */
  virtual void init(SoState *state);

 SoINTERNAL public:
  // Initializes the SoTextureMatrixElement class
  static void initClass();
  static void exitClass();
  static void emptyMatrix(SoState *state);

 protected:
  // Sets the matrix in an instance to identity
  virtual void        makeEltIdentity(int unit=0);
    
  // Multiplies into the matrix in an instance
  virtual void        multElt(const SbMatrix &matrix, int unit=0);
    
  // Each of these performs the appropriate operation on the matrix
  // in an instance
  virtual void        translateEltBy(const SbVec3f &translation, int unit=0);
  virtual void        rotateEltBy(const SbRotation &translation, int unit=0);
  virtual void        scaleEltBy(const SbVec3f &scaleFactor, int unit=0);
    
  // Sets the matrix in an instance to given matrix
  virtual void        setElt(const SbMatrix &matrix, int unit);

  // Gets the matrix from an instance
  virtual const SbMatrix &    getElt(int unit=0) const;
    
  virtual ~SoTextureMatrixElement();

  // return texture matrix for the given texture unit
  SbMatrix& getTexMatrix(const int unit) const;

private:
  // This stores the list of node id's as pointers associated to
  // each texture unit.
  typedef std::vector<SbMatrix> SbMatrixList;
  mutable SbMatrixList m_textureMatrixList;
};

#endif /* _SO_TEXTURE_MATRIX_ELEMENT */


