/*=================================================================================
 ***     THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),    ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                 ***
 ***                                                                            ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS   ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR   ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                       ***
 ***                                                                            ***
 ***                        RESTRICTED RIGHTS LEGEND                            ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS  ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN  ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT  ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN  ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.         ***
 ***                                                                            ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                    ***
 ***                              BORDEAUX, FRANCE                              ***
 ***                            ALL RIGHTS RESERVED                             ***
 =================================================================================*/

#ifndef  SO_UPDATE_AREA_ELEMENT
#define  SO_UPDATE_AREA_ELEMENT

#include <Inventor/SbLinear.h>
#include <Inventor/elements/SoSubElement.h>

/**
* Stores the rectangular area within the current viewport region that
* needs to be updated when rendering.
*
* @ingroup elements
*
*   @DESCRIPTION
*   This element stores the rectangular area within the current
*   viewport region that needs to be updated when rendering. It can be
*   used for partial rendering updates when applications know that
*   only a portion of the objects need to be rerendered. Cameras can
*   use the info in this element to set up a view volume against which
*   culling can be performed.
*
*   This element also sets the GL scissor box to limit drawing.
*
*   The update area is specified in normalized viewport coordinates,
*   where (0,0) is the lower left corner of the viewport and (1,1) is
*   the upper right corner. The area is given as an origin and a size.
*
*   @SEE_ALSO
*   PoBaseView, SoGLRenderAction
*/

class INVENTOR_API SoUpdateAreaElement : public SoElement
{

  SO_ELEMENT_HEADER( SoUpdateAreaElement );

public:
  /**
   * Sets the current update area in the state.
   */
  static void set( SoState* state, const SbVec2f& origin, const SbVec2f& size );

  /**
   * Returns current update area from the state. Returns TRUE if the
   * update area is the default, namely, the entire viewport.
   */
  static bool get( SoState* state, SbVec2f& origin, SbVec2f& size );

  /**
   * Returns the default update area origin.
   */
  static SbVec2f getDefaultOrigin() { return SbVec2f( 0.0f, 0.0f ); }

  /**
   * Returns the default update area size.
   */
  static SbVec2f getDefaultSize() { return SbVec2f( 0.0f, 0.0f ); }

  /**
   * Overrides push() method to copy values from next instance in the stack.
   */
  virtual void push( SoState* state );

  /**
   * Prints element (for debugging).
   */
  virtual void print( FILE* fp ) const;

SoINTERNAL public:
  static void initClass();
  static void exitClass();

protected:

  /** Initializes element */
  virtual void init( SoState* state );

  /**
   * Returns TRUE if the update areas match in both elements.
   */
  virtual SbBool matches( const SoElement* elt ) const;

  /**
   * Create and return a copy of this element.
   */
  virtual SoElement* copyMatchInfo() const;

  /**
   * Sets the update area parameters
   */
  virtual void setElt( SoState* state, const SbVec2f& origin, const SbVec2f& size );

  SbVec2f m_origin;
  SbVec2f m_size;

  virtual ~SoUpdateAreaElement();
};

#endif /* SO_UPDATE_AREA_ELEMENT */
