/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_MF_COLOR_
#define  _SO_MF_COLOR_

#include <Inventor/fields/SoSubField.h>
#include <Inventor/SbColor.h>
#include <Inventor/fields/SoMFVec3f.h>

//////////////////////////////////////////////////////////////////////////////
//
//  SoMFColor subclass of SoMField.
//
//////////////////////////////////////////////////////////////////////////////

/**
* Multiple-value field containing any number of RGB colors stored as three floats.
* 
* @ingroup fields
* 
* @DESCRIPTION
*   A multiple-value field that contains any number of RGB colors, stored as
*   instances of SbColor. Values may be set in either RGB (red, green, blue) or HSV
*   (hue, saturation, value) color spaces.
*   
*   SoMFColors are written to file as one or more RGB triples of floating point
*   numbers in standard scientific notation. When more than one value is present,
*   all of the values are enclosed in square brackets and separated by commas. For
*   example:
*   
*   \verbatim
     [ 1.0 0.0 0.0, 0 1 0, 0 0 1 ]
    \endverbatim
*    represents the three colors red, green, and blue.
*   
*   
*   @B Data copying:@b
*   \par
*   SoMF fields are a kind of "smart container", automatically expanding as necessary
*   to hold the data provided by the application. This is very convenient, but for large
*   blocks of data it may be desireable to avoid making a copy of the application data.
*   The \if_cpp setValuesPointer() \else \if_dotnet SetValuesBuffer() \else setValuesBuffer() \endif \endif
*   methods allow Open Inventor to directly use an array of values supplied by the application.
*   The application data is not copied.  Please see SoMFVec3f for more information and example code.
* 
* @SEE_ALSO
*    SbColor
* 
* 
*/
class INVENTOR_API SoMFColor : public SoMField {
  // Use standard field stuff
  SO_MFIELD_HEADER(SoMFColor, SbColor, const SbColor &);
  
  SO_MFIELD_SETVALUESPOINTER_HEADER(float);
  SO_FIELD_SUPPORT_MEMOBJ();
public:
  //
  // Additional convenience functions
  //
  
  /**
   * \if_java
   * Sets RGB values from the specified array of floats.
   * Each float should be in the range 0.0 to 1.0, and size of the array
   * must be a multiple of 3.
   * \else
   * Sets @B num@b RGB values from an array of arrays of 3 floats.
   * Each float should be in the range 0.0 to 1.0, and there must be 3*@B num@b floats
   * in the array.
   * \endif
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN,IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValues(start, num)}]
   * [OIVJAVA-WRAPPER-ARG IN,NO_WRAP{num},IN]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{int num = rgb != null ? rgb.length/3 : 0; onSetValues(start, num)}]
   */ 
  void setValues(int start, int num, const float rgb[][3]);
  /**
   * Sets @B num@b HSV values from an array of arrays of 3 floats.
   * Each float should be in the range 0.0 to 1.0, and there must be 3*@B num@b floats
   * in the array.
   * [OIV-WRAPPER-CUSTOM-CODE]
   */ 
  void setHSVValues(int start, int num, const float hsv[][3]);
  
  /** 
   * Sets the field to contain one and only one value, 
   * the given color (expressed as RGB floating point values
   * in the range 0.0 to 1.0), and deletes the second and subsequent
   * values.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */ 
  void setValue(const SbVec3f &vec);
  /** 
   * Sets the field to contain one and only one value, 
   * the given color (expressed as RGB floating point values
   * in the range 0.0 to 1.0), and deletes the second and subsequent
   * values.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */ 
  void setValue(float r, float g, float b);
  /** 
   * Sets the field to contain one and only one value, 
   * the given color (expressed as an array of RGB floating point values
   * in the range 0.0 to 1.0), and deletes the second and subsequent
   * values.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */ 
  void setValue(const float rgb[3]);
  
  /** 
   * Sets the field to contain one and only one value, 
   * the given color (expressed as HSV floating point values
   * in the range 0.0 to 1.0), and deletes the second and subsequent
   * values.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */ 
  void setHSVValue(float h, float s, float v);
  /** 
   * Sets the field to contain one and only one value, 
   * the given color (expressed as an array of HSV floating point values
   * in the range 0.0 to 1.0), and deletes the second and subsequent
   * values.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */ 
  void setHSVValue(const float hsv[3]);
  
  /** 
   * Sets one of N RGB colors from SbVec3f. 
   * The array will be expanded and filled with zeros as necessary.
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSet1Value(index)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSet1Value(index)}]
   */ 
  void set1Value(int index, const SbVec3f &vec);
  /** 
   * Sets one of N RGB colors from 3 floats.
   * The array will be expanded and filled with zeros as necessary.
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN,IN,IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSet1Value(index)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSet1Value(index)}]
   */ 
  void set1Value(int index, float r, float g, float b);
  /** 
   * Sets one of N RGB colors from an array of 3 floats.
   * The array will be expanded and filled with zeros as necessary.
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSet1Value(index)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSet1Value(index)}]
   */ 
  void set1Value(int index, const float rgb[3]);
  
  /** 
   * Sets one of N HSV colors from 3 floats.
   * The array will be expanded and filled with zeros as necessary.
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN,IN,IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSet1Value(index)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSet1Value(index)}]
   */ 
  void set1HSVValue(int index, float h, float s, float v);
  /** 
   * Sets one of N HSV colors from an array of 3 floats.
   * The array will be expanded and filled with zeros as necessary.
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSet1Value(index)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSet1Value(index)}]
   */ 
  void set1HSVValue(int index, const float hsv[3]);
  
 SoINTERNAL public:
  static void initClass();
  static void exitClass();
  virtual void writeBinaryValues(SoOutput*, int , int ) const;
  virtual SbBool readBinaryValues( SoInput*, int, int );

 private:
  virtual void writeBinaryValues(SoOutput *out) const;
  virtual SbBool readBinaryValues(SoInput *in, int numToRead);

  ///////////////////////EXCEPTION METHODS/////////////////////////////////////
  //
  // #unfound setValues(int start, int num, float rgb[][3])
  // #unfound setHSVValues(int start, int num, float hsv[][3])
  // #unfound setValue(SbVec3f & vec)
  // #unfound setValue(float red, float green, float blue)
  // #unfound setHSVValue(float hue, float saturation, float value)
  // #unfound setValue(float rgb[3])
  // #unfound setHSVValue(float hsv[3])
  // #unfound set1Value(index , SbVec3f & vec)
  // #unfound set1Value(index , float r, float g, float b)
  // #unfound set1HSVValue(index , float h, float s, float v)
  // #unfound set1Value(index , float rgb[3])
  // #unfound set1HSVValue(index , float hsv[3])
  // #unfound [](int i)
  // #unfound getValues(int start)
  // #unfound find(SbColor & targetValue, SbBool addIfNotFound)
  // #unfound setValues(int start, int num, SbColor * newValues)
  // #unfound set1Value(int index, SbColor & newValue)
  // #unfound =(SbColor & newValue)
  // #unfound setValue(SbColor & newValue)
  // #unfound ==(SoMFColor & f)
  // #unfound !=(SoMFColor & f)
  // #unfound startEditing()
  // #unfound finishEditing()
  // #unfound setValuesPointer(int num, SbColor * data)
  // #unfound enableDeleteValues()
  // #unfound isDeleteValuesEnabled()
  // #grouped setHSVValues(int start, int num, float hsv[][3])
  // #grouped setValue(float red, float green, float blue)
  // #grouped setHSVValue(float hue, float saturation, float value)
  // #grouped setValue(float rgb[3])
  // #grouped setHSVValue(float hsv[3])
  // #grouped set1Value(index , float r, float g, float b)
  // #grouped set1HSVValue(index , float h, float s, float v)
  // #grouped set1Value(index , float rgb[3])
  // #grouped set1HSVValue(index , float hsv[3])
  // #grouped getTypeId()
  // #grouped setValue(SbColor & newValue)
  // #grouped !=(SoMFColor & f)
  // #grouped finishEditing()
  // #grouped isDeleteValuesEnabled()
  //
  /////////////////////////////////////////////////////////////////////////////

};

#endif /* _SO_MF_COLOR_ */

