/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_MF_NODE_
#define  _SO_MF_NODE_

class SoNode;

#include <Inventor/fields/SoSubField.h>

//////////////////////////////////////////////////////////////////////////////
//
//  SoMFNode subclass of SoMField. Each field value is a pointer to an
//  SoNode. The field automatically maintains references to the nodes
//  it points to and propagates notification from the nodes.
//
//////////////////////////////////////////////////////////////////////////////

/**
* Multiple-value field containing any number of nodes.
* 
* @ingroup fields
* 
* @DESCRIPTION
*   This field maintains a set of SoNode instances\if_cpp , correctly
*   maintaining their reference counts \endif\.
*   
*   SoMFNodes are written to file as one or more nodes. When more than one value is
*   present, all of the values are enclosed in square brackets and separated by
*   commas; for example:
*   
*   \verbatim
     [ Cube {} , Sphere { radius 2.0 }, USE myTranslation ]
    \endverbatim
*    
*   
*   @B Data copying:@b
*   \par
*   SoMF fields are a kind of "smart container", automatically expanding as necessary
*   to hold the data provided by the application. This is very convenient, but for large
*   blocks of data it may be desireable to avoid making a copy of the application data.
*   The \if_cpp setValuesPointer() \else \if_dotnet SetValuesBuffer() \else setValuesBuffer() \endif \endif
*   methods allow Open Inventor to directly use an array of values supplied by the application.
*   The application data is not copied.  Please see SoMFVec3f for more information and example code.
* 
* 
*/
class INVENTOR_API SoMFNode : public SoMField {
  
  // Use standard field stuff
  SO_MFIELD_REQUIRED_HEADER(SoMFNode);
  SO_MFIELD_CONSTRUCTOR_HEADER(SoMFNode);
  SO_MFIELD_VALUE_HEADER(SoMFNode, SoNode *, SoNode *);
  SO_MFIELD_SETVALUESPOINTER_HEADER(SoNode *);
 
    /**
     * Adds a node as last one in group.
     * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
     * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
     */
    void addNode(SoNode *node);

  /**
   * Adds a node so that it becomes the one with the given index.
   * [OIVNET-WRAPPER-ARG IN,INDEX{0,Count}]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void insertNode(SoNode *node, int newNodeIndex);
  
  /**
  * Returns pointer to nth node node.
  */
  SoNode *getNode(int index) const;
  
  /**
   * Finds index of given node within group.
   */
  int findNode(const SoNode *node) const;
  
  /**
   * Returns number of nodes.
   */
  int getNumNodes() const;
  
  /**
   * Removes node with given index from group.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void removeNode(int index);
  
  /**
   * Removes first instance of given node from group
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void removeNode(SoNode *node)
    { removeNode(findNode(node)); }
  
  /**
   * Removes all nodes from group.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void removeAllNodes();
  
  /**
   * Replaces node with given index with new node.
   */
  void replaceNode(int index, SoNode *newNode);
  
  /**
   * Replaces first instance of given node with new node
   */
  void replaceNode(SoNode *oldNode, SoNode *newNode)
    { replaceNode(findNode(oldNode), newNode); }
  
  /**
   * Returns a pointer to the internally maintained array that can be modified.
   * The values in the array may be changed, but values cannot be added or removed. It is illegal to call any other
   * editing methods between startEditing() and finishEditing() (e.g. set1Value(), setValue(), etc.).
   *
   * Fields, engines or sensors connected to this field and sensors are not notified that this field has changed until
   * finishEditing() is called. Calling finishEditing() always sets the isDefault() flag to FALSE and informs engines
   * and sensors that the field changed, even if none of the values actually were changed.
   * [OIVNET-WRAPPER-RETURN-TYPE ARRAY{Count}]
   */
  SoNode** startEditing();

  /** Indicates that batch edits have finished. @see startEditing(). */
  void finishEditing();

 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  // Update a copied field to use the copy of each node if there is one
  virtual void fixCopy(SbBool copyConnections);
  
  // Override this to also check the stored nodes
  virtual SbBool referencesCopy() const;

private:
  // Override this to maintain write-references in nodes
  virtual void countWriteRefs(SoOutput *out) const;
  
  // Changes value in field without doing other notification stuff.
  // Keeps track of references and auditors.
  void setVal(int index, SoNode *newValue);

  SoNode** m_oldValues;
  int m_oldNum;

};

#endif /* _SO_MF_NODE_ */

