/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#ifndef  _SO_MF_UNIFORM_SHADER_PARAMETER_
#define  _SO_MF_UNIFORM_SHADER_PARAMETER_

#include <Inventor/fields/SoSubField.h>
#include <Inventor/nodes/SoShaderParameter.h>

/**
*
* @VSGEXT
*   Multiple-value field containing any number of SoUniformShaderParameter nodes.
*
* @ingroup fields
*
* @DESCRIPTION
*   This field maintains a set of SoUniformShaderParameter instances\if_cpp , correctly
*   maintaining their reference counts \endif\.
*
*   An SoMFUniformShaderParameter is written to file as one or more SoUniformShaderParameters.
*   When more than one value is
*   present, all of the values are enclosed in square brackets and separated by
*   commas. For example:
*
*   \verbatim
     [ ShaderParameter1f {
         name "shininess"
         value 20
       },

       ShaderParameter3f {
         name "eyePosition"
         value 0 0 0
       }
     ]
    \endverbatim
*
*   
*   @B Data copying:@b
*   \par
*   SoMF fields are a kind of "smart container", automatically expanding as necessary
*   to hold the data provided by the application. This is very convenient, but for large
*   blocks of data it may be desireable to avoid making a copy of the application data.
*   The \if_cpp setValuesPointer() \else \if_dotnet SetValuesBuffer() \else setValuesBuffer() \endif \endif
*   methods allow Open Inventor to directly use an array of values supplied by the application.
*   The application data is not copied.  Please see SoMFVec3f for more information and example code.
*
*
*/
class INVENTOR_API SoMFUniformShaderParameter : public SoMField {

  SO_MFIELD_REQUIRED_HEADER(SoMFUniformShaderParameter);
  SO_MFIELD_CONSTRUCTOR_HEADER(SoMFUniformShaderParameter);
  SO_MFIELD_VALUE_HEADER(SoMFUniformShaderParameter, SoUniformShaderParameter *, SoUniformShaderParameter *);
  SO_MFIELD_SETVALUESPOINTER_HEADER(SoUniformShaderParameter *);

  /**
   * Adds a uniform shader parameter as last one in group.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void addShaderParameter(SoUniformShaderParameter *shaderParameter);

  /**
   * Adds a uniform shader parameter so that it becomes the one with the given index.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void insertShaderParameter(SoUniformShaderParameter *shaderParameter, int newUniformShaderParameterIndex);

  /**
   * Returns the n'th uniform shader parameter node.
   * [OIVNET-WRAPPER-ARG INDEX{0,(Count-1)},IN]
   */
  SoUniformShaderParameter *getShaderParameter(int index) const;

  /**
   * Finds index of given uniform shader parameter within group.
   */
  int findShaderParameter(const SoUniformShaderParameter *shaderParameter) const;

  /**
   * Find the first parameter with the given name. Return NULL if not found.
   */
  SoUniformShaderParameter* findUniformParameterByName(const SbString& name);

  /**
   * Returns number of uniform shader parameters.
   */
  int getNumShaderParameters() const;

  /**
   * Removes uniform shader parameter with given index from group.
   * [OIVNET-WRAPPER-ARG INDEX{0,(Count-1)},IN]
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void removeShaderParameter(int index);

  /**
   * Removes first instance of given uniform shader parameter from group.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void removeShaderParameter(SoUniformShaderParameter *shaderParameter)
  { removeShaderParameter(findShaderParameter(shaderParameter)); }

  /**
   * Removes first instance of given uniform shader parameter with given name from group.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void removeShaderParameter(const SbString& name);

  /**
   * Removes all uniform shader parameters from group.
   * [OIVNET-WRAPPER-HELPER BEGIN{OnSetValue()}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetValue()}]
   */
  void removeAllShaderParameters();

  /**
   * Replaces uniform shader parameter with given index with new uniform shader parameter.
   * [OIVNET-WRAPPER-ARG INDEX{0,(Count-1)},IN]
   */
  void replaceShaderParameter(int index, SoUniformShaderParameter *newUniformShaderParameter);

  /**
   * Replaces first instance of given uniform shader parameter with new uniform shader parameter
   */
  void replaceShaderParameter(SoUniformShaderParameter *oldUniformShaderParameter,
                              SoUniformShaderParameter *newUniformShaderParameter)
  { replaceShaderParameter(findShaderParameter(oldUniformShaderParameter), newUniformShaderParameter); }

  /**
   * Returns a pointer to the internally maintained array that can be modified.
   * The values in the array may be changed, but values cannot be added or removed. It is illegal to call any other
   * editing methods between startEditing() and finishEditing() (e.g. set1Value(), setValue(), etc.).
   *
   * Fields, engines or sensors connected to this field and sensors are not notified that this field has changed until
   * finishEditing() is called. Calling finishEditing() always sets the isDefault() flag to FALSE and informs engines
   * and sensors that the field changed, even if none of the values actually were changed.
   * [OIVNET-WRAPPER-RETURN-TYPE ARRAY{Count}]
   */
  SoUniformShaderParameter** startEditing();

  /** Indicates that batch edits have finished. @see startEditing(). */
  void finishEditing();

  SoINTERNAL public:
  static void initClass();
  static void exitClass();

  // Update a copied field to use the copy of each uniform shader parameter if there is one
  virtual void fixCopy(SbBool copyConnections);

  // Override this to also check the stored uniform shader parameters
  virtual SbBool referencesCopy() const;

private:
  // Override this to maintain write-references in uniform shader parameters
  virtual void countWriteRefs(SoOutput *out) const;

  // Changes value in field without doing other notification stuff.
  // Keeps track of references and auditors.
  void setVal(int index, SoUniformShaderParameter *newValue);

  /**
   * Return true if param has the given name
   */
  static bool hasGivenName(SbString name, const SoShaderParameter* param);

  SoUniformShaderParameter** m_oldValues;
  int m_oldNum;
};

#endif /* _SO_MF_UNIFORM_SHADER_PARAMETER_ */

