/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_MFIELD_
#define  _SO_MFIELD_

#include <Inventor/fields/SoField.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoMField
//
//  Field that can have multiple values.
//
//////////////////////////////////////////////////////////////////////////////

/**
* Base class for all multiple-valued fields.
* 
* @ingroup fields
* 
* @DESCRIPTION
*   Each class derived from SoMField begins with an SoMF prefix and contains a
*   dynamic array of values of a particular type. Each has a setValues()
*   method that is passed an array of values of the correct type;
*   these values are copied into the array in the field, making extra room in the
*   array if necessary. 
*   \if_cpp 
*   The start and num parameters to this method indicate the
*   starting array index to copy into and the number of values to copy.
*   \else
*   The start parameter to this method indicates the
*   starting array index to copy into.
*   \endif
*   
*   The getValues() method for a multiple-value field returns a read-only 
*   array of values in the field.
*   \if_cpp 
*   In addition, the indexing operator "[ ]" is overloaded to return the @B i @b'th
*   value in the array; because it returns a const reference, it can be used only to
*   get values, not to set them.
*   \endif
*   \if_dotnet
*   In addition, the indexer operator [i] is implemented and allows setting/getting 
*   the @B i @b'th value in the array.
*   \endif
*   \if_java
*   In addition, getValueAt(i) returns the @B i @b'th value in the array.
*   \endif
*   
*   Methods are provided for getting the number of values in the field, inserting
*   space for new values in the middle, and deleting values.
*   
*   There are other methods that allow you to set only one value of several in the
*   field and to set the field to contain one and only one value.
*   
*   Two other methods can be used to make several changes to a multiple-value field
*   without the overhead of copying values into and out of the fields. The
*   startEditing() method returns the internal array of values in the field.
*   It can be used to change (but not add or remove) any values in the array. 
*   The finishEditing() method indicates that the editing is done and notifies 
*   any sensors or engines that may be connected to the field.
*   
*   SoMFields are written to file as a series of values separated by commas, all
*   enclosed in square brackets. If the field has no values (getNum() returns
*   zero), then only the square brackets ("[ ]") are written. The last value may
*   optionally be followed by a comma. Each field subtype defines how the values are
*   written; for example, a field whose values are integers might be written as:
*   
*   \verbatim
     [ 1, 2, 3, 4 ]
    \endverbatim
*    or:
*   
*   \verbatim
     [ 1, 2, 3, 4, ]
    \endverbatim
*    
* 
* @SEE_ALSO
*    SoNode,
*    SoEngine
* 
* 
*/
class INVENTOR_API SoMField : public SoField {

 public:
  // Destructor
#ifndef HIDDEN_FROM_DOC
  virtual ~SoMField();
#endif // HIDDEN_FROM_DOC

  /**
   * Returns the number of values currently in the field.
   * [OIVNET-WRAPPER PROPERTY{Count},VIRTUAL,GETTER,SINCE{9.3}]
   */
  int getNum() const { evaluate(); return num; }

  /**
  * Forces this field to have exactly @B num @b values, inserting or deleting
  * values as necessary.
  * If field's values are stored in an user data array and if specified number of values is
  * different from this array's length, a new array is allocated and the user's one is no more used.
  * [OIVNET-WRAPPER HELPER_BEGIN{OnSetNum(num)},PROPERTY{Count},VIRTUAL,SETTER,SINCE{9.3}]
  * [OIVJAVA-WRAPPER HELPER_BEGIN{onSetNum(num)}]
  */
  void setNum(int num);

  /**
   * Deletes @B num @b values beginning at index @B start @b (index @B start @b
   * through @B start @b + @B num @b -1 will be deleted, and any leftover values will
   * be moved down to fill in the gap created). A @B num @b of -1 means delete all
   * values from @B start @b to the last value in the field; getNum() will
   * return @B start @b as the number of values in the field after this operation
   * ( @B deleteValues(0, -1) empties the field @b ). However if @B num@b is greater
   * than or equal to the number of values in the field, @B start@b is ignored and
   * all values are deleted.
   *
   * If the field's values are stored in an user data array and if the number of values to delete is not zero,
   * a new array is allocated and the user data array is no longer used.
   * [OIVNET-WRAPPER-ARG INDEX{0,(Count-1)},IN]
   * [OIVNET-WRAPPER HELPER_BEGIN{OnDeleteValues(start, num)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onDeleteValues(start, num)}]
   */
  virtual void deleteValues(int start, int num = -1);

  /**
   * Inserts space for @B num @b values at index @B start @b. Index @B start @b
   * through @B start @b + @B num @b -1 will be moved up to make room. For example, to
   * make room for 7 new values at the beginning of the field call @B insertSpace(0,
   * 7) @b .
   * If field's values are stored in an user data array and if the number of values to insert is not zero,
   * a new array is allocated and the user's one is no more used.
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN]
   * [OIVNET-WRAPPER HELPER_BEGIN{OnInsertSpace(start, num)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onInsertSpace(start, num)}]
   */
  virtual void insertSpace(int start, int num);

  /**
   * This is equivalent to the set() method of SoField,
   * but operates on only one value. See the SoField methods for details.
   * If field's values are stored in an user data array and if specified index is greater than
   * this array's length, a new array is allocated and the user's one is no more used.
   * [OIVNET-WRAPPER HELPER_BEGIN{OnSet1Value(index)}]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSet1Value(index)}]
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE SbBool set1(int index, const char *valueString);


  /**
   * This is equivalent to the set() method of SoField,
   * but operates on only one value. See the SoField methods for details.
   * If field's values are stored in an user data array and if specified index is greater than
   * this array's length, a new array is allocated and the user's one is no more used.
   * [OIVNET-WRAPPER HELPER_BEGIN{OnSet1Value(index)}]
   * [OIVNET-WRAPPER-ARG INDEX{0,},IN]
   * [OIVJAVA-WRAPPER HELPER_BEGIN{onSet1Value(index)}]
   */
  SbBool set1(int index, const SbString& valueString);

  /**
   * This is equivalent to the get() method of SoField,
   * but operates on only one value. See the SoField methods for details.
   * [OIVNET-WRAPPER-ARG INDEX{0,(Count-1)},OUT]
   */
  void get1(int index, SbString &valueString);

  /**
   * Return the type identifier for this field class.
   */
  static SoType getClassTypeId();
  
 SoINTERNAL public:
  static void initClass();
  static void exitClass();

  static void *createInstance(SoType* dynamicType = NULL);
  
  // Data change notification
  int  getChangedStartIndex();
  int  getChangedNumValues();
  int  getChangedType();

  void valueChanged(int start = -1, int numValues = -1, SbBool resetDefault = TRUE);

  // Reads one indexed value of field from file
  virtual SbBool read1Value(SoInput *in, int index) = 0;
  
  // Writes indexed value to file
  virtual void write1Value(SoOutput *out, int index) const = 0;

  virtual void writeBinaryValues(SoOutput*, int , int ) const;
  virtual SbBool readBinaryValues(SoInput*, int , int );
  virtual void* getValuesPointer( int ) { return NULL; }

 protected:
  // Constructor
  SoMField();
  
  // Make sure there is room for newNum vals
  virtual void makeRoom(int newNum);

  // Notified changes
  void  resetChangedStatus();
  int   changedStartIndex;
  int   changedNumValues;

private:
  static SoType classTypeId;
  
  // Allocates room for num values. Copies old values (if any) into
  // new area. Deletes old area, if any.
  virtual void allocValues(int num) = 0;
  
  // Deletes all current values
  virtual void deleteAllValues() = 0;
  
  // Copies value indexed by "from" to value indexed by "to"
  virtual void copyValue(int to, int from) = 0;
  
  // Reads all values of field
  virtual SbBool readValue(SoInput *in);
  
  // Writes all values of field
  virtual void writeValue(SoOutput *out) const;
  
  // Reads array of binary values from file
  virtual SbBool readBinaryValues(SoInput *in, int numToRead);
  
  // Writes array of binary values to file
  virtual void writeBinaryValues(SoOutput *out) const;
  
  // Returns number of ASCII values to write per output line (default 1)
  virtual int   getNumValuesPerLine() const;
  
protected:
  SoMemoryObject* m_memObj;
};

#endif /* _SO_MFIELD_ */


