#if !defined(HIDDEN_FROM_DOC)
#if !defined _SB_CONST_CHAR_MAP_H_
#define _SB_CONST_CHAR_MAP_H_
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2020 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Dec 2012)
**=======================================================================*/
#include <Inventor/STL/map>
#include <Inventor/STL/functional>

/**
 * Internal use only. Specialization of std::map<const char*,value_type>
 * key is a const char* instead of a std::string (for performance purpose)
 * it manage copy/comparaison/cleanup of string contained in key pointer.
 */
template <typename T>
SoINTERNAL class SbConstCharMap
{
public:
  /** Default constructor */
  SbConstCharMap()
  {}

  /** Destructor */
  ~SbConstCharMap()
  { clear(); }

  /** compare func for 2 const char * */
  struct StrCompare {
    bool operator() (const char* str1, const char* str2) const
    {
      if ( str1 == str2 )
        return true;
      return strcmp(str1, str2) < 0;
    }
  };

  /** usual typedef for std::map */
  typedef typename std::map<const char*, T , StrCompare> UniformMap;
  typedef typename UniformMap::value_type value_type;
  typedef typename UniformMap::iterator iterator;
  typedef typename UniformMap::const_iterator const_iterator;

  /** wrap std::map<>::begin() */
  const_iterator begin() const
  { return m_uniforms.begin(); }

  /** wrap std::map<>::begin() */
  iterator begin()
  { return m_uniforms.begin(); }

  /** wrap std::map<>::end() */
  const_iterator end() const
  { return m_uniforms.end(); }

  /** wrap std::map<>::end() */
  iterator end()
  { return m_uniforms.end(); }

  /** wrap std::map<>::empty() */
  bool empty() const
  { return m_uniforms.empty(); }

  /** wrap std::map<>::size() */
  size_t size() const
  { return m_uniforms.size(); }

  /** wrap std::map<>::find() */
  iterator find(const char* key)
  { return m_uniforms.find(key); }

  /** wrap std::map<>::find() */
  const_iterator find(const char* key) const
  { return m_uniforms.find(key); }

  /** release an element (release strdup memory) */
  iterator erase (iterator it)
  {
    free((void*)it->first);
    return m_uniforms.erase(it);
  }

  /** Clear the map (release strdup memory) */
  void clear()
  {
    iterator it = m_uniforms.begin();
    while(it!=m_uniforms.end())
    {
      free((void*)it->first);
      ++it;
    }
    m_uniforms.clear();
  }

  /** return element if it exists else create it with (strdup of key) */
  T& operator[] ( const char* name )
  {
    iterator it = find(name);
    if ( it==end())
    {
#ifdef _WIN32
      return m_uniforms[_strdup(name)];
#else
      return m_uniforms[strdup(name)];
#endif
    }
    else
      return ((*it).second);
  }

  /** Copy container content */
  SbConstCharMap<T>& operator=(const SbConstCharMap<T>& copyFrom)
  {
    // first clear existing entry
    clear();
    // Then insert each element
    iterator insertedIt = begin();
    const_iterator it = copyFrom.begin();
    while(it!=copyFrom.end())
    {
#ifdef _WIN32
      insertedIt = m_uniforms.insert(insertedIt,value_type(_strdup((*it).first),(*it).second));
#else
      insertedIt = m_uniforms.insert(insertedIt,value_type(strdup((*it).first),(*it).second));
#endif
      ++it;
    }
    return *this;
  }

  /** wrap std::map<>::operator!= */
  friend int operator !=(const SbConstCharMap<T>& map1, const SbConstCharMap<T>& map2)
  { return !(map1==map2); }

  /** wrap std::map<>::operator== */
  friend int operator ==(const SbConstCharMap<T>& map1, const SbConstCharMap<T>& map2)
  {
    if ( map1.size() != map2.size() )
      return false;

    const const_iterator itEnd1 = map1.end();
    const_iterator it1 = map1.begin();
    const_iterator it2 = map2.begin();
    for (; it1 != itEnd1; ++it1, ++it2)
    {
      if ( it1->second != it2->second )
        return false;

      // ideally, we should do if ( StrCompare(map1, map2) || StrCompare(map2, map1) ) return false;
      // but this would need 2 string comparison. Hard code strcmp for perf reason.
      else if ( strcmp(it1->first, it2->first) != 0 )
        return false;
    }
    return true;
  }

private:
  // internal std::map<> storage
  UniformMap m_uniforms;
};

#endif //_SB_CONST_CHAR_MAP_H_
#endif // HIDDEN_FROM_DOC
