/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : Benjamin GRANGE (Dec 2008)
**=======================================================================*/


#ifndef SB_GL_CONTEXT_HELPER_H
#define SB_GL_CONTEXT_HELPER_H

#include <Inventor/SbBase.h>
#include <vector>

#ifndef HIDDEN_FROM_DOC

#if defined(OIV_HEADLESS)
typedef void *EGLDisplay;
typedef void *EGLConfig;
typedef void *EGLSurface;
typedef void *EGLContext;
#elif defined(_WIN32)
#elif defined(__APPLE__)
typedef struct _CGLContextObject *CGLContextObj;
typedef struct _CGLPixelFormatObject *CGLPixelFormatObj;
typedef unsigned int CGDirectDisplayID;
#else
typedef struct _XDisplay Display;
typedef unsigned long GLXDrawable;
typedef unsigned long GLXPixmap;
typedef struct __GLXcontextRec *GLXContext;
typedef unsigned long GLXFBConfigID;
typedef unsigned long GLXWindow;
typedef unsigned long GLXPbuffer;
typedef unsigned long Colormap;
typedef unsigned long Pixmap;
typedef unsigned long Window;
#if !defined(BUILDING_SB_GL_CONTEXT_HELPER)
typedef struct __GLXFBConfigRec *GLXFBConfig;
#endif
#endif

#endif //HIDDEN_FROM_DOC

class SoGLContext;

/**
 * @ingroup InventorHelpers
 *
 * Class encapsulating information about an OpenGL context:
 * a handle on an OpenGL context and its id.
 * [OIVJAVA-WRAPPER-CLASS MODULE{INVENTOR}]
 * [OIVNET-WRAPPER-CLASS CUSTOM_CODE]
 */
class SbGLShareContext
{
public:
  /** Handle on OpenGL context */
  SoGLContext *shareContext;
  /**
   * Corresponding shared context Id
   * managed by upper layer viewer class.
   */
  int shareContextID;

SoINTERNAL public:

  /** [OIVJAVA-WRAPPER VISIBILITY{Public}] */
  SbGLShareContext();

  /** [OIVJAVA-WRAPPER VISIBILITY{Public}] */
  SbGLShareContext(SoGLContext *shareContext, int shareContextID);

  /** [OIVJAVA-WRAPPER VISIBILITY{Public}] */
  SoGLContext * getContext();

  /** [OIVJAVA-WRAPPER VISIBILITY{Public}] */
  int getContextGroupId();
};


inline
SbGLShareContext::SbGLShareContext()
{
  shareContext = NULL;
  shareContextID = 0;
}

inline 
SbGLShareContext::SbGLShareContext(SoGLContext* _shareContext, int _shareContextID)
{
  shareContext = _shareContext;
  shareContextID = _shareContextID;
}

inline SoGLContext * 
SbGLShareContext::getContext()
{
  return shareContext;
}

inline int
SbGLShareContext::getContextGroupId()
{
  return shareContextID;
}

/**
 * Platform indepent functions for OpenGL context handling
 */
namespace SbGlContextHelper
{
#if defined(OIV_HEADLESS)
  typedef EGLDisplay Display;
  typedef EGLContext GLContext;
  typedef EGLConfig VisualInfo;
  typedef EGLSurface Drawable;
#elif defined(_WIN32)
  typedef HDC Display;
  typedef HGLRC GLContext;
  typedef PIXELFORMATDESCRIPTOR* VisualInfo;
  typedef int Drawable;
#elif defined(__APPLE__)
  #undef Display
  typedef int Display;
  typedef CGLContextObj GLContext;
  typedef CGLPixelFormatObj VisualInfo;
  typedef void* Drawable;
#else
  typedef ::Display* Display;
  typedef GLXContext GLContext;
  typedef void* VisualInfo;
  typedef GLXDrawable Drawable;
#endif

  /** Return the current active context */
  INVENTORGL_API GLContext getCurrentContext();

  /** Return the current active display */
  INVENTORGL_API Display getCurrentDisplay();

  /** Return the current active drawable */
  INVENTORGL_API Drawable getCurrentDrawable();

  /** Return true if the specified display is valid */
  INVENTORGL_API bool isValidDisplay(Display dpy);

  /** 
   * Returns the VisualInfo used for the specified drawable.
   * Use releaseVisualInfo() to free the returned pointer.
   */
  
  INVENTORGL_API VisualInfo getWindowVisualInfo(Display dpy, Drawable drawable);

  /**
  * Returns the VisualInfo used for the specified drawable.
  * Use releaseVisualInfo() to free the returned pointer.
  */
  INVENTORGL_API VisualInfo getWindowVisualInfo(Display dpy, Drawable drawable, GLContext ctx);

  
  /** Release a VisualInfo returned by getWindowVisualInfo() */
  INVENTORGL_API void releaseVisualInfo(VisualInfo visualInfo);

  /**
   * Create a new context
   * @param dpy specifies the display (device context on win32)
   * @param vis specifies the visual (must be set to 0 on win32)
   * @param shareList context to sharelist with (0 means no sharing); it is not used on win32, use shareContext for sharing.
   * @param direct set to true for direct rendering (meaningless on win32)
   */
  INVENTORGL_API GLContext createContext(Display dpy, VisualInfo vis, GLContext shareList, 
                                         bool direct);

  /** Destroy given context */
  INVENTORGL_API bool destroyContext(Display dpy, GLContext ctx);

  /**
   * Make given context current
   * @param dpy specifies the display (device context on win32)
   * @param drawable specifies a drawable (0 on win32)
   * @param ctx specifies the OpenGL context to switch to.
   */
  INVENTORGL_API bool makeCurrent(Display dpy, Drawable drawable, GLContext ctx);

  /**
   * Create a new context
   * @param dpy specifies the display (device context on win32)
   * @param vis specifies the visual (must be set to 0 on win32)
   * @param shareList context to sharelist with (0 means no sharing); it is not used on win32, use shareContext for sharing.
   * @param direct set to true for direct rendering (meaningless on win32)
   * @param attribs holds options for context creation
   */
  SoINTERNAL INVENTORGL_API GLContext createContextAttribs(Display dpy, VisualInfo vis, GLContext shareList,
                                                           bool direct, const std::vector<int> &attribs);
}

#endif


