/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : C. OGNIER (Jun 2003)
**=======================================================================*/

#ifndef _SBRASTERIMAGE_
#define _SBRASTERIMAGE_


#include <SoDeprecationRules.h>
#include <Inventor/SbLinear.h>
#include <Inventor/sys/SoGLType.h>
#include <Inventor/SbDataType.h>
#include <Inventor/devices/SoBufferObject.h>
#include <Inventor/devices/SoCpuBufferObject.h>

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

/**
 *
 *
 * @VSGEXT Class encapsulating a raster image and its attributes.
 * 
 * @ingroup Image
 *
 * This class encapsulates a raster image, including its size and 
 * number of color components.
 *
 * 
 * @SEE_ALSO
 *    SoRasterImageRW
 * 
 * 
 */
class INVENTORIMAGE_API SbRasterImage {


public:

  /**
   * Components type.
   */
  enum Components {
    /**
     *  Unknown number of components.
     */
    UNKNOWN = -1,
    
    /**
     *  Number of components 1 (luminance).
     */  
    LUMINANCE = 1,

    /**
     *  Number of components 2 (luminance + transparency).
     */  
    LUMINANCE_TRANSPARENCY = 2,

    /**
     *  Number of components 3 (RGB) (default). 
     */  
    RGB = 3, // The default
    
    /**
     *  Number of components 4 (RGB + transparency).
     */  
    RGB_TRANSPARENCY = 4,

    /**
     *  Number of components 4 (RGBE).
     *  This is used by some HDR formats.
     */  
    RGBE = 5
  };


  /**
   * Copy policy.
   */
  enum CopyPolicy {
    /**
     *  Open Inventor will make a copy of the data (default). 
     */
    COPY = 0,
	/**
	 *  Passed buffer used\if_cpp, user will delete \endif\.
	 */
	NO_COPY = 1,
    /**
     *  Passed buffer used, SbRasterImage will delete.
     * Use this if memory is allocated with "new". 
     */
    NO_COPY_AND_DELETE = 2,
    /**
     *  Passed buffer used, SbRasterImage will free.
     * Use this if memory is allocated with "malloc". 
     */
    NO_COPY_AND_FREE = 3
  };


  /**
   * Constructor that creates an empty image.
   */
  SbRasterImage();

  /**
   * Constructor that initializes with an image.
   * Image data is assumed to be packed (no padding between pixels or rows)
   * and pixel components are assumed to be in RGBA order.
   *
   * \if_cpp
   * Given some image data for which we have:
   *   - imageSize  : Dimensions of image
   *   - imageBPP   : Number of bytes per pixel (1, 2, 3, ...)
   *   - imageComps : Type of pixel (L, LA, RGB, RGBA, ...)
   *   - imageData  : Pointer to image data
   *
   * Case 1: Create raster image from image data, retaining ownership of memory.
   * \par
   * \code
   *   size_t numBytes = imageSize[0] * imageSize[1] * imageBPP;
   *   SoRef<SoCpuBufferObject> buffer = new SoCpuBufferObject( (void*)imageData, numBytes );
   *   SbRasterImage* rasterImage = new SbRasterImage( imageSize, imageComps, buffer.ptr() );
   * \endcode
   *
   * Case 2:
   * Create raster image from image data, making a copy of the data.
   * \par
   * \code
   *   SoRef<SoCpuBufferObject> buffer = new SoCpuBufferObject();
   *     size_t numBytes = imageSize[0] * imageSize[1] * imageBPP;
   *     buffer->setSize( numBytes ); // Allocates memory owned by buffer object
   *     memcpy( buffer->map( SoBufferObject::SET ), imageData, numBytes );
   *     buffer->unmap();
   *   SbRasterImage* rasterImage = new SbRasterImage( imageSize, imageComps, buffer.ptr() );
   * \endcode
   * \endif
   */
  SbRasterImage( const SbVec2i32 &size, Components components,
                 SoBufferObject* bufferObject );

  /**
   * Destructor.
   */
  virtual ~SbRasterImage();
 
  /**
   * Sets the raster image size in pixels. @BR
   * The parameter is the size of the whole raster image. 
   * The maximum size that can be specified using this method is
   * 32767 by 32767 (because the parameter is an SbVec2s).  To
   * specify a larger image size use the method setSize_i32.
   */
  void setSize(const SbVec2s& size);

   /**
   * Sets the raster image size in pixels. @BR
   * The parameter is the size of the whole raster image to be saved.
   * Use for sizes with at least one side greater than 32767.
   */
  void setSize_i32(const SbVec2i32& size);

  /**
   * Returns the raster image size in pixels.
   */
  SbVec2s getSize() const;

  /**
   * Returns the raster image size in pixels for large images.
   */
  SbVec2i32 getSize_i32() const;

  /**
   * Sets the number of components in each pixel of the raster image. @BR
   * For example, an RGB image has 3 components.
   */
  void setComponents(SbRasterImage::Components comp);

  /**
   * Returns the number of components of each pixel as an enum. @BR
   * For example, an RGBA image returns RGB_TRANSPARENCY.
   */
  SbRasterImage::Components getComponents() const;

  /**
   * Returns the size in bytes of each pixel in the image. @BR
   * Returns -1 in case of invalid or unknown components type.
   *
   * E.g: For an RGB image containing byte data, the pixel size is 3 bytes.
   */
  int getComponentsSize() const;

  /**
   * Returns the number of components of each pixel. @BR
   * Returns -1 in case of invalid or unknown components type.
   *
   * E.g: For an RGB image it returns 3 (R, G and B).
   */
  int getComponentsCount() const;

  /**
   * Sets the pixel buffer of the raster image. @BR
   * Image data is assumed to be packed (no padding between pixels or rows)
   * and pixel components are assumed to be in RGBA order.
   *
   * \if_cpp
   * Given some image data for which we have:
   *   - imageSize  : Dimensions of image
   *   - imageBPP   : Number of bytes per pixel (1, 2, 3, ...)
   *   - imageData  : Pointer to image data
   *
   * Case 1: Set image data, retaining ownership of memory.
   * \par
   * \code
   * SbRasterImage* rasterImage . . .
   *   size_t numBytes = imageSize[0] * imageSize[1] * imageBPP;
   *   SoRef<SoCpuBufferObject> buffer = new SoCpuBufferObject( (void*)imageData, numBytes );
   * rasterImage->setBuffer( buffer.ptr() );
   * \endcode
   *
   * Case 2: Set image data, making a copy of image data.
   * \par
   * \code
   * SbRasterImage* rasterImage . . .
   * SoRef<SoCpuBufferObject> buffer = new SoCpuBufferObject();
   *   size_t numBytes = imageSize[0] * imageSize[1] * imageBPP;
   *   buffer->setSize( numBytes );  // Allocates memory owned by buffer object
   *   memcpy( buffer->map( SoBufferObject::SET ), (void*)imageData, numBytes );
   *   buffer->unmap();
   * rasterImage->setBuffer( buffer.ptr() );
   * \endcode
   * \endif
   */
  void setBuffer(SoBufferObject* bufferObject);
  
  /**
   * Returns the pixel buffer of the raster image.
   * \if_cpp
   * To get a pointer to the raw image data:
   * \par
   * \code
   * SbRasterImage* rasterImage . . . 
   * SoBufferObject* imageBuffer = rasterImage->getBufferObject();
   * SoRef<SoCpuBufferObject> cpuBuffer = new SoCpuBufferObject();
   * imageBuffer->map( cpuBuffer.ptr(), SoBufferObject::READ_ONLY);
   *
   *   unsigned char* imageData = (unsigned char*)cpuBuffer->map(SoBufferObject::READ_ONLY);
   *   . . .
   * cpuBuffer->unmap();
   * imageBuffer->unmap(cpuBuffer.ptr());
   * \endcode
   * \endif
   */
  SoBufferObject* getBufferObject() const;

  /**
   * 
   * Returns the number of mipmaps contained by the buffer.
   */ 
  int getNumMipmaps() {return m_numMipmaps;};

  /**
   * Returns the current buffer copy policy.
   */
  CopyPolicy getCopyPolicy() const;

  /** 
   * Removes the Alpha channel from the current raster image.
   * If the copy policy is COPY (Open Inventor owns the memory) then the buffer
   * is reallocated with the correct size.
   * Otherwise the content of the buffer is just modified. 
   * 
   * @return True if the format contains an alpha channel and it was possible to remove it.
   */
  SbBool removeAlphaChannel();

#if SoDEPRECATED_BEGIN(9400)
  /**
   * Constructor that initializes with an image.
   */
  SoDEPRECATED_METHOD(9400,"Use SbRasterImage constructor with SoBufferObject instead.")
  SbRasterImage( const SbVec2s &size, Components components,
                 const unsigned char *bytes, CopyPolicy copy=COPY );
  /**
   * Constructor that initializes with a large image.
   */
  SoDEPRECATED_METHOD(9400,"Use SbRasterImage constructor with SoBufferObject instead.")
  SbRasterImage( const SbVec2i32 &size, Components components,
                 const unsigned char *bytes, CopyPolicy copy=COPY );

  /**
   * Sets the pixel buffer of the raster image. @BR
   * By default, the memory policy is COPY.
   * One of the NO_COPY options is the most likely to be efficient.
   * For interoperability with other classes, pixel buffer must be packed and
   * must respect the RGB ordering.
   */
  SoDEPRECATED_METHOD(9400,"Use SbRasterImage::setBuffer(SoBufferObject* bufferObject) instead.")
  void setBuffer(unsigned char* buffer, CopyPolicy copy = COPY);
  
  /**
   * Returns the pixel buffer of the raster image.
   * [OIV-WRAPPER-RETURN-TYPE ARRAY{GetSize_i32().Value[0]*GetSize_i32().Value[1]*((int)GetComponentsCount())}]
   * [OIVJAVA-WRAPPER-RETURN-TYPE ARRAY{getSize_i32()[0]*getSize_i32()[1]*(int)getComponentsCount()}]
   */
  SoDEPRECATED_METHOD(9400,"Use SbRasterImage::getBufferObject() instead.")
  unsigned char* getBuffer() const;

#endif /** @DEPRECATED_END */

SoINTERNAL public:

#if SoDEPRECATED_BEGIN(9400)
  /**
   * Constructor that initializes with an image, with an SoSFImage::DataType parameter.
   */
  SoDEPRECATED_METHOD(9400,"Use SbRasterImage constructor with CpuBufferObject instead.")
  SbRasterImage( const SbVec2i32 &size, Components components, SbDataType::DataType dataType,
                 const void *bytes, CopyPolicy copy=COPY );

#endif /** @DEPRECATED_END */

  /**
   * Constructor that initializes with an image, with an SoSFImage::DataType parameter.
   */
  SbRasterImage( const SbVec2i32 &size, Components components, SbDataType::DataType dataType,
                 SoBufferObject* bufferObject );


  void setNumMipmaps(int nmm) {m_numMipmaps = nmm;};

  void setDataType(SbDataType::DataType dataType) {m_dataType = dataType;};
  SbDataType::DataType getDataType() const
  {
    return m_dataType;
  }

  void setCompressedInternalFormat(uint8_t intFormat) {m_compressedInternalFormat = intFormat;};
  uint8_t getCompressedInternalFormat() const
  {
    return m_compressedInternalFormat;
  }

  /*
  * Copy a sub image inside this image at the given position.
  * If the the size of the sub image exceeds this image, only the part that fits will be copied.
  */
  bool copy(const SbRasterImage& subimage, const SbVec2i32& destPos = SbVec2i32(0, 0));
  /*
  * Copy a a portion of a sub image inside this image at the given position.
  * If the the portion exceeds this image, only the part that fits will be copied.
  */
  bool copy(const SbRasterImage& subimage, const SbVec2i32& destPos, const SbVec2i32& sourcePos, const SbVec2i32& sourceSize);

protected:
  
  /** 
   * This function returns the size in bytes of the current data type.
   */
  int getDataTypeSizeInBytes() const;

private:

  void commonConstructor( const SbVec2i32 &size, Components components, SbDataType::DataType dataType,
                 SoBufferObject* bufferObject );

  SbVec2i32 m_size;
  Components m_components;
  /**
   * Internal pointer to buffer provided by m_cpuBufferObject except
   * if the setBuffer(unsigned char* buffer, CopyPolicy copy = COPY) method is used
   */
  mutable unsigned char* m_buffer;
  /**
   * Buffer object used to manage buffer
   */
  SoRef<SoBufferObject> m_bufferObject;
  /**
   * CpuBuffer object to map in cpu memory (needed by getBuffer() method
   */
  mutable SoRef<SoCpuBufferObject> m_cpuBufferObject;
  CopyPolicy m_copy;
  int m_numMipmaps;
  uint8_t m_compressedInternalFormat;

  SbDataType::DataType m_dataType;
};

#ifdef _WIN32
#pragma warning(pop)
#endif

inline SbRasterImage::CopyPolicy 
SbRasterImage::getCopyPolicy() const
{
  return m_copy;
}

#endif


