/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2024 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : C. OGNIER (Jun 2003)
**=======================================================================*/


#ifndef _SORASTERIMAGEFILE_
#define _SORASTERIMAGEFILE_

#include <Inventor/image/SoRasterImageIO.h>
#include <Inventor/image/SoRasterImageRW.h>

/**
 *
 * @VSGEXT Class for raster image file input and output.
 *
 * @ingroup Image
 *
 * Class for raster image file input and output.
 *
 * @SEE_ALSO
 *    SoRasterImageIO,
 *    SoRasterImageRW,
 *    SbRasterImage
 *
 */
class INVENTOR_API SoRasterImageFile : public SoRasterImageIO
{
  SO_IMAGE_RASTER_RW_HEADER()

public:
  /**
   * Constructor.
   */
  SoRasterImageFile();
  
  /**
   * Constructor that takes a file name.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE SoRasterImageFile(const char* fileName);

  /**
   * Constructor that takes a file name.
   */
  SoRasterImageFile( const SbString& fileName );

  /**
   * Constructor that takes a file pointer.
   */
  SoRasterImageFile(const FILE* fp);

  /**
   * Destructor.
   */
  virtual ~SoRasterImageFile();

  /**
   * Returns the file name that was given in the filename constructor.
   * If the filename constructor was not used, this method returns an empty string.
   * To get the file name as a full path, the methods
   * getFullPathFileName() or getStringFullPathFileName() should be used.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE virtual const char* getFileName() const;

  /**
   * Returns the file name that was given in the filename constructor.
   * If the filename constructor was not used, this method returns an empty string.
   * To get the file name as a full path, the methods
   * getFullPathFileName() or getStringFullPathFileName() should be used.
   */
  virtual SbString getStringFileName() const;

  /**
   * Returns the file name as a full path.
   * If the filename constructor was not used, this method returns an empty string.
   *
   * This method should be called after openRead() or openWrite(), or else it will return an empty string.
   * To get the file name as it was passed to the constructor,
   * the methods getFileName() or getStringFileName() should be used.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE virtual const char* getFullPathFileName() const;

  /**
   * Returns the file name as a full path.
   * If the filename constructor was not used, this method returns an empty string.
   * This method should be called after openRead() or openWrite(), or else it will return an empty string.
   * To get the file name as it was passed to the constructor,
   * the methods getFileName() or getStringFileName() should be used.
   */
  virtual SbString getStringFullPathFileName() const;

  /**
   * Sets the file name.
   *
   * @UNICODE_WARNING
   */
  SoNONUNICODE virtual void setFileName(const char* fileName);

 /**
   * Sets the file name.
   */
  virtual void setFileName( const SbString& fileName );

  /**
   * Sets the file pointer.
   */
  virtual void setFilePointer(const FILE* fp);

  /**
   * Returns the file pointer.
   */
  virtual FILE* getFilePointer() const;

  /**
   * Opens the file (specified with #setFileName) for reading.
   * It first tries to open the specified filename.
   * If that doesn't work, it tries opening the filename appended
   * to each of the directory paths specified in @I directoryList @i.
   * Returns TRUE if successful.
   * Note: directoryList can be NULL.
   *
   * @B Note:@b SbStringList is a list of pointers, not a list of objects, so
   * the destructor for this class does @I not@i free the memory associated with
   * the SbString objects.  The application is responsible for deleting each object
   * in the list.
   */
  virtual SbBool openRead( const SbStringList * directoryList );

  /**
   * Try to access the file for reading. 
   * (See "openRead") Returns TRUE if the file can be opened.
   * The file remains closed.
   *
   * @B Note:@b SbStringList is a list of pointers, not a list of objects, so
   * the destructor for this class does @I not@i free the memory associated with
   * the SbString objects.  The application is responsible for deleting each object
   * in the list.
   */
  virtual SbBool checkRead( const SbStringList * directoryList );

   /** 
   * Open the file (specified with #setFileName) for writing.
   * The filename should be a full path.  Returns TRUE if successful.
   */
  virtual SbBool openWrite( void );

  /**
   * Close the file.
   */
  virtual void closeFile();

 /**
   * Sets the filePointer to NULL for files closed internally
   * such as RGB files.
   */
  virtual void nullFilePtr();

  /**
   * Sets the file open binary mode.
   */
  virtual void setBinary(SbBool isBinary);

  /**
   * Returns the file open binary mode.
   */
  virtual SbBool isBinary() const;

  /**
   * Returns the file suffix.
   */
  virtual SbString getFileSuffix() const;

#if SoDEPRECATED_BEGIN(9000)

  /**
   * Opens the file with the specified open mode.
   */
  SoDEPRECATED_METHOD(9000,"Use openRead() or openWrite() instead.")
  virtual SbBool openFile(SoRasterImageRW::OpenMode openMode);

#endif/** @DEPRECATED_END */

SoINTERNAL public :

  /**
   * Set the file suffix
   * This suffix can be used by writer to get the proper internal image writer
   */
  virtual void setFileSuffix( const SbString& suffix);

protected:
  void constructorCommon();

  SbString m_fileName;
  SbString m_fullPath;   // full path, after a successful openRead
  SbString m_fileSuffix;
  FILE*  m_fp;
  SbBool m_isBinary;
  SoRasterImageRW::OpenMode m_openMode;
};

#endif
