/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2017 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : David Beilloin (Jul 2014)
**=======================================================================*/

#ifndef _SO_CAD_INPUT_READER_PARAMETERS_H_
#define _SO_CAD_INPUT_READER_PARAMETERS_H_

#include <Inventor/SoInputParameters.h>
#include <Inventor/io/CAD/SoCADFileFormat.h>

#if defined(_WIN32)
#pragma warning( push )
#pragma warning( disable: 4251 ) // 'identifier' : class 'type' needs to have dll-interface to be used by clients of class 'type2'
#endif


/**
 * @VSGEXT CAD Input reader Parameters when importing CAD file type.
 *
 * @ingroup InventorIOCAD
 *
 * @DESCRIPTION
 *
 * This class provides tessellation and import options for the CADReader module.
 * It is used with the SoInput::setInputParameters() method.
 *
 * @SEE_ALSO
 *   SoCADInputReader, SoCADFileFormat
 */
class CADFILEFORMAT_API SoCADInputReaderParameters: public  SoInputParameters
{
public:
  /** Constructor. */
  SoCADInputReaderParameters();

  /** Destructor. */
  virtual ~SoCADInputReaderParameters();

  /** 
   * Parameters used to tessellate the model during import.
   * [OIVJAVA-WRAPPER-CLASS NO_WRAP]
   */
  struct CADFILEFORMAT_API TessellationOption
  {
    TessellationOption();

    /**
    * Set the maximum distance allowed between tessellated mesh triangle and the original surface.
    * This maximum distance is a linear distance (chordal error).
    * A value of 0.0 means it is deactivated.
    * Smaller value means more precise tessellation and thus more memory consumption.
    *
    * Example: @BR
    *         @IMAGE Cad_linearTolerance.png
    *
    * Default: 0.0 (deactivated)
    */
    double linearTolerance;

    /**
    * Set the maximum deviation angle allowed between tessellated mesh triangle and the original surface.
    * This maximum angle is a angular measure in radian.
    * A value of 0.0 means it is deactivated.
    * Smaller value means more precise tessellation and thus more memory consumption.
    *
    * Example: @BR
    *         @IMAGE Cad_angularTolerance.png
    *
    * Default: 0.17444 radian (10)
    */
    double angularTolerance;

    /**
    * Set the maximum edges length ratio of generated triangles.
    * Best quality is achieved with a value of 1.0.
    * A value smaller than 1.0 is clamp to 1.0.
    * Smaller value means more precise tessellation and thus more memory consumption.
    *
    * Example: @BR
    *      @TABLE_1B
    *         @TR maxTriangleEdgesRatio = 1000.0              @TD maxTriangleEdgesRatio = 1.0
    *         @TR @IMAGE Cad_ratio0.png
    *                   @TD @IMAGE Cad_ratio1.png
    *      @TABLE_END
    *
    * Default: 20.0
    */
    double maxTriangleEdgesRatio;

  };

  /** 
  */
  enum Quality
  {
    /**
    * Low quality / less time and memory consumption.
    */
    LOW,

    /**
    * Medium quality / reasonnable quality, time and memory consumption in most case.
    */
    MEDIUM,

    /**
    * High Quality / more time and memory consumption.
    */
    HIGH,
  };

  /**
  * Specify how the scene graph resulting from the import should be generated.
  * [OIVNET-WRAPPER-CLASS PACK{1}]
  * [OIVJAVA-WRAPPER-CLASS NO_WRAP]
  */
  struct CADFILEFORMAT_API ImportOption
  {
    ImportOption();

    /** 
    * Some CAD formats store faceted data in addition to the B-Rep.
    * If forceTesselation is true, then this representation is ignored and a tesselation will be generated from the B-Rep.
    * If forceTesselation is false, the faceted representation is used and other parameters are ignored (including tesselationparameters).
    *
    * Note: using faceted data greatly improves import performance (both time and memory consumption).
    *
    * Default: false
    *
    * @M_SINCE 9.6
    */
    bool            forceTesselation;

    /**
    * Import Edges for the model.
    * Default: false
    */
    bool						importEdges;

    /**
    * Import OpenShell for the model.
    * It is composed of points or polylines.
    * 
    * Default: false
    */
    bool						importOpenShells;

    /**
    * Import PMI (Product and Manufacturing Information) from the model.
    * It is composed of text, measures and view points.
    *
    * Note: while importing text we keep the Font name information from the source
    * if the given font is not available then the Open Inventor default font is used
    * which can be used through OIV_FONT_NAME environment variable or programmatically
    * by calling SoFontNameElement::setDefault(). See SoFont class doc to get some recommended
    * font that support most symbols glyphs.
    * 
    * Default: false
    */
    bool						importPMI;

    /**
    * Merge imported geometry in a single Open Inventor Geometry object.
    * - multiple faces of a shell are merged in a single SoTriangleSet.
    * - multiple edges of a shell are merged in a single SoLineSet.
    *
    * Default: false
    */
    bool						mergeFaces;
  };


  /**
  *  Specify predefined tessellation options. 
  *  Advanced users can use the TessellationOption struct to have more control.
  * [OIVJAVA-WRAPPER JNI_NAME{setTessellationOptionImpl2}]
  */
  void setTessellationOption(Quality  quality = MEDIUM);

  /**
  * Specify tessellation options.
  * [OIVJAVA-WRAPPER CUSTOM_CODE]
  */
  void setTessellationOption(const TessellationOption&  option);
  
  /**
  *  Return current tessellation options.
  * [OIVJAVA-WRAPPER CUSTOM_CODE]
  */
  const TessellationOption& getTessellationOption() const;

  /**
  * Specify import options.
  * [OIVJAVA-WRAPPER CUSTOM_CODE]
  */
  void setImportOption(const ImportOption&  option);

  /**
  * Return current import options.
  * [OIVJAVA-WRAPPER CUSTOM_CODE]
  */
  const ImportOption& getImportOption() const;


private:

  TessellationOption  m_tessellationOption;
  ImportOption m_importOption;
};

#if defined(_WIN32)
#pragma warning( pop )
#endif

#endif /* _SO_CAD_INPUT_READER_PARAMETERS_H_ */

