/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2018 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef _SO_AUDITOR_LIST_
#define _SO_AUDITOR_LIST_

#include <Inventor/misc/SoNotification.h>
#include <Inventor/STL/vector>

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoAuditorList
//
//  SoAuditorList class. This class maintains a list of instances that
//  audit (receive notification from) an SoBase or SoField. Each entry
//  in the list consists of a pointer to the auditor (base or field)
//  instance and a type code of type SoNotRec::Type.
//
//  The type of the auditor object pointer depends on the type code,
//  as follows:
//
//      Type code:      Auditor object pointer:
//
//      CONTAINER       The SoFieldContainer containing the field
//      PARENT          The parent node
//      SENSOR          The SoDataSensor instance
//      FIELD           The destination field instance
//      ENGINE          The destination field instance
//      UNDEFINED       null
//
//////////////////////////////////////////////////////////////////////////////
SoINTERNAL class INVENTOR_API SoAuditorList {
 private:
  struct AuditorListValue {
    void* auditor;
    SoNotRec::Type type;

    AuditorListValue(void* auditor_, SoNotRec::Type type_)
      : auditor(auditor_), type(type_)
    {
    }

    AuditorListValue()
      : auditor(0), type(SoNotRec::UNDEFINED)
    {
    }
  };
  typedef std::vector<AuditorListValue> ListContainer;

 public:
  typedef AuditorListValue value_type;
  // Constructor and destructor.
  SoAuditorList();
  virtual ~SoAuditorList();

  typedef ListContainer::iterator iterator;
  typedef ListContainer::const_iterator const_iterator;

  // Adds an auditor of the given type to the list
  void append(void *auditor, SoNotRec::Type type);

  // Finds an auditor in the list, returning the index or -1 if not found
  iterator find(void *auditor, SoNotRec::Type type);
  const_iterator find(void *auditor, SoNotRec::Type type) const;

  // Removes an auditor from the list
  size_t remove(void *auditor, SoNotRec::Type type);
  // Returns number of auditors in list
  virtual int getLength() const;
  size_t size() const;
  bool empty() const;
  // Call a functor object on all auditor/type pairs in the list. This
  // is safe even if auditors are being added or removed.
  template<typename Func>
    void traverse(const Func& func)
  {
    if (m_list.empty())
      return;
    m_isTraversing++;
    ListContainer::size_type i = m_list.size();
    do
    {
      --i;
      const value_type& v = m_list[i];
      void* auditor = v.auditor;
      if (auditor)
      {
        SoNotRec::Type type = v.type;
        func(auditor, type);
      }
    } while (i != 0);
    m_isTraversing--;
  }
  // Traverse in forward direction i.e., the order that auditors were
  // added. For compatibility.
    template<typename Func>
    void traverse_forward(const Func& func)
  {
    if (m_list.empty())
      return;
    m_isTraversing++;
    for (ListContainer::size_type i = 0, e = m_list.size();
         i != e;
         ++i)
    {
      const value_type& v = m_list[i];
      void* auditor = v.auditor;
      if (auditor)
      {
        SoNotRec::Type type = v.type;
        func(auditor, type);
      }
    }
    m_isTraversing--;
  }
  // Propagates notification to all auditors in list
  void notify(SoNotList *list);

  iterator begin();
  const_iterator begin() const;
  const_iterator cbegin() const;
  iterator end();
  const_iterator end() const;
  const_iterator cend() const;
 private:

  ListContainer m_list;

  size_t numAuds;
  struct AuditorMap;
  AuditorMap* m_fastFindDict;
  int m_isTraversing;

  void compact();
};

#ifdef _WIN32
#pragma warning(pop)
#endif

#endif /* _SO_AUDITOR_LIST_ */


