/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
** Modified by : David Mott (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/

#ifndef SO_BASE_LIST_H
#define SO_BASE_LIST_H

#include <Inventor/SbPList.h>
#include <Inventor/misc/SoBase.h>
#include <Inventor/sensors/SoListSensor.h>

/**
 * Maintains a list of pointers to instances of the SoBase classes.
 *
 * @ingroup General
 *
 * @DESCRIPTION
 *   This subclass of SbPList holds lists of pointers to instances of classes derived
 *   from SoBase (an abstract class). A flag indicates whether adding an instance
 *   pointer to the list should add a reference to the instance. If this flag is
 *   TRUE, then adding and removing pointers from the list updates reference counts
 *   in the corresponding instances.
 *
 * @SEE_ALSO
 *    SoBase,
 *    SoNodeList,
 *    SoPathList
 *
 *
 * [OIV-WRAPPER-CUSTOM-CODE]
 */
class INVENTOR_API SoBaseList : public SbPList
{
 public:
  /**
   * Constructor.
   */
  SoBaseList();

  /**
   * Constructor that pre-allocates storage for @B size @b pointers.
   */
  SoBaseList(int size);

  /**
   * Constructor that copies the contents of another list.
   */
  SoBaseList(const SoBaseList &l);

  /**
   * Destructor.
   */
  virtual ~SoBaseList();

  /**
   * Adds a pointer to the end of the list.
   */
  void append(SoBase* ptr);

  /**
   * Inserts given pointer in list before pointer with given index.
   */
  void insert(SoBase* ptr, int addBefore);

  /**
   * Removes pointer with given index.
   */
  virtual void remove(int which);

  /**
   * Removes all pointers after one with given index, inclusive.
   */
  virtual void truncate(int start);

  /**
   * Copies a list, keeping all reference counts correct.
   */
  void copy(const SoBaseList &l);

  /**
   * Copies a list, keeping all reference counts correct.
   */
  SoBaseList& operator =(const SoBaseList &l)
  { copy(l) ; return *this; }

  /** @copydoc SbPList::operator[]() */
  SoBase* operator [](int i) const
  { return ( static_cast<SoBase*> ( (*static_cast<const SbPList *>(this) ) [i] ) ); }

  /**
   * Sets an element of a list.
   */
  void set(int i, SoBase *ptr);

  /**
   * Indicates whether to call ref() and unref() for bases in the list
   * when adding/removing them. The default value is TRUE.
   */
  void addReferences(SbBool flag);

  /** 
   * Forces each attached SoListSensor to be notified.
   * Attached sensor will be notified using the passed index
   * as the trigger index.
   * If index is greater than the number of elements in the list,
   * index is set to the last element.
   * If the list is empty, index is set to -1.
   */
  void touch( int index = -1 );

  /**
   * Add an auditor to the list.
   */
   void addAuditor(void *auditor, SoNotRec::Type type);

   /**
    * Remove an auditor to the list.
    */
   void removeAuditor(void *auditor, SoNotRec::Type type);

   /**
    * Get current auditor list.
    */
   const SoAuditorList* getAuditors() const
   { 
     if (m_notificationInfo)
       return m_notificationInfo->m_auditors;
     return NULL;
   }

   /**
    * Get last changed index element of the list.
    */
   int getChangedIndex() const 
   { 
     if (m_notificationInfo)
       return m_notificationInfo->m_changedIndex;
     return -1;
   }

   /**
    * Get last changed type of the list.
    */ 
   SoListSensor::ChangeType getChangedType() const
   {
     if (m_notificationInfo)
       return m_notificationInfo->m_changedType;
     return SoListSensor::UNSPECIFIED;
   }
 
   /**
    * setup notification info for auditor mechanism.
    */
  void setNotificationInfo(const int changedIndex, const SoListSensor::ChangeType changedType)
  {
    if (m_notificationInfo)
    {
      m_notificationInfo->m_changedIndex = changedIndex;
      m_notificationInfo->m_changedType  = changedType;
    }
  }

SoINTERNAL public:
  // Really does a truncate. Flag indicates whether to notify.
  void truncate(int start, SbBool doNotify);

protected:
  // Inner class that handle auditors notification infos.
  class SoNotificationInfo
  {
  public:
    // Constructor.
    SoNotificationInfo();
    // Destructor.
    ~SoNotificationInfo();
    // List of auditors: objects to pass notification to
    SoAuditorList *m_auditors;
    // changed element index
    int m_changedIndex;
    // type of change
    SoListSensor::ChangeType m_changedType;
  };

  // start notification.
  void startNotify();

  bool getAddReferencesStatus() { return addRefs; }

 private:
  // If TRUE (the default), this refs and unrefs things in the list
  bool addRefs;

  // Auditor notification infos
  SoNotificationInfo *m_notificationInfo;
};

#endif
