/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul S. Strauss (MMM yyyy)
** Modified by : Nick Thompson (MMM yyyy)
** Modified by : Gavin Bell (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_BASE_
#define  _SO_BASE_

#include <Inventor/misc/SoBasic.h>
#include <Inventor/misc/SoAuditorList.h>
#include <Inventor/misc/SoRefCounter.h>
#include <Inventor/SbString.h>
#include <Inventor/SoTypedObject.h>
#include <Inventor/threads/SbThreadSpinlock.h>

class SoBaseList;
class SoInput;
class SoNode;
class SoNotList;
class SoOutput;
class SoPath;

#ifdef _WIN32
#pragma warning(push)
#pragma warning(disable:4251)
#endif

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoBase
//
//  Base class for most important SO classes. All subclasses of this
//  class may be read and written, in the form:
//      NameOfClass {
//          <stuff inside>
//      }
//  where NameOfClass is the thing returned by getFileName(). For example,
//  the "FileName" of the SoPath class is "Path".
//
//////////////////////////////////////////////////////////////////////////////


/**
* Base class for all nodes, paths, and engines.
*
* @ingroup General
*
* @DESCRIPTION
*   Abstract base class for Open Inventor node, path, and engine classes. This class
*   handles reference counting, notification, and naming.
*
*   Instances should be referenced when they will be used outside of the routine in
*   which they were initialized. (A typical example of this is maintaining a
*   reference to the root of a graph.)
*
* @SEE_ALSO
*    SoFieldContainer,
*    SoNode,
*    SoPath,
*    SoEngine,
*    SoDB
*
* [OIV-WRAPPER-CLASS DERIVABLE,CUSTOM_DISPOSE,EXTRA_STATIC_CONSTRUCTOR]
*/
class INVENTOR_API SoBase: public SoRefCounter, public SoTypedObject {
 public:

  /**
   * Marks an instance as modified, simulating a change to it. This will notify
   * auditors (parent nodes, connected engines, and so on) of a change to this object
   * and cause attached sensors to be triggered.
   */
  virtual void                touch()         { startNotify(); }

  /**
   * Returns type identifier for this class.
   */
  static SoType               getClassTypeId();

  /**
   * Returns the name of an instance. If the instance has not been named, an empty
   * string is returned. Objects that are named can be looked up using the
   * getByName() methods of SoNode, SoEngine, or SoPath.
   */
  virtual SbName              getName() const;

 /**
  * Sets the name of an instance. Object names are preserved when objects are
  * written to or read from files. Object names must not begin with a digit, and
  * must not contain spaces or control characters, periods, single or double quote
  * characters, backslashes, curly braces, square brackets or the plus character. 
  * \if_cpp
  * The SbName::isBaseNameChar() and SbName::isBaseNameStartChar() methods can
  * be used to validate names input by users. 
  * \endif
  * This method will replace any bad characters in the name with underscore 
  * characters, and will print out an error message if the application is using 
  * the Open Inventor debugging library.
  *
  * The same name may be assigned to more than one instance.
  *
  * Avoid using names that begin with "OIV_". Such names may be used internally.
  *
  * Nodes may be queried by name using the SoNode::getByName methods.
  * Node names may also be used with SoSearchAction.
  * [OIVNET-WRAPPER-CUSTOM-CODE]
  */
  virtual void                setName(const SbName &name);

  /**
   * Sets this to be a ScaleViz synchronizable object.
   * By default all objects are synchronizable.
   */
  inline void setSynchronizable( const bool b );

  /**
   * Gets the ScaleViz synchronizable state of this object.
   */
  inline bool isSynchronizable() const ;

 protected:

  // Constructor is protected - this is an abstract class
  SoBase();

  // Virtual destructor so that subclasses are deleted properly
  virtual ~SoBase();

  /**
  * @copydoc SoRefCounter::notifyDelete()
  * [OIV-WRAPPER-NO-WRAP]
  */
  virtual bool notifyDelete() const;

  // Actually deletes an instance. Allows subclasses to do other
  // stuff before the deletion if necessary.
  virtual void                destroy();

  // Returns current write counter
  static uint32_t     getCurrentWriteCounter();

  // Returns TRUE if the instance has multiple write references
  SbBool                      hasMultipleWriteRefs() const
    { return writeStuff.multWriteRef; }

  // Unknown nodes and engines write a different name for themselves
  // than their typeId; this virtual method lets them do that (by
  // default the typeId name is returned)
  virtual const char* getFileFormatName() const;

  // This set of enums is used when reading and writing the base.
  // Each enum represents a different bit in a bit field
  // which will be written.
  enum BaseFlags {
    IS_ENGINE        = 1,
    IS_GROUP         = 2
  };

  // Reads stuff into instance of subclass. Return FALSE on error.
  // If reading binary file format, the flags specify whether the
  // object was written as an engine or a group; unknown nodes and
  // groups need this information to read themselves in properly.
  virtual SbBool      readInstance(SoInput *in, unsigned short flags) = 0;

 SoINTERNAL public:

  // Setup type information
  static void initClass();
  static void exitClass();

  // Enable/disable ART synchronization. If false, all ART serialization is disabled.
  // Can be usefull to disable serialization in some part of code.
  // Default is true. Return previous value.
  static bool setSynchronizationEnabled(bool value)
  {
    bool old = s_synchronizationEnabled;
    s_synchronizationEnabled = value;
    return old;
  }

  static void resetWriteCounter();

  // Increments the current write counter at the start of a write operation
  static void incrementCurrentWriteCounter();

  // Decrements the current write counter after a write operation,
  // in some rare cases
  static void decrementCurrentWriteCounter();

  // Initiates notification from an instance. The default method
  // does nothing, because some classes (path, sensor) never
  // initiate notification. This is used by touch().
  virtual void startNotify();

  // Propagates modification notification through an instance. The
  // default method here does not create and add a new record. It
  // merely propagates the current record list to all auditors. This
  // method may be used by subclasses to do the propagation after
  // modifying the list appropriately.
  virtual void notify(SoNotList *list);

  // Adds/removes an auditor to/from list
  void addAuditor(void *auditor, SoNotRec::Type type);
  void removeAuditor(void *auditor, SoNotRec::Type type);

  // Returns auditor list-- used by SoField and SoEngineOutput to
  // trace forward connections
  const SoAuditorList &getAuditors() const { return auditors; }

  // Internal methods used to maintain the global name dictionary
  static void addName(SoBase *, const char *);
  static void removeName(SoBase *, const char *);

  // Helper routines used to get stuff out of nameDict
  static SoBase *getNamedBase(const SbName &, const SoType&);
  static int getNamedBases(const SbName &, SoBaseList &, const SoType&);

  // Set the net name of an instance.
  virtual void setNetName( const SbName &netName);

  // Get the net name of an instance.
  virtual SbName getNetName() const;

  // Compute automatically the net name of the instance.
  virtual void generateNetName();

  // Sets this a ScaleViz synchronized object.
  inline void setSynchronized( const bool b );

  // Gets the ScaleViz synchronized state of this object.
  inline bool isSynchronized() const;

  /**
   * Writes a header (name, open brace) or footer (close brace) to
   * file defined by SoOutput. writeHeader returns TRUE if no
   * further writing is necessary for an instance.
   * isEngine/isGroup are exactly what they sound like, and must be
   * passed in so that unknown nodes/engines can be correctly
   * created when reading the binary file format.
   */
  SbBool writeHeader(SoOutput* , SbBool , SbBool ) const;
  void writeFooter(SoOutput* ) const;

  virtual bool isGroup() { return false; }

  // Reads one instance of some subclass of SoBase. Returns pointer
  // to read-in instance in base, or NULL on EOF. Returns FALSE on
  // error. The last parameter is a subclass type to match. If
  // the returned base is not of this type, it is an error. A type
  // of SoBase::getClassTypeId() will match any base.
  static SbBool read(SoInput *in, SoBase *&base, const SoType &expectedType, SbBool createAndRead = TRUE);

  // Adds a reference to the instance when writing. isFromField
  // indicates whether the reference is from a field-to-field
  // connection.
  virtual void addWriteReference(SoOutput *out, SbBool isFromField = FALSE);

  // Returns TRUE if the instance should be written, based on the
  // write-reference info already accumulated
  SbBool shouldWrite();

  // This defaults to "+" and is used when naming nodes that are DEF's and USE'd.
  // The ivdowngrade converter needs to set it to other than "+" since that was
  // an illegal character for Inventor V1.0 names.
  static void setInstancePrefix(const SbString &c);

    // These are all packed into one word to save space
  struct WriteStuff {
    // This contains the value of the counter the last time the
    // instance was written.
    unsigned int writeCounter : 26;

    // This is TRUE if the object is named.
    unsigned int hasName : 1;
    unsigned int hasNetName : 1;

    // This is TRUE if more than one reference is made to the
    // instance for writing, meaning that we need to DEF it
    unsigned int multWriteRef : 1;

    // This is TRUE if the instance was referenced via a
    // field-to-field connection, which is not a "real" reference
    unsigned int writeRefFromField : 1;

    // This is TRUE if the instance was already synchronized
    // through ScaleViz
    unsigned int isSynchronized : 1;

    // This is TRUE if the instance can be synchronized
    // through ScaleViz
    unsigned int isSynchronizable : 1;
    
    // Default Constructor
    WriteStuff();
  };

  WriteStuff  writeStuff;

 private:

  static SoType classTypeId;

  static SbString instancePrefix;

  // This is incremented once per write operation. It is used to
  // determine which instances to write.
  static uint32_t s_currentWriteCounter;
  static SbThreadRWMutex s_writeCounterMutex;

  // List of auditors: objects to pass notification to
  SoAuditorList auditors;

  // If false, disable all ART serialization. True by default.
  static bool s_synchronizationEnabled;

  // These are used internally by writeHeader()
  void writeDef(SoOutput *, int) const;
  void writeRef(SoOutput *, int) const;
  void writeAnnotation(SoOutput *) const;
  void writeAnnotation1(SoOutput *) const;

  static SbBool readReference(SoInput *in, SoBase *&base);
  static SbBool readBase(SoInput *in, SbName &className, SoBase *&base, SbBool createAndRead = TRUE);
  static SbBool readBaseInstance( SoInput *in,
                                  const SbName &className, const SbName &refName,
                                  const SbName &netName, SoBase *&base, SbBool createAndRead = TRUE);
  static SoBase* createInstance(SoInput *in, SbName className, unsigned short ioFlags);

  static void flushInput(SoInput *in);

  // This dictionary stores SbPLists keyed by name (the SbPLists
  // contain SoBases-- a BaseList isn't used because we don't want
  // the items on the list to be reference counted, otherwise they
  // will never get deleted).
  static SbDict *nameObjDict;

  // And this dictionary maps the other way, from an SoBase * to a
  // name.
  static SbDict *objNameDict;

  friend class SoVRMLDB;
};

#ifdef _WIN32
#pragma warning(pop)
#endif

/**
 * Internal class used for ScaleViz objects serialization.
 */
SoINTERNAL class INVENTOR_API SoBaseInitializer
{
public:
  SoBaseInitializer(SoBase*);
  ~SoBaseInitializer();
  static inline bool isActive()
  {
    s_activeMutex.lock();
    bool ret =  m_active;
    s_activeMutex.unlock();
    return ret;
  }
private:
  SoBase* m_base;
  bool m_sync;
  bool m_prevActive;
  static bool m_active;
  static SbThreadSpinlock s_activeMutex;
};


void
SoBase::setSynchronizable( const bool b )
{
  writeStuff.isSynchronizable = b;
}

bool
SoBase::isSynchronizable() const
{
  return writeStuff.isSynchronizable && s_synchronizationEnabled;
}

void
SoBase::setSynchronized( const bool b )
{
  writeStuff.isSynchronized = b;
}

bool
SoBase::isSynchronized() const
{
  return writeStuff.isSynchronized;
}

#endif /* _SO_BASE_ */

