/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2023 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#ifndef SO_REF_H
#define SO_REF_H

#include <Inventor/misc/SoAutoRef.h>

/**
* Smart pointer for any class inheriting SoRefCounter.
* 
* @ingroup General
* 
* @DESCRIPTION
*
* This class implements a smart pointer for any reference counted class (class
* that inherits SoRefCounter).  This includes nodes, engines, paths, buffer
* objects and others.  We strongly recommend using SoRef to manage the life span
* of these objects (instead of the old style ref() and unref() methods).
*
* The reference count of the managed object is automatically incremented when
* the SoRef is created and then decremented when the SoRef is destroyed,
* for example by going out of scope, or when 'null' is assigned to the SoRef object.
* If this decrement changes the managed object's reference count to zero, the
* managed object is destroyed.
*
* For example SoRef allows a reference counted object to be safely cleaned up
* when it goes out of scope in a local block, as shown in the example below.
* \code
*    {
*      // Create temporary buffer object
*      SoRef<SoCpuBufferObject> cpuObj = new SoCpuBufferObject;
*      cpuObj->map(SoBufferObject::READ_ONLY);
*      // Use the buffer object
*      . . .
*      cpuObj->unmap();
*      // Buffer object will be automatically cleaned up
*    }
* \endcode
* You can also explicitly cleanup a reference counted object like this:
* \code
*   SoRef<SoPath> tempPath = ...    // Create temporary object
*     action->apply( tempPath );
*   tempPath = nullptr;             // Cleanup temporary object
* \endcode
* Generally, in a function, you should not use SoRef on a reference counted object
* that will be returned by the function. In this case the calling code would receive
* a pointer to an invalid object after the SoRef goes out of scope and unref's the object.
*
* Note that implicit casting to a pointer is not supported.
* So when a method expects a pointer, for example addChild(), 
* you must use the explicit ptr() method. For example:
* \code
*    SoRef<SoCone> cone = new SoCone();
*    // Use
*    root->addChild( cone.ptr() );
*
*    // NOT
*    root->addChild( cone );
* \endcode
*
* @SEE_ALSO
*    SoRefCounter
* 
* [OIV-WRAPPER-NO-WRAP]
*/

template <typename T>
class SoRef : public SoAutoRef
{
public:
  /// Default constructor.
  SoRef()
  {};

  /// Copy constructor.
  SoRef(const SoRef<T>& that)
  :SoAutoRef(that)
  {
  }

  /// Construct from C++ pointer.
  SoRef(T* pIn)
  :SoAutoRef(dynamic_cast<T*>(pIn))
  {
  }

  /// Construct from another SoRef
  template<typename Y>
  SoRef(const SoRef<Y>& that)
  {
    m_pointer = that.ptr();
    if (m_pointer)
      m_pointer->ref();
  }

  /// Release reference and return the object stored.
  inline T* releaseNoDelete()
  {
    T* p = dynamic_cast<T*>(m_pointer);
    if (m_pointer)
    {
      m_pointer->unrefNoDelete();
      m_pointer = NULL;
    }
    return p;
  }


  /// Assign another @c SoRef.
  SoRef<T>& operator=(const SoRef<T>& that)
  {
    (*this) = (T*)(that.m_pointer);
    return *this;
  }

  /// Assign to another @c SoRef if the types are assignable (from derived to base classes).
  template<typename Y>
  SoRef<T>& operator=(const SoRef<Y>& that)
  {
    (*this) = dynamic_cast<T*>(that.ptr());
    return *this;
  }

  /// Assign ordinary C pointer.
  SoRef<T>& operator=(T* ptr)
  {
    if (m_pointer != ptr) {
      if (m_pointer)
        m_pointer->unref();
      m_pointer = ptr;
      if (m_pointer)
        m_pointer->ref();
    }
    return *this;
  }

  /// Dereference pointer.
  T& operator*() const
  { return *((T*)m_pointer); }

  /// Cast to C pointer.
  T* operator->() const
  { return ((T*)(m_pointer)); }

  /// Cast to C pointer.
  T* ptr() const 
  { return ((T*)(m_pointer)); }

};

#endif // SO_REF_H


