/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2021 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Author      : VSG (MMM YYYY)
**=======================================================================*/

#ifndef SO_REF_COUNTER_H
#define SO_REF_COUNTER_H

#include <stdio.h>
#include <Inventor/SbBase.h>
#include <Inventor/threads/SbThreadSpinlock.h>
#include <Inventor/errors/SoDebugError.h>
#include <functional>

class SoRefCounter;

// callback function prototypes
typedef void SoBaseRefCountCB(void *userData, const SoRefCounter *base);


/**
* Base class for ref-counted objects.
* 
* @ingroup General
* 
* @DESCRIPTION
*
* If a single object or more precisely a pointer to a single object
* is used at multiple locations in a program there should be some
* way to detect if the object is still in use or if it can be destroyed.
*
* Reference counting as implemented by this class provides such a way.
* The behavior is the same as reference counting of Open Inventor nodes
* except that there is no automatic ref/unref (unless you use SoRef).
* Manual reference counting means that the user has to call the methods
* @c ref() and @c unref() whenever he wants to use or discard an object.
* These methods increment or decrement the reference count.  The initial
* reference count is zero.  The object is destroyed automatically if the
* is decremented to zero, i.e., the object isn't in use anymore.
*
* The related class SoRef implements smart pointers to
* reference counted objects. Such pointers automatically keep track of
* reference counting and therefore are safer than explicit use of @c ref()
* and @c unref().
*
* @SEE_ALSO
*    SoRef
* 
* 
* [OIV-WRAPPER-NO-WRAP]
*/
class INVENTORBASE_API SoRefCounter
{
public:
    /**
     * Adds a reference to an instance.
     * This is defined to be const so users can call it on any instance, even const ones.
     */
    void ref() const;

    /** 
     * Removes a reference from an instance.
     * This method removes a reference from the object. If the reference
     * count goes to zero the object is automatically deleted.
     * This is defined to be const so users can call it on any instance, even const ones.
     */
    void unref() const;

    /**
     * unrefNoDelete() should be called when it is desired to decrement the
     * reference count, but not delete the instance if this brings the reference count
     * to zero. This is most useful in returning an object to a zero-reference-count
     * state, like it was when it was created by @B new@b.
     * This is defined to be const so users can call it on any instance, even const ones.
     */
    void unrefNoDelete() const;

    /**
     * Returns current reference count.
     */
    inline int getRefCount() const
    { return m_refcount; }

    /**
     * lock this instance.
     */
    inline void lock() const
    { m_refMutex.lock(); }

    /**
     * unlock this instance.
     */
    inline void unlock() const
    { m_refMutex.unlock(); }


SoINTERNAL public:
    /** init static members of this class */
    static void initClass();

    /** reset static members of this class */
    static void exitClass();

    struct OnDeleteCB{
      SoRefCounter* context;
      std::function<void(SoRefCounter* caller, SoRefCounter* context)> callback;
      bool isValid;
      OnDeleteCB(SoRefCounter* ptr, const std::function<void(SoRefCounter* caller, SoRefCounter* context)> &cb): context(ptr), callback(cb), isValid(true){}
    };

    /**
     * Set a global callback called by invokeRefCountCB when a 
     * ref() / unref() has occurred on any SoRefCounter
     * object.
     * Note that this callback would be called only if a related
     * user data has been associated to an instance of SoRefCounter
     * using the setRefCountCallbackData instance method.
     */
    static void setRefCountCallback(SoBaseRefCountCB *f)
    { s_refCountCBFunc = f; }

    /**
     * Set the user data pointer associated to the RefCountCB.
     * If no user data is provided, the callback would not be called.
     * Since the callback is global for all SoRefCounter objects, this 
     * behavior prevents the call of the callback for all SoRefCounter
     * object. 
     * Considering that the callback for now is only used in wrapped APIs
     * and that a filter is done on the user data (only considering ref/unref 
     * events when the user data is not null), it's preferable to perform
     * the filter on c++ side to avoid overhead due to transitions between
     * managed and unmanaged worlds (see bug #40379).
     */
    void setRefCountCallbackData(void *userData = NULL)
    { m_refCountCBData = userData; }

    /**
     * Invoke the refCount callback if any each time the refcount is modified.
     */
    void invokeRefCountCB() const;

    /** Turns on/off reference count tracing (for debugging) */
    static inline void setTraceRefs(const bool enable)
    { s_traceRefs = enable; }

    /** get reference count */
    static inline SbBool getTraceRefs()
    { return s_traceRefs; }

    /**
     * Force the refcounter a specific value.
     * this should be never used (specific to ScaleViz).
     */
    void setRefCount(const int count);

    // Add callback function called juste before object deletion. If a context is used, this callback is automaticaly removed when the context object is deleted.
    void addOnDeleteCB(SoRefCounter *context, const std::function<void()> &callBackFunction);
    void addOnDeleteCB(SoRefCounter *context, const std::function<void(SoRefCounter* caller)> &callBackFunction);
    void addOnDeleteCB(SoRefCounter *context, const std::function<void(SoRefCounter* caller, SoRefCounter* context)> &callBackFunction);
    void addOnDeleteCB(const std::function<void()> &callBackFunction);
    void addOnDeleteCB(const std::function<void(SoRefCounter* caller)> &callBackFunction);


  protected:
    /** default constructor. */
    SoRefCounter();

    /** Copy constructor; explicitly DOES NOT copy the refcount. */
    SoRefCounter(const SoRefCounter &);

    /** Assignment operator; explicitly DOES NOT copy the refcount. */
    SoRefCounter & operator=(const SoRefCounter &)
    { return *this; }

    /** Destructor is protected. Use unref() instead. */
    virtual ~SoRefCounter();


    /** Will be called just before the object is deleted. 
     * You might want to override the method to detach objects or
     * do some other cleanup which needs to be done.
     * Some cleanup can't be deferred to the destructor because all
     * the derived stuff is already gone. default is true.
     */
    virtual bool notifyDelete() const
    { return true; }

    /**
     * Actually deletes an instance. Allows subclasses to do other
     * stuff before the deletion if necessary.
     */
    virtual void destroy()
    { delete this; }

private:
  /** Turns on/off reference count tracing (for debugging) */
  static bool s_traceRefs;

  /** RefCount is a single callback function, not a list */
  static SoBaseRefCountCB* s_refCountCBFunc;
  void* m_refCountCBData;

  /** Reference count (consider ref/unref as a const operation) */
  mutable int m_refcount;
  mutable SbThreadSpinlock m_refMutex;

  // OnDeleteCB handling
  std::vector<OnDeleteCB> *m_onDeleteCBVec;
  std::vector<SoRefCounter*> *m_contextVec;
  void cleanUpContextsCB();
  void cleanUpContextsLists();

};

/** helper macro usefull in destructor implementation */
#define SO_UNREF_RESET(object)\
{\
  if ( object )\
  {\
    object->unref();\
    object=NULL;\
  }\
}

// Included here to provide this service by default everywhere SoRefCounter 
// is already used, avoiding app to add a #include <Inventor/misc/SoAutoRef.h>
#include <Inventor/misc/SoAutoRef.h>

#endif // SO_REF_COUNTER_H


