#ifndef SO_ALGEBRAIC_SPHERE_H
#define SO_ALGEBRAIC_SPHERE_H

#include <Inventor/nodes/SoAlgebraicShape.h>
#include <Inventor/fields/SoSFFloat.h>


/**
 * @VSGEXT Algebraic sphere shape node.
 *
 * @ingroup ShapeNodes
 * 
 * @DESCRIPTION
 *   This node represents a sphere. By default, the sphere is centered at the origin
 *   and has a radius of 1.
 *
 *   The sphere is transformed by the current cumulative transformation and is drawn
 *   with the current lighting model and material.
 *
 *   This node can generally be used in place of the SoSphere geometric shape node.
 *   Unlike the geometric shape nodes, which create triangle geometry to represent their
 *   shape, the algebraic shape nodes compute and render their shape on the GPU.
 *   Algebraic shapes can also be used with SoMultipleInstance to efficiently render
 *   a very large number of shapes.
 *
 *   Please see SoAlgebraicShape for important notes and limitations for algebraic shapes.
 * 
 * @FILE_FORMAT_DEFAULT
 *    AlgebraicSphere {
 *    @TABLE_FILE_FORMAT
 *       @TR radius   @TD 1
 *    @TABLE_END
 *    }
 * 
 * @SEE_ALSO
 *    SoAlgebraicShape,
 *    SoMultipleInstance,
 *    SoSphere
 */
class INVENTOR_API SoAlgebraicSphere : public SoAlgebraicShape
{
  SO_NODE_HEADER(SoAlgebraicSphere);

public:

  /**
   * Radius of sphere. This must be greater than 0. Default is 1.
   */
  SoSFFloat radius;

  /**
   * Default constructor.
   */
  SoAlgebraicSphere ();

protected:

  /**
   * Destructor.
   */
  virtual ~SoAlgebraicSphere () {}

  /**
   * Compute the bounding box of the shape
   * @param box the bounding box
   * @param center the center of the bbox
   */
  virtual void computeBBox ( SbBox3f &box, SbVec3f &center );

SoEXTENDER public:

  /**
   * Specialized ray pick action.
   */
  virtual void rayPick ( SoRayPickAction *action );

  /**
   * Alternate representation (i.e. SoSphere) for unhandled actions
   */
  virtual SoNode* getAlternateRep ( SoAction* action );

SoINTERNAL public:

  //@{
  /**
   * Registration.
   */
  static void initClass ();
  static void exitClass ();
  //@}

};

#endif // SO_ALGEBRAIC_SPHERE_H
