/*=======================================================================
 * Copyright 1991-1996, Silicon Graphics, Inc.
 * ALL RIGHTS RESERVED
 *
 * UNPUBLISHED -- Rights reserved under the copyright laws of the United
 * States.   Use of a copyright notice is precautionary only and does not
 * imply publication or disclosure.
 *
 * U.S. GOVERNMENT RESTRICTED RIGHTS LEGEND:
 * Use, duplication or disclosure by the Government is subject to restrictions
 * as set forth in FAR 52.227.19(c)(2) or subparagraph (c)(1)(ii) of the Rights
 * in Technical Data and Computer Software clause at DFARS 252.227-7013 and/or
 * in similar or successor clauses in the FAR, or the DOD or NASA FAR
 * Supplement.  Contractor/manufacturer is Silicon Graphics, Inc.,
 * 2011 N. Shoreline Blvd. Mountain View, CA 94039-7311.
 *
 * THE CONTENT OF THIS WORK CONTAINS CONFIDENTIAL AND PROPRIETARY
 * INFORMATION OF SILICON GRAPHICS, INC. ANY DUPLICATION, MODIFICATION,
 * DISTRIBUTION, OR DISCLOSURE IN ANY FORM, IN WHOLE, OR IN PART, IS STRICTLY
 * PROHIBITED WITHOUT THE PRIOR EXPRESS WRITTEN PERMISSION OF SILICON
 * GRAPHICS, INC.
**=======================================================================*/
/*=======================================================================
** Author      : Paul Isaacs (MMM yyyy)
**=======================================================================*/
/*=======================================================================
 *** THE CONTENT OF THIS WORK IS PROPRIETARY TO FEI S.A.S, (FEI S.A.S.),            ***
 ***              AND IS DISTRIBUTED UNDER A LICENSE AGREEMENT.                     ***
 ***                                                                                ***
 ***  REPRODUCTION, DISCLOSURE,  OR USE,  IN WHOLE OR IN PART,  OTHER THAN AS       ***
 ***  SPECIFIED  IN THE LICENSE ARE  NOT TO BE  UNDERTAKEN  EXCEPT WITH PRIOR       ***
 ***  WRITTEN AUTHORIZATION OF FEI S.A.S.                                           ***
 ***                                                                                ***
 ***                        RESTRICTED RIGHTS LEGEND                                ***
 ***  USE, DUPLICATION, OR DISCLOSURE BY THE GOVERNMENT OF THE CONTENT OF THIS      ***
 ***  WORK OR RELATED DOCUMENTATION IS SUBJECT TO RESTRICTIONS AS SET FORTH IN      ***
 ***  SUBPARAGRAPH (C)(1) OF THE COMMERCIAL COMPUTER SOFTWARE RESTRICTED RIGHT      ***
 ***  CLAUSE  AT FAR 52.227-19  OR SUBPARAGRAPH  (C)(1)(II)  OF  THE RIGHTS IN      ***
 ***  TECHNICAL DATA AND COMPUTER SOFTWARE CLAUSE AT DFARS 52.227-7013.             ***
 ***                                                                                ***
 ***                   COPYRIGHT (C) 1996-2014 BY FEI S.A.S,                        ***
 ***                        BORDEAUX, FRANCE                                        ***
 ***                      ALL RIGHTS RESERVED                                       ***
**=======================================================================*/
/*=======================================================================
** Modified by : VSG (MMM YYYY)
**=======================================================================*/


#ifndef  _SO_ANTI_SQUISH_
#define  _SO_ANTI_SQUISH_

#include <Inventor/fields/SoFieldData.h>
#include <Inventor/nodes/SoNode.h>
#include <Inventor/nodes/SoTransformation.h>
#include <Inventor/fields/SoSFBool.h>
#include <Inventor/fields/SoSFEnum.h>

//////////////////////////////////////////////////////////////////////////////
//
//  Class: SoAntiSquish
//
//  undoes any uneven scales in the current transformation matrix.
//  during traversal
//
//////////////////////////////////////////////////////////////////////////////

/**
 * Transformation node that undoes non-uniform 3D scales.
 * 
 * @ingroup TransformNodes
 * 
 * @DESCRIPTION
 *   This node removes nonuniform 3D scaling from the current transformation matrix
 *   when traversed by an action. It is used by draggers such as the
 *   SoTrackballDragger that need to stay uniformly scaled no matter where they are
 *   located in the scene graph.
 *   
 *   The magnitude of the new scale is determined by the current transformation matrix
 *   and the sizing field. This node does not change the translation or rotation in
 *   the matrix.
 * 
 * @FILE_FORMAT_DEFAULT
 *    AntiSquish {
 *    @TABLE_FILE_FORMAT
 *       @TR sizing         @TD AVERAGE_DIMENSION
 *       @TR recalcAlways   @TD TRUE
 *    @TABLE_END
 *    }
 * 
 * @ACTION_BEHAVIOR
 *    SoGLRenderAction,  SoCallbackAction,  SoGetBoundingBoxAction,  SoGetMatrixAction,  SoRayPickAction @BR
 *        Appends the current transformation with a new matrix to create an unsquished
 *        result.
 * 
 * 
 * @SEE_ALSO
 *    SoCenterballDragger,
 *    SoJackDragger,
 *    SoTrackballDragger,
 *    SoTransformerDragger,
 *    SoTransformation,
 *    SoTransformBoxDragger
 * 
 * 
 */
class INVENTOR_API SoAntiSquish : public SoTransformation {

  SO_NODE_HEADER(SoAntiSquish);

 public:

  /**
   * Creates an anti-squish node with default settings.
   */
  SoAntiSquish();

  /** Sizing */
  enum Sizing {
    /**
     *  Fits the other two axes to match the X axis, whose size is unchanged 
     */
    X,
    /**
     *  Fits the other two axes to match the Y axis, whose size is unchanged 
     */
    Y,
    /**
     *  Fits the other two axes to match the Z axis, whose size is unchanged 
     */
    Z,
    /**
     *  Uses average of 3 scales in the matrix 
     */
    AVERAGE_DIMENSION,
    /**
     *  Uses biggest of 3 scales in the matrix 
     */
    BIGGEST_DIMENSION,
    /**
     *  Uses smallest of 3 scales in the matrix 
     */
    SMALLEST_DIMENSION,
    /**
     *  Accounts for shearing; transforms a cube by the matrix and then uses length of
     * longest diagonal 
     */
    LONGEST_DIAGONAL
  };

  // Fields
  /**
   * Determines which of the algorithms enumerated by the type Sizing will be used to
   * select the new scale when the x,y, and z scales are not equal.
   * @useenum{Sizing}.  Default is AVERAGE_DIMENSION.
   */
  SoSFEnum                sizing;
  /**
   * If recalcAlways is TRUE, this node calculates its unsquishing matrix every time
   * it is traversed. If FALSE, then this only occurs during the first traversal
   * folllowing a call to recalc().  Default is TRUE.
   * 
   */
  SoSFBool                recalcAlways;

  /**
   * Sets a flag so that the next time the node is traversed, it will recalculate its
   * unsquishing matrix.
   */
  void recalc();

 SoEXTENDER public:
  virtual void        callback(SoCallbackAction *action);
  virtual void        GLRender(SoGLRenderAction *action);
  virtual void        getBoundingBox(SoGetBoundingBoxAction *action);
  virtual void        getMatrix(SoGetMatrixAction *action);
  virtual void        pick(SoPickAction *action);
  
  // Called by actions that need to change the state with a new matrix.
  void                doAction(SoAction *action);

 SoINTERNAL public:
  static void         initClass();
  static void         exitClass();

 protected:
  virtual ~SoAntiSquish();

  SbMatrix getUnsquishingMatrix( SbMatrix squishedMatrix, SbBool doInverse,
                                 SbMatrix &inverseAnswer );

 private:
  SbMatrix savedAnswer, savedInverseAnswer;

  SbBool recalcNextTime;
};

#endif /* _SO_ANTI_SQUISH_ */

